import numpy as np

from skfem.element.element_h1 import ElementH1
from skfem.refdom import RefTet


class ElementTetP3(ElementH1):
    """Piecewise cubic element."""

    nodal_dofs = 1
    facet_dofs = 1
    edge_dofs = 2
    maxdeg = 3
    dofnames = ["u", "u", "u", "u"]
    doflocs = np.array([[0., 0., 0.], #nodal O  ==0
                        [1., 0., 0.], #nodal x  ==1
                        [0., 1., 0.], #nodal y  ==2
                        [0., 0., 1.], #nodal z  ==3
                        [1./ 3, 0., 0.], #edge O->x  ==4
                        [2./ 3, 0., 0.], #edge O->x  ==5
                        [2./ 3, 1./ 3, 0.], #edge x->y  ==6
                        [1./ 3, 2./ 3, 0.], #edge x->y  ==7
                        [0., 1./ 3, 0.], #edge O->y  ==8
                        [0., 2./ 3, 0.], #edge O->y  ==9
                        [0., 0., 1./ 3], #edge O->z  ==10
                        [0., 0., 2./ 3], #edge O->z  ==11
                        [2./ 3, 0., 1./ 3], #edge x->z  ==12
                        [1./ 3, 0., 2./ 3], #edge x->z  ==13
                        [0., 2./ 3, 1./ 3], #edge y->z  ==14
                        [0., 1./ 3, 2./ 3], #edge y->z  ==15
                        [1./ 3, 1./ 3, 0.], #facet Oxy  ==16
                        [1./ 3, 0., 1./ 3], #facet Oxz  ==17
                        [0., 1./ 3, 1./ 3], #facet Oyz  ==18
                        [1./ 3, 1./ 3, 1./ 3]]) #facet xyz  ==19
    refdom = RefTet

    def lbasis(self, X, i):
        x, y, z = X

        if i == 0:  # at (0,0,0)
            phi = -(3.*x + 3.*y + 3.*z - 1.)*(3.*x + 3.*y + 3.*z - 2.)*(x/2 + y/2 + z/2 - 1./2)
            dphi = np.array([
                18.*x + 18.*y + 18.*z - 27.*x*y - 27.*x*z - 27.*y*z 
                - (27.*x**2)/2 - (27.*y**2)/2 - (27.*z**2)/2 - 11./2,
                18.*x + 18.*y + 18.*z - 27.*x*y - 27.*x*z - 27.*y*z 
                - (27.*x**2)/2 - (27.*y**2)/2 - (27.*z**2)/2 - 11./2,
                18.*x + 18.*y + 18.*z - 27.*x*y - 27.*x*z - 27.*y*z 
                - (27.*x**2)/2 - (27.*y**2)/2 - (27.*z**2)/2 - 11./2
            ])
        elif i == 1:  # at (1,0,0)
            phi = (x*(3.*x - 1.)*(3.*x - 2))/2
            dphi = np.array([
                (27.*x**2)/2 - 9.*x + 1.,
                0*y,
                0*z
            ])
        elif i == 2:  # at (0,1,0)
            phi = (y*(3.*y - 1.)*(3.*y - 2.))/2
            dphi = np.array([
                0*x,
                (27.*y**2)/2 - 9.*y + 1.,
                0*z
            ])
        elif i == 3:  # at (0, 0, 1)
            phi = (z*(3.*z - 1.)*(3.*z - 2.))/2
            dphi = np.array([
                0*x,
                0*x,
                (27.*z**2)/2 - 9.*z + 1.
            ])
        elif i == 4:  # at (1/3, 0, 0)
            phi = x*(3.*x + 3.*y + 3.*z - 2.)*((9.*x)/2 + (9.*y)/2 + (9.*z)/2 - 9./2)
            dphi = np.array([
                (81.*x**2)/2 + 54.*x*y + 54.*x*z - 45.*x + (27.*y**2)/2 
                + 27.*y*z - (45.*y)/2 + (27.*z**2)/2 - (45.*z)/2 + 9.,
                (9.*x*(6.*x + 6.*y + 6.*z - 5.))/2,
                (9.*x*(6.*x + 6.*y + 6.*z - 5.))/2
            ])
        elif i == 5:  # at (2/3, 0, 0)
            phi = -(9.*x*(3.*x - 1.)*(x + y + z - 1.))/2
            dphi = np.array([
                36.*x + (9.*y)/2 + (9.*z)/2 - 27.*x*y - 27.*x*z - (81.*x**2)/2 - 9./2,
                -(9.*x*(3.*x - 1.))/2,
                -(9.*x*(3.*x - 1.))/2
            ])
        elif i == 6:  # at (2/3, 1/3, 0)
            phi = (9.*x*y*(3.*x - 1.))/2
            dphi = np.array([
                (9.*y*(6.*x - 1.))/2,
                (9.*x*(3.*x - 1.))/2,
                0*z
            ])
        elif i == 7:  # at (1/3, 2/3, 0)
            phi = (9.*x*y*(3.*y - 1.))/2
            dphi = np.array([
                (9.*y*(3.*y - 1.))/2,
                (9.*x*(6.*y - 1.))/2,
                0*z
            ])
        elif i == 8:  # at (0, 1/3, 0)
            phi = y*(3.*x + 3.*y + 3.*z - 2.)*((9.*x)/2 + (9.*y)/2 + (9.*z)/2 - 9./2)
            dphi = np.array([
                (9.*y*(6.*x + 6.*y + 6.*z - 5.))/2,
                (27.*x**2)/2 + 54.*x*y + 27.*x*z - (45.*x)/2 + (81.*y**2)/2 
                + 54.*y*z - 45.*y + (27.*z**2)/2 - (45.*z)/2 + 9.,
                (9.*y*(6.*x + 6.*y + 6.*z - 5.))/2
            ])
        elif i == 9:  # at (0, 2/3, 0)
            phi = -(9.*y*(3.*y - 1.)*(x + y + z - 1.))/2
            dphi = np.array([
                -(9.*y*(3.*y - 1.))/2,
                (9.*x)/2 + 36.*y + (9.*z)/2 - 27.*x*y - 27.*y*z - (81.*y**2)/2 - 9./2,
                -(9.*y*(3.*y - 1.))/2
            ])
        elif i == 10:  # at (0, 0, 1/3)
            phi = z*(3.*x + 3.*y + 3.*z - 2.)*((9.*x)/2 + (9.*y)/2 + (9.*z)/2 - 9./2)
            dphi = np.array([
                (9.*z*(6.*x + 6.*y + 6.*z - 5.))/2,
                (9.*z*(6.*x + 6.*y + 6.*z - 5.))/2,
                (27.*x**2)/2 + 27.*x*y + 54.*x*z - (45.*x)/2 
                + (27.*y**2)/2 + 54.*y*z - (45.*y)/2 + (81.*z**2)/2 - 45.*z + 9.
            ])
        elif i == 11:  # at (0, 0, 2/3)
            phi = -(9.*z*(3.*z - 1.)*(x + y + z - 1.))/2
            dphi = np.array([
                -(9.*z*(3.*z - 1.))/2,
                -(9.*z*(3.*z - 1.))/2,
                (9.*x)/2 + (9.*y)/2 + 36.*z - 27.*x*z - 27.*y*z 
                - (81.*z**2)/2 - 9./2
            ])
        elif i == 12:  # at (2/3, 0, 1/3)
            phi = (9.*x*z*(3.*x - 1.))/2
            dphi = np.array([
                (9.*z*(6.*x - 1.))/2,
                0*y,
                (9.*x*(3.*x - 1.))/2
            ])
        elif i == 13:  # at (1/3, 0.0, 2/3)
            phi = (9.*x*z*(3.*z - 1.))/2
            dphi = np.array([
                (9.*z*(3.*z - 1.))/2,
                0*y,
                (9.*x*(6.*z - 1.))/2
            ])
        elif i == 14:  # at (0, 2/3, 1/3)
            phi = (9.*y*z*(3.*y - 1.))/2
            dphi = np.array([
                0*x,
                (9.*z*(6.*y - 1.))/2,
                (9.*y*(3.*y - 1.))/2
            ])
        elif i == 15:  # at (0, 1/3, 2/3)
            phi = (9.*y*z*(3.*z - 1.))/2
            dphi = np.array([
                0*x,
                (9.*z*(3.*z - 1.))/2,
                (9.*y*(6.*z - 1.))/2
            ])
        elif i == 16:  # at (1/3, 1/3, 0)
            phi = -x*y*(27.*x + 27.*y + 27.*z - 27.)
            dphi = np.array([
                -27.*y*(2.*x + y + z - 1.),
                -27.*x*(x + 2.*y + z - 1.),
                -27.*x*y
            ])
        elif i == 17:  # at (1/3, 0, 1/3)
            phi = -x*z*(27.*x + 27.*y + 27.*z - 27.)
            dphi = np.array([
                -27.*z*(2.*x + y + z - 1.),
                -27.*x*z,
                -27.*x*(x + y + 2.*z - 1.)
            ])
        elif i == 18:  # at (0, 1/3, 1/3)
            phi = -y*z*(27.*x + 27.*y + 27.*z - 27.)
            dphi = np.array([
                -27.*y*z,
                -27.*z*(x + 2.*y + z - 1.),
                -27.*y*(x + y + 2.*z - 1.)
            ]) 
        elif i == 19:  # at (1/3, 1/3, 1/3)
            phi = 27.*x*y*z
            dphi = np.array([
                27.*y*z,
                27.*x*z,
                27.*x*y
            ])
        else:
            self._index_error()

        return phi, dphi
