"""Functions to bake entire Ladybug core objects into the Rhino scene.

The methods here are intended to help translate groups of geometry that are commonly
generated by several objects in Ladybug core (ie. legends, compasses, etc.)
"""

from .bakegeometry import add_mesh3d_to_scene
from .baketext import add_text_to_scene


def add_legend_to_scene(legend, layer_name=None):
    """Add a Ladybug Legend object to the Rhino scene.

    Args:
        legend: A Ladybug Legend object to be added to the Rhino scene.
        layer_name: Optional text string for the layer name on which to place the
            text. If None, text will be added to the current layer.

    Returns:
        A list of IDs that point to the objects in the Rhino scene in the following
        order:

        -   legend_mesh -- A colored mesh for the legend.

        -   legend_title -- A text object for the legend title.

        -   legend_text -- Text objects for the rest of the legend text.
    """
    _height = legend.legend_parameters.text_height
    _font = legend.legend_parameters.font
    legend_mesh = add_mesh3d_to_scene(legend.segment_mesh, layer_name)
    legend_title = add_text_to_scene(legend.title, legend.title_location, _height, _font)
    if legend.legend_parameters.continuous_legend is False:
        legend_text = [add_text_to_scene(txt, loc, _height, _font, 0, 5, layer_name)
                       for txt, loc in
                       zip(legend.segment_text, legend.segment_text_location)]
    elif legend.legend_parameters.vertical is True:
        legend_text = [add_text_to_scene(txt, loc, _height, _font, 0, 3, layer_name)
                       for txt, loc in
                       zip(legend.segment_text, legend.segment_text_location)]
    else:
        legend_text = [add_text_to_scene(txt, loc, _height, _font, 1, 5, layer_name) for
                       txt, loc in
                       zip(legend.segment_text, legend.segment_text_location)]
    return [legend_mesh] + [legend_title] + legend_text
