"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithCondition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const operators_1 = require("../operators");
const _1_base_1 = require("./1-base");
/**
 * Adds "condition" functionality to the Policy Statement
 */
class PolicyStatementWithCondition extends _1_base_1.PolicyStatementBase {
    constructor() {
        super(...arguments);
        this.floydConditions = {};
        this.cdkConditionsApplied = false;
    }
    /**
     * Injects conditions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyConditions();
            return super.toJSON();
        }
        const statement = super.toJSON();
        if (this.hasConditions()) {
            statement.Condition = this.floydConditions;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyConditions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    freeze() {
        // @ts-ignore only available after swapping 1-base
        if (!this.frozen) {
            this.cdkApplyConditions();
        }
        // @ts-ignore only available after swapping 1-base
        return super.freeze();
    }
    cdkApplyConditions() {
        if (this.hasConditions() && !this.cdkConditionsApplied) {
            Object.keys(this.floydConditions).forEach((operator) => {
                Object.keys(this.floydConditions[operator]).forEach((key) => {
                    const condition = {};
                    condition[key] = this.floydConditions[operator][key];
                    // @ts-ignore only available after swapping 1-base
                    this.addCondition(operator, condition);
                });
            });
            this.cdkConditionsApplied = true;
        }
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.floydConditions).length > 0;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (this.servicePrefix.length && key.indexOf(':') < 0) {
            key = this.servicePrefix + ':' + key;
        }
        if (typeof operator === 'undefined') {
            operator = new operators_1.Operator().stringLike();
        }
        var op = '';
        if (typeof operator === 'string') {
            op = operator;
        }
        else {
            op = operator.toString();
        }
        // For boolean/number operators, IAM accepts both, booleans/numbers and
        // their string representation. To be consistent with how the IAM console
        // displays/generates the values, we convert them to strings.
        if (['boolean', 'number'].includes(typeof value)) {
            value = value.toString();
        }
        if (!(op in this.floydConditions)) {
            this.floydConditions[op] = {};
        }
        this.floydConditions[op][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifAwsCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator || new operators_1.Operator().forAnyValue().stringEquals());
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:CurrentTime', value, operator || new operators_1.Operator().dateLessThanEquals());
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:EpochTime', value, operator || new operators_1.Operator().dateLessThanEquals());
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator || new operators_1.Operator().numericLessThan());
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if(`aws:MultiFactorAuthPresent`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator || new operators_1.Operator().arnLike());
    }
    /**
     * Check whether the call to your resource is being made directly by an AWS [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services). For example, AWS CloudTrail uses the service principal `cloudtrail.amazonaws.com` to write logs to your Amazon S3 bucket. The request context key is set to true when a service uses a service principal to perform a direct action on your resources. The context key is set to false if the service uses the credentials of an IAM principal to make a request on the principal's behalf. It is also set to false if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     *
     * **Availability:** This key is present in the request context for all signed API requests that use AWS credentials.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalisawsservice
     *
     * @param value Weather the call to your resource is being made directly by an AWS service principal. **Default:** `true`
     */
    ifAwsPrincipalIsAWSService(value) {
        return this.if(`aws:PrincipalIsAWSService`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) name in the policy with the service principal that is making requests to your resources. You can use this key to check whether this call is made by a specific service principal. When a service principal makes a direct request to your resource, the `aws:PrincipalServiceName` key contains the name of the service principal. For example, the AWS CloudTrail service principal name is `cloudtrail.amazonaws.com`.
     *
     * **Availability:** This key is present in the request when the call is made by an AWS service principal. This key is not present in any other situation, including the following:
     *
     * - If the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     * - If the service uses the credentials of an IAM principal to make a request on the principal's behalf.
     * - If the call is made directly by an IAM principal.
     *
     * You can use this condition key to limit access to your trusted identities and expected network locations, while safely granting access to an AWS service.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalservicename
     *
     * @param value AWS service name, e.g. `cloudtrail.amazonaws.com`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalServiceName(value, operator) {
        return this.if('aws:PrincipalServiceName', value, operator);
    }
    /**
     * This key provides a list of all [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) names that belong to the service. This is an advanced condition key. You can use it to restrict the service from accessing your resource from a specific Region only. Some services may create Regional service principals to indicate a particular instance of the service within a specific Region. You can limit access to a resource to a particular instance of the service. When a service principal makes a direct request to your resource, the `aws:PrincipalServiceNamesList` contains an unordered list of all service principal names associated with the Regional instance of the service.
     *
     * **Availability:** This key is present in the request when the call is made by an AWS service principal. This key is not present in any other situation, including the following:
     *
     * - If the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     * - If the service uses the credentials of an IAM principal to make a request on the principal's behalf.
     * - If the call is made directly by an IAM principal.
     *
     * `aws:PrincipalServiceNamesList` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. You must use the `ForAnyValue` or `ForAllValues` set operators with the `StringLike` [condition operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String) when you use this key. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array, such as `("Key":["Value1", "Value2"])`. You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Using multiple keys and values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalservicenameslist
     *
     * @param value AWS service name list
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringLike`
     */
    ifAwsPrincipalServiceNamesList(value, operator) {
        return this.if('aws:PrincipalServiceNamesList', value, operator || new operators_1.Operator().stringLike().forAnyValue());
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if(`aws:SecureTransport`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator || new operators_1.Operator().arnLike());
    }
    /**
     * Compare the source identity that was set by the principal with the source identity that you specify in the policy.
     *
     * **Availability:** This key is included in the request context after a source identity has been set when a role is assumed using any AWS STS assume-role CLI command, or AWS STS `AssumeRole` API operation.
     *
     * You can use this key in a policy to allow actions in AWS by principals that have set a source identity when assuming a role. Activity for the role's specified source identity appears in [AWS CloudTrail](https://docs.aws.amazon.com/IAM/latest/UserGuide/cloudtrail-integration.html#cloudtrail-integration_signin-tempcreds). This makes it easier for administrators to determine who or what performed actions with a role in AWS.
     *
     * Unlike [sts:RoleSessionName](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_rolesessionname), after the source identity is set, the value cannot be changed. It is present in the request context for all actions taken by the role. The value persists into subsequent role sessions when you use the session credentials to assume another role. Assuming one role from another is called [role chaining](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-role-chaining).
     *
     * The [sts:SourceIdentity](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_sourceidentity) key is present in the request when the principal initially sets a source identity while assuming a role using any AWS STS assume-role CLI command, or AWS STS `AssumeRole` API operation. The `aws:SourceIdentity` key is present in the request for any actions that are taken with a role session that has a source identity set.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceidentity
     *
     * @param value The source identity
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceIdentity(value, operator) {
        return this.if('aws:SourceIdentity', value, operator);
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator || new operators_1.Operator().ipAddress());
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifAwsTokenIssueTime(value, operator) {
        return this.if('aws:TokenIssueTime', dateToString(value), operator || new operators_1.Operator().dateGreaterThanEquals());
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifAwsViaAWSService(value) {
        return this.if(`aws:ViaAWSService`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator || new operators_1.Operator().ipAddress());
    }
}
exports.PolicyStatementWithCondition = PolicyStatementWithCondition;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithCondition[_a] = { fqn: "iam-floyd.PolicyStatementWithCondition", version: "0.391.0" };
function dateToString(value) {
    if (typeof value.getMonth === 'function') {
        value = value.toISOString();
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiMi1jb25kaXRpb25zLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiMi1jb25kaXRpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNENBQXdDO0FBQ3hDLHNDQUErQztBQWdCL0M7O0dBRUc7QUFDSCxNQUFhLDRCQUE2QixTQUFRLDZCQUFtQjtJQUFyRTs7UUFDWSxvQkFBZSxHQUFlLEVBQUUsQ0FBQztRQUNuQyx5QkFBb0IsR0FBRyxLQUFLLENBQUM7S0FneUJ0QztJQTl4QkM7Ozs7T0FJRztJQUNJLE1BQU07UUFDWCxrREFBa0Q7UUFDbEQsSUFBSSxPQUFPLElBQUksQ0FBQyxZQUFZLElBQUksVUFBVSxFQUFFO1lBQzFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1lBQzFCLE9BQU8sS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1NBQ3ZCO1FBQ0QsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBRWpDLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRSxFQUFFO1lBQ3hCLFNBQVMsQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQztTQUM1QztRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFTSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBQzFCLGtEQUFrRDtRQUNsRCxPQUFPLEtBQUssQ0FBQyxlQUFlLEVBQUUsQ0FBQztJQUNqQyxDQUFDO0lBRU0sTUFBTTtRQUNYLGtEQUFrRDtRQUNsRCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNoQixJQUFJLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztTQUMzQjtRQUNELGtEQUFrRDtRQUNsRCxPQUFPLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztJQUN4QixDQUFDO0lBRU8sa0JBQWtCO1FBQ3hCLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixFQUFFO1lBQ3RELE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUFFO2dCQUNyRCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtvQkFDMUQsTUFBTSxTQUFTLEdBQVEsRUFBRSxDQUFDO29CQUMxQixTQUFTLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztvQkFDckQsa0RBQWtEO29CQUNsRCxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztnQkFDekMsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUM7U0FDbEM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksRUFBRSxDQUFDLEdBQVcsRUFBRSxLQUFVLEVBQUUsUUFBNEI7UUFDN0QsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUNyRCxHQUFHLEdBQUcsSUFBSSxDQUFDLGFBQWEsR0FBRyxHQUFHLEdBQUcsR0FBRyxDQUFDO1NBQ3RDO1FBRUQsSUFBSSxPQUFPLFFBQVEsS0FBSyxXQUFXLEVBQUU7WUFDbkMsUUFBUSxHQUFHLElBQUksb0JBQVEsRUFBRSxDQUFDLFVBQVUsRUFBRSxDQUFDO1NBQ3hDO1FBRUQsSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDO1FBQ1osSUFBSSxPQUFPLFFBQVEsS0FBSyxRQUFRLEVBQUU7WUFDaEMsRUFBRSxHQUFHLFFBQVEsQ0FBQztTQUNmO2FBQU07WUFDTCxFQUFFLEdBQUcsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO1NBQzFCO1FBRUQsdUVBQXVFO1FBQ3ZFLHlFQUF5RTtRQUN6RSw2REFBNkQ7UUFDN0QsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxLQUFLLENBQUMsRUFBRTtZQUNoRCxLQUFLLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDO1NBQzFCO1FBRUQsSUFBSSxDQUFDLENBQUMsRUFBRSxJQUFJLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRTtZQUNqQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQztTQUMvQjtRQUNELElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO1FBRXRDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBRUksY0FBYyxDQUNuQixLQUF3QixFQUN4QixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQ1osZUFBZSxFQUNmLEtBQUssRUFDTCxRQUFRLElBQUksSUFBSSxvQkFBUSxFQUFFLENBQUMsV0FBVyxFQUFFLENBQUMsWUFBWSxFQUFFLENBQ3hELENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUJBQW1CLENBQ3hCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGtCQUFrQixDQUN2QixLQUF3QixFQUN4QixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQkFBZ0IsQ0FDckIsS0FBd0MsRUFDeEMsUUFBNEI7UUFFNUIsSUFBSSxPQUFRLEtBQWMsQ0FBQyxRQUFRLEtBQUssVUFBVSxFQUFFO1lBQ2xELEtBQUssR0FBSSxLQUFjLENBQUMsV0FBVyxFQUFFLENBQUM7U0FDdkM7YUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDL0IsS0FBSyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtnQkFDekIsSUFBSSxPQUFRLElBQWEsQ0FBQyxRQUFRLEtBQUssVUFBVSxFQUFFO29CQUNqRCxJQUFJLEdBQUksSUFBYSxDQUFDLFdBQVcsRUFBRSxDQUFDO2lCQUNyQztnQkFDRCxPQUFPLElBQUksQ0FBQztZQUNkLENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQ1osaUJBQWlCLEVBQ2pCLEtBQUssRUFDTCxRQUFRLElBQUksSUFBSSxvQkFBUSxFQUFFLENBQUMsa0JBQWtCLEVBQUUsQ0FDaEQsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxjQUFjLENBQ25CLEtBQTBELEVBQzFELFFBQTRCO1FBRTVCLElBQUksT0FBUSxLQUFjLENBQUMsUUFBUSxLQUFLLFVBQVUsRUFBRTtZQUNsRCxLQUFLLEdBQUksS0FBYyxDQUFDLFdBQVcsRUFBRSxDQUFDO1NBQ3ZDO2FBQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQy9CLEtBQUssR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7Z0JBQ3pCLElBQUksT0FBUSxJQUFhLENBQUMsUUFBUSxLQUFLLFVBQVUsRUFBRTtvQkFDakQsSUFBSSxHQUFJLElBQWEsQ0FBQyxXQUFXLEVBQUUsQ0FBQztpQkFDckM7Z0JBQ0QsT0FBTyxJQUFJLENBQUM7WUFDZCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBQ0QsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUNaLGVBQWUsRUFDZixLQUFLLEVBQ0wsUUFBUSxJQUFJLElBQUksb0JBQVEsRUFBRSxDQUFDLGtCQUFrQixFQUFFLENBQ2hELENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBRUksdUJBQXVCLENBQzVCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWix3QkFBd0IsRUFDeEIsS0FBSyxFQUNMLFFBQVEsSUFBSSxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxlQUFlLEVBQUUsQ0FDN0MsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSwyQkFBMkIsQ0FBQyxLQUFlO1FBQ2hELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWiw0QkFBNEIsRUFDNUIsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksRUFDM0MsSUFBSSxvQkFBUSxFQUFFLENBQUMsSUFBSSxFQUFFLENBQ3RCLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCLENBQzFCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksaUJBQWlCLENBQ3RCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWixrQkFBa0IsRUFDbEIsS0FBSyxFQUNMLFFBQVEsSUFBSSxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FDckMsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLDBCQUEwQixDQUFDLEtBQWU7UUFDL0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUNaLDJCQUEyQixFQUMzQixPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUMzQyxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FDdEIsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1CQUFtQixDQUN4QixLQUF3QixFQUN4QixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxzQkFBc0IsQ0FDM0IsS0FBd0IsRUFDeEIsUUFBNEI7UUFFNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0kseUJBQXlCLENBQzlCLEtBQWEsRUFDYixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSw4QkFBOEIsQ0FDbkMsS0FBZSxFQUNmLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWiwrQkFBK0IsRUFDL0IsS0FBSyxFQUNMLFFBQVEsSUFBSSxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FDdEQsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxpQkFBaUIsQ0FDdEIsR0FBVyxFQUNYLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsR0FBRyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQkFBa0IsQ0FDdkIsS0FBd0IsRUFDeEIsUUFBNEI7UUFFNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksb0JBQW9CLENBQ3pCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWUsQ0FDcEIsR0FBVyxFQUNYLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsR0FBRyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZ0JBQWdCLENBQ3JCLEdBQVcsRUFDWCxLQUF3QixFQUN4QixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLEdBQUcsRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxvQkFBb0IsQ0FBQyxLQUFlO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWixxQkFBcUIsRUFDckIsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksRUFDM0MsSUFBSSxvQkFBUSxFQUFFLENBQUMsSUFBSSxFQUFFLENBQ3RCLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxrQkFBa0IsQ0FDdkIsS0FBd0IsRUFDeEIsUUFBNEI7UUFFNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGNBQWMsQ0FDbkIsS0FBd0IsRUFDeEIsUUFBNEI7UUFFNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUNaLGVBQWUsRUFDZixLQUFLLEVBQ0wsUUFBUSxJQUFJLElBQUksb0JBQVEsRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUNyQyxDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLG1CQUFtQixDQUN4QixLQUF3QixFQUN4QixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWixjQUFjLEVBQ2QsS0FBSyxFQUNMLFFBQVEsSUFBSSxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxTQUFTLEVBQUUsQ0FDdkMsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxjQUFjLENBQ25CLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlLENBQ3BCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxtQkFBbUIsQ0FDeEIsS0FBb0IsRUFDcEIsUUFBNEI7UUFFNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUNaLG9CQUFvQixFQUNwQixZQUFZLENBQUMsS0FBSyxDQUFDLEVBQ25CLFFBQVEsSUFBSSxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxDQUNuRCxDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksY0FBYyxDQUNuQixLQUF3QixFQUN4QixRQUE0QjtRQUU1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksV0FBVyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDdkUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksa0JBQWtCLENBQUMsS0FBZTtRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQ1osbUJBQW1CLEVBQ25CLE9BQU8sS0FBSyxLQUFLLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQzNDLElBQUksb0JBQVEsRUFBRSxDQUFDLElBQUksRUFBRSxDQUN0QixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCLENBQ3JCLEtBQXdCLEVBQ3hCLFFBQTRCO1FBRTVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FDWixpQkFBaUIsRUFDakIsS0FBSyxFQUNMLFFBQVEsSUFBSSxJQUFJLG9CQUFRLEVBQUUsQ0FBQyxTQUFTLEVBQUUsQ0FDdkMsQ0FBQztJQUNKLENBQUM7O0FBanlCSCxvRUFreUJDOzs7QUFFRCxTQUFTLFlBQVksQ0FBQyxLQUE2QjtJQUNqRCxJQUFJLE9BQVEsS0FBYyxDQUFDLFFBQVEsS0FBSyxVQUFVLEVBQUU7UUFDbEQsS0FBSyxHQUFJLEtBQWMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztLQUN2QztJQUNELE9BQU8sS0FBZSxDQUFDO0FBQ3pCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBPcGVyYXRvciB9IGZyb20gJy4uL29wZXJhdG9ycyc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnRCYXNlIH0gZnJvbSAnLi8xLWJhc2UnO1xuXG4vKipcbiAqIEEgQ29uZGl0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29uZGl0aW9uIHtcbiAgW2tleTogc3RyaW5nXTogU3RyaW5nO1xufVxuXG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiBDb25kaXRpb24nc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbmRpdGlvbnMge1xuICBba2V5OiBzdHJpbmddOiBDb25kaXRpb247XG59XG5cbi8qKlxuICogQWRkcyBcImNvbmRpdGlvblwiIGZ1bmN0aW9uYWxpdHkgdG8gdGhlIFBvbGljeSBTdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFBvbGljeVN0YXRlbWVudFdpdGhDb25kaXRpb24gZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnRCYXNlIHtcbiAgcHJvdGVjdGVkIGZsb3lkQ29uZGl0aW9uczogQ29uZGl0aW9ucyA9IHt9O1xuICBwcml2YXRlIGNka0NvbmRpdGlvbnNBcHBsaWVkID0gZmFsc2U7XG5cbiAgLyoqXG4gICAqIEluamVjdHMgY29uZGl0aW9ucyBpbnRvIHRoZSBzdGF0ZW1lbnQuXG4gICAqXG4gICAqIE9ubHkgcmVsZXZhbnQgZm9yIHRoZSBtYWluIHBhY2thZ2UuIEluIENESyBtb2RlIHRoaXMgb25seSBjYWxscyBzdXBlci5cbiAgICovXG4gIHB1YmxpYyB0b0pTT04oKTogYW55IHtcbiAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgIGlmICh0eXBlb2YgdGhpcy5hZGRSZXNvdXJjZXMgPT0gJ2Z1bmN0aW9uJykge1xuICAgICAgdGhpcy5jZGtBcHBseUNvbmRpdGlvbnMoKTtcbiAgICAgIHJldHVybiBzdXBlci50b0pTT04oKTtcbiAgICB9XG4gICAgY29uc3Qgc3RhdGVtZW50ID0gc3VwZXIudG9KU09OKCk7XG5cbiAgICBpZiAodGhpcy5oYXNDb25kaXRpb25zKCkpIHtcbiAgICAgIHN0YXRlbWVudC5Db25kaXRpb24gPSB0aGlzLmZsb3lkQ29uZGl0aW9ucztcbiAgICB9XG5cbiAgICByZXR1cm4gc3RhdGVtZW50O1xuICB9XG5cbiAgcHVibGljIHRvU3RhdGVtZW50SnNvbigpOiBhbnkge1xuICAgIHRoaXMuY2RrQXBwbHlDb25kaXRpb25zKCk7XG4gICAgLy8gQHRzLWlnbm9yZSBvbmx5IGF2YWlsYWJsZSBhZnRlciBzd2FwcGluZyAxLWJhc2VcbiAgICByZXR1cm4gc3VwZXIudG9TdGF0ZW1lbnRKc29uKCk7XG4gIH1cblxuICBwdWJsaWMgZnJlZXplKCkge1xuICAgIC8vIEB0cy1pZ25vcmUgb25seSBhdmFpbGFibGUgYWZ0ZXIgc3dhcHBpbmcgMS1iYXNlXG4gICAgaWYgKCF0aGlzLmZyb3plbikge1xuICAgICAgdGhpcy5jZGtBcHBseUNvbmRpdGlvbnMoKTtcbiAgICB9XG4gICAgLy8gQHRzLWlnbm9yZSBvbmx5IGF2YWlsYWJsZSBhZnRlciBzd2FwcGluZyAxLWJhc2VcbiAgICByZXR1cm4gc3VwZXIuZnJlZXplKCk7XG4gIH1cblxuICBwcml2YXRlIGNka0FwcGx5Q29uZGl0aW9ucygpIHtcbiAgICBpZiAodGhpcy5oYXNDb25kaXRpb25zKCkgJiYgIXRoaXMuY2RrQ29uZGl0aW9uc0FwcGxpZWQpIHtcbiAgICAgIE9iamVjdC5rZXlzKHRoaXMuZmxveWRDb25kaXRpb25zKS5mb3JFYWNoKChvcGVyYXRvcikgPT4ge1xuICAgICAgICBPYmplY3Qua2V5cyh0aGlzLmZsb3lkQ29uZGl0aW9uc1tvcGVyYXRvcl0pLmZvckVhY2goKGtleSkgPT4ge1xuICAgICAgICAgIGNvbnN0IGNvbmRpdGlvbjogYW55ID0ge307XG4gICAgICAgICAgY29uZGl0aW9uW2tleV0gPSB0aGlzLmZsb3lkQ29uZGl0aW9uc1tvcGVyYXRvcl1ba2V5XTtcbiAgICAgICAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgICAgICAgIHRoaXMuYWRkQ29uZGl0aW9uKG9wZXJhdG9yLCBjb25kaXRpb24pO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgICAgdGhpcy5jZGtDb25kaXRpb25zQXBwbGllZCA9IHRydWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrcyB3ZWF0aGVyIGEgY29uZGl0aW9uIHdhcyBhcHBsaWVkIHRvIHRoZSBwb2xpY3kuXG4gICAqL1xuICBwdWJsaWMgaGFzQ29uZGl0aW9ucygpOiBib29sZWFuIHtcbiAgICByZXR1cm4gT2JqZWN0LmtleXModGhpcy5mbG95ZENvbmRpdGlvbnMpLmxlbmd0aCA+IDA7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGNvbmRpdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBrZXkgVGhlIGtleSBvZiB0aGUgY29uZGl0aW9uXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2sgZm9yXG4gICAqIEBwYXJhbSBvcGVyYXRvciBbT3BlcmF0b3JdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sKSBvZiB0aGUgY29uZGl0aW9uLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWYoa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnksIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICBpZiAodGhpcy5zZXJ2aWNlUHJlZml4Lmxlbmd0aCAmJiBrZXkuaW5kZXhPZignOicpIDwgMCkge1xuICAgICAga2V5ID0gdGhpcy5zZXJ2aWNlUHJlZml4ICsgJzonICsga2V5O1xuICAgIH1cblxuICAgIGlmICh0eXBlb2Ygb3BlcmF0b3IgPT09ICd1bmRlZmluZWQnKSB7XG4gICAgICBvcGVyYXRvciA9IG5ldyBPcGVyYXRvcigpLnN0cmluZ0xpa2UoKTtcbiAgICB9XG5cbiAgICB2YXIgb3AgPSAnJztcbiAgICBpZiAodHlwZW9mIG9wZXJhdG9yID09PSAnc3RyaW5nJykge1xuICAgICAgb3AgPSBvcGVyYXRvcjtcbiAgICB9IGVsc2Uge1xuICAgICAgb3AgPSBvcGVyYXRvci50b1N0cmluZygpO1xuICAgIH1cblxuICAgIC8vIEZvciBib29sZWFuL251bWJlciBvcGVyYXRvcnMsIElBTSBhY2NlcHRzIGJvdGgsIGJvb2xlYW5zL251bWJlcnMgYW5kXG4gICAgLy8gdGhlaXIgc3RyaW5nIHJlcHJlc2VudGF0aW9uLiBUbyBiZSBjb25zaXN0ZW50IHdpdGggaG93IHRoZSBJQU0gY29uc29sZVxuICAgIC8vIGRpc3BsYXlzL2dlbmVyYXRlcyB0aGUgdmFsdWVzLCB3ZSBjb252ZXJ0IHRoZW0gdG8gc3RyaW5ncy5cbiAgICBpZiAoWydib29sZWFuJywgJ251bWJlciddLmluY2x1ZGVzKHR5cGVvZiB2YWx1ZSkpIHtcbiAgICAgIHZhbHVlID0gdmFsdWUudG9TdHJpbmcoKTtcbiAgICB9XG5cbiAgICBpZiAoIShvcCBpbiB0aGlzLmZsb3lkQ29uZGl0aW9ucykpIHtcbiAgICAgIHRoaXMuZmxveWRDb25kaXRpb25zW29wXSA9IHt9O1xuICAgIH1cbiAgICB0aGlzLmZsb3lkQ29uZGl0aW9uc1tvcF1ba2V5XSA9IHZhbHVlO1xuXG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgc2VydmljZXMgd2l0aCB0aGUgc2VydmljZXMgdGhhdCBtYWRlIHJlcXVlc3RzIG9uIGJlaGFsZiBvZiB0aGUgSUFNIHByaW5jaXBhbCAodXNlciBvciByb2xlKS4gV2hlbiBhIHByaW5jaXBhbCBtYWtlcyBhIHJlcXVlc3QgdG8gYW4gQVdTIHNlcnZpY2UsIHRoYXQgc2VydmljZSBtaWdodCB1c2UgdGhlIHByaW5jaXBhbCdzIGNyZWRlbnRpYWxzIHRvIG1ha2Ugc3Vic2VxdWVudCByZXF1ZXN0cyB0byBvdGhlciBzZXJ2aWNlcy5cbiAgICpcbiAgICogVGhlIGBhd3M6Q2FsbGVkVmlhYCBrZXkgY29udGFpbnMgYW4gb3JkZXJlZCBsaXN0IG9mIGVhY2ggc2VydmljZSBpbiB0aGUgY2hhaW4gdGhhdCBtYWRlIHJlcXVlc3RzIG9uIHRoZSBwcmluY2lwYWwncyBiZWhhbGYuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIHByZXNlbnQgaW4gdGhlIHJlcXVlc3Qgd2hlbiBhIHNlcnZpY2UgdGhhdCBzdXBwb3J0cyBhd3M6Q2FsbGVkVmlhIHVzZXMgdGhlIGNyZWRlbnRpYWxzIG9mIGFuIElBTSBwcmluY2lwYWwgdG8gbWFrZSBhIHJlcXVlc3QgdG8gYW5vdGhlciBzZXJ2aWNlLiBUaGlzIGtleSBpcyBub3QgcHJlc2VudCBpZiB0aGUgc2VydmljZSB1c2VzIGEgW3NlcnZpY2Ugcm9sZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Rlcm1zLWFuZC1jb25jZXB0cy5odG1sI2lhbS10ZXJtLXNlcnZpY2Utcm9sZSkgb3IgW3NlcnZpY2UtbGlua2VkIHJvbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1zZXJ2aWNlLWxpbmtlZC1yb2xlKSB0byBtYWtlIGEgY2FsbCBvbiB0aGUgcHJpbmNpcGFsJ3MgYmVoYWxmLiBUaGlzIGtleSBpcyBhbHNvIG5vdCBwcmVzZW50IHdoZW4gdGhlIHByaW5jaXBhbCBtYWtlcyB0aGUgY2FsbCBkaXJlY3RseS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLWNhbGxlZHZpYVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHNlcnZpY2UocykgdG8gY2hlY2sgZm9yXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgRm9yQW55VmFsdWU6U3RyaW5nRXF1YWxzYFxuICAgKi9cblxuICBwdWJsaWMgaWZBd3NDYWxsZWRWaWEoXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoXG4gICAgICAnYXdzOkNhbGxlZFZpYScsXG4gICAgICB2YWx1ZSxcbiAgICAgIG9wZXJhdG9yIHx8IG5ldyBPcGVyYXRvcigpLmZvckFueVZhbHVlKCkuc3RyaW5nRXF1YWxzKClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIHNlcnZpY2VzIHdpdGggdGhlIGZpcnN0IHNlcnZpY2UgdGhhdCBtYWRlIGEgcmVxdWVzdCBvbiBiZWhhbGYgb2YgdGhlIElBTSBwcmluY2lwYWwgKHVzZXIgb3Igcm9sZSkuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgW2F3czpDYWxsZWRWaWFdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1jYWxsZWR2aWEpLlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBwcmVzZW50IGluIHRoZSByZXF1ZXN0IHdoZW4gYSBzZXJ2aWNlIHRoYXQgc3VwcG9ydHMgYXdzOkNhbGxlZFZpYSB1c2VzIHRoZSBjcmVkZW50aWFscyBvZiBhbiBJQU0gcHJpbmNpcGFsIHRvIG1ha2UgYSByZXF1ZXN0IHRvIGFub3RoZXIgc2VydmljZS4gVGhpcyBrZXkgaXMgbm90IHByZXNlbnQgaWYgdGhlIHNlcnZpY2UgdXNlcyBhIFtzZXJ2aWNlIHJvbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1zZXJ2aWNlLXJvbGUpIG9yIFtzZXJ2aWNlLWxpbmtlZCByb2xlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfdGVybXMtYW5kLWNvbmNlcHRzLmh0bWwjaWFtLXRlcm0tc2VydmljZS1saW5rZWQtcm9sZSkgdG8gbWFrZSBhIGNhbGwgb24gdGhlIHByaW5jaXBhbCdzIGJlaGFsZi4gVGhpcyBrZXkgaXMgYWxzbyBub3QgcHJlc2VudCB3aGVuIHRoZSBwcmluY2lwYWwgbWFrZXMgdGhlIGNhbGwgZGlyZWN0bHkuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1jYWxsZWR2aWFmaXJzdFxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHNlcnZpY2UocykgdG8gY2hlY2sgZm9yXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c0NhbGxlZFZpYUZpcnN0KFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6Q2FsbGVkVmlhRmlyc3QnLCB2YWx1ZSwgb3BlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIHNlcnZpY2VzIHdpdGggdGhlIGxhc3Qgc2VydmljZSB0aGF0IG1hZGUgYSByZXF1ZXN0IG9uIGJlaGFsZiBvZiB0aGUgSUFNIHByaW5jaXBhbCAodXNlciBvciByb2xlKS4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBbYXdzOkNhbGxlZFZpYV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLWNhbGxlZHZpYSkuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIHByZXNlbnQgaW4gdGhlIHJlcXVlc3Qgd2hlbiBhIHNlcnZpY2UgdGhhdCBzdXBwb3J0cyBhd3M6Q2FsbGVkVmlhIHVzZXMgdGhlIGNyZWRlbnRpYWxzIG9mIGFuIElBTSBwcmluY2lwYWwgdG8gbWFrZSBhIHJlcXVlc3QgdG8gYW5vdGhlciBzZXJ2aWNlLiBUaGlzIGtleSBpcyBub3QgcHJlc2VudCBpZiB0aGUgc2VydmljZSB1c2VzIGEgW3NlcnZpY2Ugcm9sZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Rlcm1zLWFuZC1jb25jZXB0cy5odG1sI2lhbS10ZXJtLXNlcnZpY2Utcm9sZSkgb3IgW3NlcnZpY2UtbGlua2VkIHJvbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1zZXJ2aWNlLWxpbmtlZC1yb2xlKSB0byBtYWtlIGEgY2FsbCBvbiB0aGUgcHJpbmNpcGFsJ3MgYmVoYWxmLiBUaGlzIGtleSBpcyBhbHNvIG5vdCBwcmVzZW50IHdoZW4gdGhlIHByaW5jaXBhbCBtYWtlcyB0aGUgY2FsbCBkaXJlY3RseS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLWNhbGxlZHZpYWxhc3RcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSBzZXJ2aWNlKHMpIHRvIGNoZWNrIGZvclxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NDYWxsZWRWaWFMYXN0KFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6Q2FsbGVkVmlhTGFzdCcsIHZhbHVlLCBvcGVyYXRvcik7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgZGF0ZSBhbmQgdGltZSBvZiB0aGUgcmVxdWVzdCB3aXRoIHRoZSBkYXRlIGFuZCB0aW1lIHRoYXQgeW91IHNwZWNpZnkuIFRvIHZpZXcgYW4gZXhhbXBsZSBwb2xpY3kgdGhhdCB1c2VzIHRoaXMgY29uZGl0aW9uIGtleSwgc2VlIFtBV1M6IEFsbG93cyBBY2Nlc3MgV2l0aGluIFNwZWNpZmljIERhdGVzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2V4YW1wbGVzX2F3cy1kYXRlcy5odG1sKS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgYWx3YXlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1jdXJyZW50dGltZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIGRhdGUgYW5kIHRpbWUgdG8gY2hlY2sgZm9yLiBDYW4gYmUgYSBzdHJpbmcgaW4gb25lIG9mIHRoZSBbVzNDIGltcGxlbWVudGF0aW9ucyBvZiB0aGUgSVNPIDg2MDEgZGF0ZV0oaHR0cHM6Ly93d3cudzMub3JnL1RSL05PVEUtZGF0ZXRpbWUpIChlLmcuIGAyMDIwLTA0LTAxVDAwOjAwOjAwWmApIG9yIGluIGVwb2NoIChVTklYKSB0aW1lIG9yIGEgYERhdGUoKWAgb2JqZWN0IChKYXZhU2NyaXB0IG9ubHkpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtkYXRlIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19EYXRlKS4gKipEZWZhdWx0OioqIGBEYXRlTGVzc1RoYW5FcXVhbHNgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NDdXJyZW50VGltZShcbiAgICB2YWx1ZTogRGF0ZSB8IHN0cmluZyB8IChEYXRlIHwgc3RyaW5nKVtdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgaWYgKHR5cGVvZiAodmFsdWUgYXMgRGF0ZSkuZ2V0TW9udGggPT09ICdmdW5jdGlvbicpIHtcbiAgICAgIHZhbHVlID0gKHZhbHVlIGFzIERhdGUpLnRvSVNPU3RyaW5nKCk7XG4gICAgfSBlbHNlIGlmIChBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgICAgdmFsdWUgPSB2YWx1ZS5tYXAoKGl0ZW0pID0+IHtcbiAgICAgICAgaWYgKHR5cGVvZiAoaXRlbSBhcyBEYXRlKS5nZXRNb250aCA9PT0gJ2Z1bmN0aW9uJykge1xuICAgICAgICAgIGl0ZW0gPSAoaXRlbSBhcyBEYXRlKS50b0lTT1N0cmluZygpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBpdGVtO1xuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmlmKFxuICAgICAgJ2F3czpDdXJyZW50VGltZScsXG4gICAgICB2YWx1ZSxcbiAgICAgIG9wZXJhdG9yIHx8IG5ldyBPcGVyYXRvcigpLmRhdGVMZXNzVGhhbkVxdWFscygpXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSBkYXRlIGFuZCB0aW1lIG9mIHRoZSByZXF1ZXN0IGluIGVwb2NoIG9yIFVuaXggdGltZSB3aXRoIHRoZSB2YWx1ZSB0aGF0IHlvdSBzcGVjaWZ5LiBUaGlzIGtleSBhbHNvIGFjY2VwdHMgdGhlIG51bWJlciBvZiBzZWNvbmRzIHNpbmNlIEphbnVhcnkgMSwgMTk3MC5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgYWx3YXlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1lcG9jaHRpbWVcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSBkYXRlIGFuZCB0aW1lIHRvIGNoZWNrIGZvci4gQ2FuIGJlIGEgc3RyaW5nIGluIG9uZSBvZiB0aGUgW1czQyBpbXBsZW1lbnRhdGlvbnMgb2YgdGhlIElTTyA4NjAxIGRhdGVdKGh0dHBzOi8vd3d3LnczLm9yZy9UUi9OT1RFLWRhdGV0aW1lKSAoZS5nLiBgMjAyMC0wNC0wMVQwMDowMDowMFpgKSBvciBpbiBlcG9jaCAoVU5JWCkgdGltZSBvciBhIGBEYXRlKClgIG9iamVjdCAoSmF2YVNjcmlwdCBvbmx5KVxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbZGF0ZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19EYXRlKSBhbmQgW251bWVyaWMgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX051bWVyaWMpLiAqKkRlZmF1bHQ6KiogYERhdGVMZXNzVGhhbkVxdWFsc2BcbiAgICovXG4gIHB1YmxpYyBpZkF3c0Vwb2NoVGltZShcbiAgICB2YWx1ZTogbnVtYmVyIHwgRGF0ZSB8IHN0cmluZyB8IChudW1iZXIgfCBEYXRlIHwgc3RyaW5nKVtdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgaWYgKHR5cGVvZiAodmFsdWUgYXMgRGF0ZSkuZ2V0TW9udGggPT09ICdmdW5jdGlvbicpIHtcbiAgICAgIHZhbHVlID0gKHZhbHVlIGFzIERhdGUpLnRvSVNPU3RyaW5nKCk7XG4gICAgfSBlbHNlIGlmIChBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgICAgdmFsdWUgPSB2YWx1ZS5tYXAoKGl0ZW0pID0+IHtcbiAgICAgICAgaWYgKHR5cGVvZiAoaXRlbSBhcyBEYXRlKS5nZXRNb250aCA9PT0gJ2Z1bmN0aW9uJykge1xuICAgICAgICAgIGl0ZW0gPSAoaXRlbSBhcyBEYXRlKS50b0lTT1N0cmluZygpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBpdGVtO1xuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmlmKFxuICAgICAgJ2F3czpFcG9jaFRpbWUnLFxuICAgICAgdmFsdWUsXG4gICAgICBvcGVyYXRvciB8fCBuZXcgT3BlcmF0b3IoKS5kYXRlTGVzc1RoYW5FcXVhbHMoKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgbnVtYmVyIG9mIHNlY29uZHMgc2luY2UgdGhlIHJlcXVlc3RpbmcgcHJpbmNpcGFsIHdhcyBhdXRob3JpemVkIHVzaW5nIE1GQSB3aXRoIHRoZSBudW1iZXIgdGhhdCB5b3Ugc3BlY2lmeS4gRm9yIG1vcmUgaW5mb3JtYXRpb24gYWJvdXQgTUZBLCBzZWUgW1VzaW5nIE11bHRpLUZhY3RvciBBdXRoZW50aWNhdGlvbiAoTUZBKSBpbiBBV1NdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9jcmVkZW50aWFsc19tZmEuaHRtbCkuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQgb25seSBpZiB0aGUgcHJpbmNpcGFsIHdhcyBhdXRoZW50aWNhdGVkIHVzaW5nIE1GQS4gSWYgTUZBIHdhcyBub3QgdXNlZCwgdGhpcyBrZXkgaXMgbm90IHByZXNlbnQuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1tdWx0aWZhY3RvcmF1dGhhZ2VcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIE51bWJlciBvZiBzZWNvbmRzXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtudW1lcmljIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19OdW1lcmljKS4gKipEZWZhdWx0OioqIGBOdW1lcmljTGVzc1RoYW5gXG4gICAqL1xuXG4gIHB1YmxpYyBpZkF3c011bHRpRmFjdG9yQXV0aEFnZShcbiAgICB2YWx1ZTogbnVtYmVyIHwgbnVtYmVyW10sXG4gICAgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZ1xuICApIHtcbiAgICByZXR1cm4gdGhpcy5pZihcbiAgICAgICdhd3M6TXVsdGlGYWN0b3JBdXRoQWdlJyxcbiAgICAgIHZhbHVlLFxuICAgICAgb3BlcmF0b3IgfHwgbmV3IE9wZXJhdG9yKCkubnVtZXJpY0xlc3NUaGFuKClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIHdoZXRoZXIgbXVsdGktZmFjdG9yIGF1dGhlbnRpY2F0aW9uIChNRkEpIHdhcyB1c2VkIHRvIHZhbGlkYXRlIHRoZSB0ZW1wb3Jhcnkgc2VjdXJpdHkgY3JlZGVudGlhbHMgdGhhdCBtYWRlIHRoZSByZXF1ZXN0LlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IG9ubHkgd2hlbiB0aGUgcHJpbmNpcGFsIHVzZXMgdGVtcG9yYXJ5IGNyZWRlbnRpYWxzIHRvIG1ha2UgdGhlIHJlcXVlc3QuIFRoZSBrZXkgaXMgbm90IHByZXNlbnQgaW4gQVdTIENMSSwgQVdTIEFQSSwgb3IgQVdTIFNESyByZXF1ZXN0cyB0aGF0IGFyZSBtYWRlIHVzaW5nIGxvbmctdGVybSBjcmVkZW50aWFscy5cbiAgICpcbiAgICogVGVtcG9yYXJ5IGNyZWRlbnRpYWxzIGFyZSB1c2VkIHRvIGF1dGhlbnRpY2F0ZSBJQU0gcm9sZXMsIGZlZGVyYXRlZCB1c2VycywgSUFNIHVzZXJzIHdpdGggdGVtcG9yYXJ5IHRva2VucyBmcm9tIGBzdHM6R2V0U2Vzc2lvblRva2VuYCwgYW5kIHVzZXJzIG9mIHRoZSBBV1MgTWFuYWdlbWVudCBDb25zb2xlLiBJQU0gdXNlcnMgaW4gdGhlIEFXUyBNYW5hZ2VtZW50IENvbnNvbGUgdW5rbm93aW5nbHkgdXNlIHRlbXBvcmFyeSBjcmVkZW50aWFscy4gVXNlcnMgc2lnbiBpbnRvIHRoZSBjb25zb2xlIHVzaW5nIHRoZWlyIHVzZXIgbmFtZSBhbmQgcGFzc3dvcmQsIHdoaWNoIGFyZSBsb25nLXRlcm0gY3JlZGVudGlhbHMuIEhvd2V2ZXIsIGluIHRoZSBiYWNrZ3JvdW5kLCB0aGUgY29uc29sZSBnZW5lcmF0ZXMgdGVtcG9yYXJ5IGNyZWRlbnRpYWxzIG9uIGJlaGFsZiBvZiB0aGUgdXNlci4gVG8gbGVhcm4gd2hpY2ggc2VydmljZXMgc3VwcG9ydCB1c2luZyB0ZW1wb3JhcnkgY3JlZGVudGlhbHMsIHNlZSBbQVdTIFNlcnZpY2VzIFRoYXQgV29yayB3aXRoIElBTV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9hd3Mtc2VydmljZXMtdGhhdC13b3JrLXdpdGgtaWFtLmh0bWwpLlxuICAgKlxuICAgKiBUaGUgYGF3czpNdWx0aUZhY3RvckF1dGhQcmVzZW50YCBrZXkgaXMgbm90IHByZXNlbnQgd2hlbiBhbiBBUEkgb3IgQ0xJIGNvbW1hbmQgaXMgY2FsbGVkIHdpdGggbG9uZy10ZXJtIGNyZWRlbnRpYWxzLCBzdWNoIGFzIHVzZXIgYWNjZXNzIGtleSBwYWlycy4gVGhlcmVmb3JlIHdlIHJlY29tbWVuZCB0aGF0IHdoZW4geW91IGNoZWNrIGZvciB0aGlzIGtleSB0aGF0IHlvdSB1c2UgdGhlIFsuLi5JZkV4aXN0c10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19JZkV4aXN0cykgdmVyc2lvbnMgb2YgdGhlIGNvbmRpdGlvbiBvcGVyYXRvcnMuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1tdWx0aWZhY3RvcmF1dGhwcmVzZW50XG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBXZWF0aGVyIHRoZSBNRkEgc2hvdWxkIGJlIHByZXNlbnQgb3IgYWJzZW50LiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NNdWx0aUZhY3RvckF1dGhQcmVzZW50KHZhbHVlPzogYm9vbGVhbikge1xuICAgIHJldHVybiB0aGlzLmlmKFxuICAgICAgYGF3czpNdWx0aUZhY3RvckF1dGhQcmVzZW50YCxcbiAgICAgIHR5cGVvZiB2YWx1ZSAhPT0gJ3VuZGVmaW5lZCcgPyB2YWx1ZSA6IHRydWUsXG4gICAgICBuZXcgT3BlcmF0b3IoKS5ib29sKClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIGFjY291bnQgdG8gd2hpY2ggdGhlIHJlcXVlc3RpbmcgcHJpbmNpcGFsIGJlbG9uZ3Mgd2l0aCB0aGUgYWNjb3VudCBpZGVudGlmaWVyIHRoYXQgeW91IHNwZWNpZnkuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGFsd2F5cyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcHJpbmNpcGFsYWNjb3VudFxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgQWNjb3VudCBpZGVudGlmaWVyKHMpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ByaW5jaXBhbEFjY291bnQoXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoJ2F3czpQcmluY2lwYWxBY2NvdW50JywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSBbQW1hem9uIFJlc291cmNlIE5hbWVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfaWRlbnRpZmllcnMuaHRtbCNpZGVudGlmaWVycy1hcm5zKSAoQVJOKSBvZiB0aGUgcHJpbmNpcGFsIHRoYXQgbWFkZSB0aGUgcmVxdWVzdCB3aXRoIHRoZSBBUk4gdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogRm9yIElBTSByb2xlcywgdGhlIHJlcXVlc3QgY29udGV4dCByZXR1cm5zIHRoZSBBUk4gb2YgdGhlIHJvbGUsIG5vdCB0aGUgQVJOIG9mIHRoZSB1c2VyIHRoYXQgYXNzdW1lZCB0aGUgcm9sZS4gVG8gbGVhcm4gd2hpY2ggdHlwZXMgb2YgcHJpbmNpcGFscyB5b3UgY2FuIHNwZWNpZnkgaW4gdGhpcyBjb25kaXRpb24ga2V5LCBzZWUgW1NwZWNpZnlpbmcgYSBQcmluY2lwYWxdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfcHJpbmNpcGFsLmh0bWwjUHJpbmNpcGFsX3NwZWNpZnlpbmcpLlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBhbHdheXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dC5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXByaW5jaXBhbGFyblxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgUHJpbmNpcGxlIEFSTihzKVxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbQVJOIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pIGFuZCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NQcmluY2lwYWxBcm4oXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoXG4gICAgICAnYXdzOlByaW5jaXBhbEFybicsXG4gICAgICB2YWx1ZSxcbiAgICAgIG9wZXJhdG9yIHx8IG5ldyBPcGVyYXRvcigpLmFybkxpa2UoKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgd2hldGhlciB0aGUgY2FsbCB0byB5b3VyIHJlc291cmNlIGlzIGJlaW5nIG1hZGUgZGlyZWN0bHkgYnkgYW4gQVdTIFtzZXJ2aWNlIHByaW5jaXBhbF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19wcmluY2lwYWwuaHRtbCNwcmluY2lwYWwtc2VydmljZXMpLiBGb3IgZXhhbXBsZSwgQVdTIENsb3VkVHJhaWwgdXNlcyB0aGUgc2VydmljZSBwcmluY2lwYWwgYGNsb3VkdHJhaWwuYW1hem9uYXdzLmNvbWAgdG8gd3JpdGUgbG9ncyB0byB5b3VyIEFtYXpvbiBTMyBidWNrZXQuIFRoZSByZXF1ZXN0IGNvbnRleHQga2V5IGlzIHNldCB0byB0cnVlIHdoZW4gYSBzZXJ2aWNlIHVzZXMgYSBzZXJ2aWNlIHByaW5jaXBhbCB0byBwZXJmb3JtIGEgZGlyZWN0IGFjdGlvbiBvbiB5b3VyIHJlc291cmNlcy4gVGhlIGNvbnRleHQga2V5IGlzIHNldCB0byBmYWxzZSBpZiB0aGUgc2VydmljZSB1c2VzIHRoZSBjcmVkZW50aWFscyBvZiBhbiBJQU0gcHJpbmNpcGFsIHRvIG1ha2UgYSByZXF1ZXN0IG9uIHRoZSBwcmluY2lwYWwncyBiZWhhbGYuIEl0IGlzIGFsc28gc2V0IHRvIGZhbHNlIGlmIHRoZSBzZXJ2aWNlIHVzZXMgYSBbc2VydmljZSByb2xlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfdGVybXMtYW5kLWNvbmNlcHRzLmh0bWwjaWFtLXRlcm0tc2VydmljZS1yb2xlKSBvciBbc2VydmljZS1saW5rZWQgcm9sZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Rlcm1zLWFuZC1jb25jZXB0cy5odG1sI2lhbS10ZXJtLXNlcnZpY2UtbGlua2VkLXJvbGUpIHRvIG1ha2UgYSBjYWxsIG9uIHRoZSBwcmluY2lwYWwncyBiZWhhbGYuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIHByZXNlbnQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBmb3IgYWxsIHNpZ25lZCBBUEkgcmVxdWVzdHMgdGhhdCB1c2UgQVdTIGNyZWRlbnRpYWxzLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcHJpbmNpcGFsaXNhd3NzZXJ2aWNlXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBXZWF0aGVyIHRoZSBjYWxsIHRvIHlvdXIgcmVzb3VyY2UgaXMgYmVpbmcgbWFkZSBkaXJlY3RseSBieSBhbiBBV1Mgc2VydmljZSBwcmluY2lwYWwuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ByaW5jaXBhbElzQVdTU2VydmljZSh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihcbiAgICAgIGBhd3M6UHJpbmNpcGFsSXNBV1NTZXJ2aWNlYCxcbiAgICAgIHR5cGVvZiB2YWx1ZSAhPT0gJ3VuZGVmaW5lZCcgPyB2YWx1ZSA6IHRydWUsXG4gICAgICBuZXcgT3BlcmF0b3IoKS5ib29sKClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIGlkZW50aWZpZXIgb2YgdGhlIG9yZ2FuaXphdGlvbiBpbiBBV1MgT3JnYW5pemF0aW9ucyB0byB3aGljaCB0aGUgcmVxdWVzdGluZyBwcmluY2lwYWwgYmVsb25ncyB3aXRoIHRoZSBpZGVudGlmaWVyIHlvdSBzcGVjaWZ5LlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IG9ubHkgaWYgdGhlIHByaW5jaXBhbCBpcyBhIG1lbWJlciBvZiBhbiBvcmdhbml6YXRpb24uXG4gICAqXG4gICAqIFRoaXMgZ2xvYmFsIGtleSBwcm92aWRlcyBhbiBhbHRlcm5hdGl2ZSB0byBsaXN0aW5nIGFsbCB0aGUgYWNjb3VudCBJRHMgZm9yIGFsbCBBV1MgYWNjb3VudHMgaW4gYW4gb3JnYW5pemF0aW9uLiBZb3UgY2FuIHVzZSB0aGlzIGNvbmRpdGlvbiBrZXkgdG8gc2ltcGxpZnkgc3BlY2lmeWluZyB0aGUgYFByaW5jaXBhbGAgZWxlbWVudCBpbiBhIHJlc291cmNlLWJhc2VkIHBvbGljeS4gWW91IGNhbiBzcGVjaWZ5IHRoZSBvcmdhbml6YXRpb24gSUQgaW4gdGhlIGNvbmRpdGlvbiBlbGVtZW50LiBXaGVuIHlvdSBhZGQgYW5kIHJlbW92ZSBhY2NvdW50cywgcG9saWNpZXMgdGhhdCBpbmNsdWRlIHRoZSBgYXdzOlByaW5jaXBhbE9yZ0lEYCBrZXkgYXV0b21hdGljYWxseSBpbmNsdWRlIHRoZSBjb3JyZWN0IGFjY291bnRzIGFuZCBkb24ndCByZXF1aXJlIG1hbnVhbCB1cGRhdGluZy5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXByaW5jaXBhbG9yZ2lkXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBPcmdhbml6YXRpb24gSUQocykgaW4gZm9ybWF0IGBvLXh4eHh4eHh4eHh4YFxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NQcmluY2lwYWxPcmdJRChcbiAgICB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sXG4gICAgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZ1xuICApIHtcbiAgICByZXR1cm4gdGhpcy5pZignYXdzOlByaW5jaXBhbE9yZ0lEJywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSBBV1MgT3JnYW5pemF0aW9ucyBwYXRoIGZvciB0aGUgcHJpbmNpcGFsIHdobyBpcyBtYWtpbmcgdGhlIHJlcXVlc3QgdG8gdGhlIHBhdGggeW91IHByb3ZpZGUuIFRoYXQgcHJpbmNpcGFsIGNhbiBiZSBhbiBJQU0gdXNlciwgSUFNIHJvbGUsIGZlZGVyYXRlZCB1c2VyLCBvciBBV1MgYWNjb3VudCByb290IHVzZXIuXG4gICAqXG4gICAqIFRoaXMgY29uZGl0aW9uIGVuc3VyZXMgdGhhdCB0aGUgcmVxdWVzdGVyIGlzIGFuIGFjY291bnQgbWVtYmVyIHdpdGhpbiB0aGUgc3BlY2lmaWVkIG9yZ2FuaXphdGlvbiByb290IG9yIG9yZ2FuaXphdGlvbmFsIHVuaXRzIChPVXMpIGluIEFXUyBPcmdhbml6YXRpb25zLiBBbiBBV1MgT3JnYW5pemF0aW9ucyBwYXRoIGlzIGEgdGV4dCByZXByZXNlbnRhdGlvbiBvZiB0aGUgc3RydWN0dXJlIG9mIGFuIE9yZ2FuaXphdGlvbnMgZW50aXR5LiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhYm91dCB1c2luZyBhbmQgdW5kZXJzdGFuZGluZyBwYXRocywgc2VlIFVuZGVyc3RhbmQgdGhlIFtBV1MgT3JnYW5pemF0aW9ucyBFbnRpdHkgUGF0aF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2FjY2Vzc19wb2xpY2llc19hY2Nlc3MtYWR2aXNvci12aWV3LWRhdGEtb3Jncy5odG1sI2FjY2Vzc19wb2xpY2llc19hY2Nlc3MtYWR2aXNvci12aWV3aW5nLW9yZ3MtZW50aXR5LXBhdGgpLlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IG9ubHkgaWYgdGhlIHByaW5jaXBhbCBpcyBhIG1lbWJlciBvZiBhbiBvcmdhbml6YXRpb24uXG4gICAqXG4gICAqICoqTm90ZToqKiBPcmdhbml6YXRpb24gSURzIGFyZSBnbG9iYWxseSB1bmlxdWUgYnV0IE9VIElEcyBhbmQgcm9vdCBJRHMgYXJlIHVuaXF1ZSBvbmx5IHdpdGhpbiBhbiBvcmdhbml6YXRpb24uIFRoaXMgbWVhbnMgdGhhdCBubyB0d28gb3JnYW5pemF0aW9ucyBzaGFyZSB0aGUgc2FtZSBvcmdhbml6YXRpb24gSUQuIEhvd2V2ZXIsIGFub3RoZXIgb3JnYW5pemF0aW9uIG1pZ2h0IGhhdmUgYW4gT1Ugb3Igcm9vdCB3aXRoIHRoZSBzYW1lIElEIGFzIHlvdXJzLiBXZSByZWNvbW1lbmQgdGhhdCB5b3UgYWx3YXlzIGluY2x1ZGUgdGhlIG9yZ2FuaXphdGlvbiBJRCB3aGVuIHlvdSBzcGVjaWZ5IGFuIE9VIG9yIHJvb3QuXG4gICAqXG4gICAqIGBhd3M6UHJpbmNpcGFsT3JnUGF0aHNgIGlzIGEgbXVsdGl2YWx1ZWQgY29uZGl0aW9uIGtleS4gTXVsdGl2YWx1ZWQga2V5cyBpbmNsdWRlIG9uZSBvciBtb3JlIHZhbHVlcyBpbiBhIGxpc3QgZm9ybWF0LiBUaGUgcmVzdWx0IGlzIGEgbG9naWNhbCBgT1JgLiBXaGVuIHlvdSB1c2UgbXVsdGlwbGUgdmFsdWVzIHdpdGggdGhlIGBGb3JBbnlWYWx1ZTpgIGNvbmRpdGlvbiBvcGVyYXRvciwgdGhlIHByaW5jaXBhbCdzIHBhdGggbXVzdCBtYXRjaCBvbmUgb2YgdGhlIHBhdGhzIHByb3ZpZGVkLiBGb3IgcG9saWNpZXMgdGhhdCBpbmNsdWRlIG11bHRpcGxlIHZhbHVlcyBmb3IgYSBzaW5nbGUga2V5LCB5b3UgbXVzdCBlbmNsb3NlIHRoZSBjb25kaXRpb25zIHdpdGhpbiBicmFja2V0cyBsaWtlIGFuIGFycmF5IChgXCJLZXlcIjpbXCJWYWx1ZTFcIiwgXCJWYWx1ZTJcIl1gKS4gWW91IHNob3VsZCBhbHNvIGluY2x1ZGUgdGhlc2UgYnJhY2tldHMgd2hlbiB0aGVyZSBpcyBhIHNpbmdsZSB2YWx1ZS4gRm9yIG1vcmUgaW5mb3JtYXRpb24gYWJvdXQgbXVsdGl2YWx1ZWQgY29uZGl0aW9uIGtleXMsIHNlZSBbQ3JlYXRpbmcgYSBDb25kaXRpb24gd2l0aCBNdWx0aXBsZSBLZXlzIG9yIFZhbHVlc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19tdWx0aS12YWx1ZS1jb25kaXRpb25zLmh0bWwpLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcHJpbmNpcGFsb3JncGF0aHNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIE9yZ2FuaXphdGlvbiBwYXRoKHMpIGluIHRoZSBmb3JtYXQgb2YgYG8teHh4eHh4eHh4eHgvci14eHh4eHh4eHh4L291LXh4eHgteHh4eHh4eHgvb3UteHh4eC14eHh4eHh4eC9gXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUHJpbmNpcGFsT3JnUGF0aHMoXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoJ2F3czpQcmluY2lwYWxPcmdQYXRocycsIHZhbHVlLCBvcGVyYXRvcik7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgW3NlcnZpY2UgcHJpbmNpcGFsXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3ByaW5jaXBhbC5odG1sI3ByaW5jaXBhbC1zZXJ2aWNlcykgbmFtZSBpbiB0aGUgcG9saWN5IHdpdGggdGhlIHNlcnZpY2UgcHJpbmNpcGFsIHRoYXQgaXMgbWFraW5nIHJlcXVlc3RzIHRvIHlvdXIgcmVzb3VyY2VzLiBZb3UgY2FuIHVzZSB0aGlzIGtleSB0byBjaGVjayB3aGV0aGVyIHRoaXMgY2FsbCBpcyBtYWRlIGJ5IGEgc3BlY2lmaWMgc2VydmljZSBwcmluY2lwYWwuIFdoZW4gYSBzZXJ2aWNlIHByaW5jaXBhbCBtYWtlcyBhIGRpcmVjdCByZXF1ZXN0IHRvIHlvdXIgcmVzb3VyY2UsIHRoZSBgYXdzOlByaW5jaXBhbFNlcnZpY2VOYW1lYCBrZXkgY29udGFpbnMgdGhlIG5hbWUgb2YgdGhlIHNlcnZpY2UgcHJpbmNpcGFsLiBGb3IgZXhhbXBsZSwgdGhlIEFXUyBDbG91ZFRyYWlsIHNlcnZpY2UgcHJpbmNpcGFsIG5hbWUgaXMgYGNsb3VkdHJhaWwuYW1hem9uYXdzLmNvbWAuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIHByZXNlbnQgaW4gdGhlIHJlcXVlc3Qgd2hlbiB0aGUgY2FsbCBpcyBtYWRlIGJ5IGFuIEFXUyBzZXJ2aWNlIHByaW5jaXBhbC4gVGhpcyBrZXkgaXMgbm90IHByZXNlbnQgaW4gYW55IG90aGVyIHNpdHVhdGlvbiwgaW5jbHVkaW5nIHRoZSBmb2xsb3dpbmc6XG4gICAqXG4gICAqIC0gSWYgdGhlIHNlcnZpY2UgdXNlcyBhIFtzZXJ2aWNlIHJvbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1zZXJ2aWNlLXJvbGUpIG9yIFtzZXJ2aWNlLWxpbmtlZCByb2xlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfdGVybXMtYW5kLWNvbmNlcHRzLmh0bWwjaWFtLXRlcm0tc2VydmljZS1saW5rZWQtcm9sZSkgdG8gbWFrZSBhIGNhbGwgb24gdGhlIHByaW5jaXBhbCdzIGJlaGFsZi5cbiAgICogLSBJZiB0aGUgc2VydmljZSB1c2VzIHRoZSBjcmVkZW50aWFscyBvZiBhbiBJQU0gcHJpbmNpcGFsIHRvIG1ha2UgYSByZXF1ZXN0IG9uIHRoZSBwcmluY2lwYWwncyBiZWhhbGYuXG4gICAqIC0gSWYgdGhlIGNhbGwgaXMgbWFkZSBkaXJlY3RseSBieSBhbiBJQU0gcHJpbmNpcGFsLlxuICAgKlxuICAgKiBZb3UgY2FuIHVzZSB0aGlzIGNvbmRpdGlvbiBrZXkgdG8gbGltaXQgYWNjZXNzIHRvIHlvdXIgdHJ1c3RlZCBpZGVudGl0aWVzIGFuZCBleHBlY3RlZCBuZXR3b3JrIGxvY2F0aW9ucywgd2hpbGUgc2FmZWx5IGdyYW50aW5nIGFjY2VzcyB0byBhbiBBV1Mgc2VydmljZS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXByaW5jaXBhbHNlcnZpY2VuYW1lXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBBV1Mgc2VydmljZSBuYW1lLCBlLmcuIGBjbG91ZHRyYWlsLmFtYXpvbmF3cy5jb21gXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ByaW5jaXBhbFNlcnZpY2VOYW1lKFxuICAgIHZhbHVlOiBzdHJpbmcsXG4gICAgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZ1xuICApIHtcbiAgICByZXR1cm4gdGhpcy5pZignYXdzOlByaW5jaXBhbFNlcnZpY2VOYW1lJywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGlzIGtleSBwcm92aWRlcyBhIGxpc3Qgb2YgYWxsIFtzZXJ2aWNlIHByaW5jaXBhbF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19wcmluY2lwYWwuaHRtbCNwcmluY2lwYWwtc2VydmljZXMpIG5hbWVzIHRoYXQgYmVsb25nIHRvIHRoZSBzZXJ2aWNlLiBUaGlzIGlzIGFuIGFkdmFuY2VkIGNvbmRpdGlvbiBrZXkuIFlvdSBjYW4gdXNlIGl0IHRvIHJlc3RyaWN0IHRoZSBzZXJ2aWNlIGZyb20gYWNjZXNzaW5nIHlvdXIgcmVzb3VyY2UgZnJvbSBhIHNwZWNpZmljIFJlZ2lvbiBvbmx5LiBTb21lIHNlcnZpY2VzIG1heSBjcmVhdGUgUmVnaW9uYWwgc2VydmljZSBwcmluY2lwYWxzIHRvIGluZGljYXRlIGEgcGFydGljdWxhciBpbnN0YW5jZSBvZiB0aGUgc2VydmljZSB3aXRoaW4gYSBzcGVjaWZpYyBSZWdpb24uIFlvdSBjYW4gbGltaXQgYWNjZXNzIHRvIGEgcmVzb3VyY2UgdG8gYSBwYXJ0aWN1bGFyIGluc3RhbmNlIG9mIHRoZSBzZXJ2aWNlLiBXaGVuIGEgc2VydmljZSBwcmluY2lwYWwgbWFrZXMgYSBkaXJlY3QgcmVxdWVzdCB0byB5b3VyIHJlc291cmNlLCB0aGUgYGF3czpQcmluY2lwYWxTZXJ2aWNlTmFtZXNMaXN0YCBjb250YWlucyBhbiB1bm9yZGVyZWQgbGlzdCBvZiBhbGwgc2VydmljZSBwcmluY2lwYWwgbmFtZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBSZWdpb25hbCBpbnN0YW5jZSBvZiB0aGUgc2VydmljZS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgcHJlc2VudCBpbiB0aGUgcmVxdWVzdCB3aGVuIHRoZSBjYWxsIGlzIG1hZGUgYnkgYW4gQVdTIHNlcnZpY2UgcHJpbmNpcGFsLiBUaGlzIGtleSBpcyBub3QgcHJlc2VudCBpbiBhbnkgb3RoZXIgc2l0dWF0aW9uLCBpbmNsdWRpbmcgdGhlIGZvbGxvd2luZzpcbiAgICpcbiAgICogLSBJZiB0aGUgc2VydmljZSB1c2VzIGEgW3NlcnZpY2Ugcm9sZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Rlcm1zLWFuZC1jb25jZXB0cy5odG1sI2lhbS10ZXJtLXNlcnZpY2Utcm9sZSkgb3IgW3NlcnZpY2UtbGlua2VkIHJvbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1zZXJ2aWNlLWxpbmtlZC1yb2xlKSB0byBtYWtlIGEgY2FsbCBvbiB0aGUgcHJpbmNpcGFsJ3MgYmVoYWxmLlxuICAgKiAtIElmIHRoZSBzZXJ2aWNlIHVzZXMgdGhlIGNyZWRlbnRpYWxzIG9mIGFuIElBTSBwcmluY2lwYWwgdG8gbWFrZSBhIHJlcXVlc3Qgb24gdGhlIHByaW5jaXBhbCdzIGJlaGFsZi5cbiAgICogLSBJZiB0aGUgY2FsbCBpcyBtYWRlIGRpcmVjdGx5IGJ5IGFuIElBTSBwcmluY2lwYWwuXG4gICAqXG4gICAqIGBhd3M6UHJpbmNpcGFsU2VydmljZU5hbWVzTGlzdGAgaXMgYSBtdWx0aXZhbHVlZCBjb25kaXRpb24ga2V5LiBNdWx0aXZhbHVlZCBrZXlzIGluY2x1ZGUgb25lIG9yIG1vcmUgdmFsdWVzIGluIGEgbGlzdCBmb3JtYXQuIFRoZSByZXN1bHQgaXMgYSBsb2dpY2FsIGBPUmAuIFlvdSBtdXN0IHVzZSB0aGUgYEZvckFueVZhbHVlYCBvciBgRm9yQWxsVmFsdWVzYCBzZXQgb3BlcmF0b3JzIHdpdGggdGhlIGBTdHJpbmdMaWtlYCBbY29uZGl0aW9uIG9wZXJhdG9yXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykgd2hlbiB5b3UgdXNlIHRoaXMga2V5LiBGb3IgcG9saWNpZXMgdGhhdCBpbmNsdWRlIG11bHRpcGxlIHZhbHVlcyBmb3IgYSBzaW5nbGUga2V5LCB5b3UgbXVzdCBlbmNsb3NlIHRoZSBjb25kaXRpb25zIHdpdGhpbiBicmFja2V0cyBsaWtlIGFuIGFycmF5LCBzdWNoIGFzIGAoXCJLZXlcIjpbXCJWYWx1ZTFcIiwgXCJWYWx1ZTJcIl0pYC4gWW91IHNob3VsZCBhbHNvIGluY2x1ZGUgdGhlc2UgYnJhY2tldHMgd2hlbiB0aGVyZSBpcyBhIHNpbmdsZSB2YWx1ZS4gRm9yIG1vcmUgaW5mb3JtYXRpb24gYWJvdXQgbXVsdGl2YWx1ZWQgY29uZGl0aW9uIGtleXMsIHNlZSBbVXNpbmcgbXVsdGlwbGUga2V5cyBhbmQgdmFsdWVzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX211bHRpLXZhbHVlLWNvbmRpdGlvbnMuaHRtbCNyZWZlcmVuY2VfcG9saWNpZXNfbXVsdGkta2V5LW9yLXZhbHVlLWNvbmRpdGlvbnMpLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcHJpbmNpcGFsc2VydmljZW5hbWVzbGlzdFxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgQVdTIHNlcnZpY2UgbmFtZSBsaXN0XG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgRm9yQW55VmFsdWU6U3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ByaW5jaXBhbFNlcnZpY2VOYW1lc0xpc3QoXG4gICAgdmFsdWU6IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoXG4gICAgICAnYXdzOlByaW5jaXBhbFNlcnZpY2VOYW1lc0xpc3QnLFxuICAgICAgdmFsdWUsXG4gICAgICBvcGVyYXRvciB8fCBuZXcgT3BlcmF0b3IoKS5zdHJpbmdMaWtlKCkuZm9yQW55VmFsdWUoKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgdGFnIGF0dGFjaGVkIHRvIHRoZSBwcmluY2lwYWwgbWFraW5nIHRoZSByZXF1ZXN0IHdpdGggdGhlIHRhZyB0aGF0IHlvdSBzcGVjaWZ5LiBJZiB0aGUgcHJpbmNpcGFsIGhhcyBtb3JlIHRoYW4gb25lIHRhZyBhdHRhY2hlZCwgdGhlIHJlcXVlc3QgY29udGV4dCBpbmNsdWRlcyBvbmUgYXdzOlByaW5jaXBhbFRhZyBrZXkgZm9yIGVhY2ggYXR0YWNoZWQgdGFnIGtleS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBpZiB0aGUgcHJpbmNpcGFsIGlzIHVzaW5nIGFuIElBTSB1c2VyIHdpdGggYXR0YWNoZWQgdGFncy4gSXQgaXMgaW5jbHVkZWQgZm9yIGEgcHJpbmNpcGFsIHVzaW5nIGFuIElBTSByb2xlIHdpdGggYXR0YWNoZWQgdGFncyBvciBbc2Vzc2lvbiB0YWdzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfc2Vzc2lvbi10YWdzLmh0bWwpLlxuICAgKlxuICAgKiBZb3UgY2FuIGFkZCBjdXN0b20gYXR0cmlidXRlcyB0byBhIHVzZXIgb3Igcm9sZSBpbiB0aGUgZm9ybSBvZiBhIGtleS12YWx1ZSBwYWlyLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhYm91dCBJQU0gdGFncywgc2VlIFtUYWdnaW5nIElBTSBVc2VycyBhbmQgUm9sZXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF90YWdzLmh0bWwpLiBZb3UgY2FuIHVzZSBgYXdzOlByaW5jaXBhbFRhZ2AgdG8gW2NvbnRyb2wgYWNjZXNzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvYWNjZXNzX2lhbS10YWdzLmh0bWwjYWNjZXNzX2lhbS10YWdzX2NvbnRyb2wtcmVzb3VyY2VzKSBmb3IgQVdTIHByaW5jaXBhbHMuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1wcmluY2lwYWx0YWdcbiAgICpcbiAgICogQHBhcmFtIGtleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHRhZyB2YWx1ZShzKSB0byBjaGVjayBhZ2FpbnN0XG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ByaW5jaXBhbFRhZyhcbiAgICBrZXk6IHN0cmluZyxcbiAgICB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sXG4gICAgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZ1xuICApIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlByaW5jaXBhbFRhZy8ke2tleX1gLCB2YWx1ZSwgb3BlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIHR5cGUgb2YgcHJpbmNpcGFsIG1ha2luZyB0aGUgcmVxdWVzdCB3aXRoIHRoZSBwcmluY2lwYWwgdHlwZSB0aGF0IHlvdSBzcGVjaWZ5LiBGb3IgZGV0YWlscyBhYm91dCBob3cgdGhlIGluZm9ybWF0aW9uIGFwcGVhcnMgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBmb3IgZGlmZmVyZW50IHByaW5jaXBhbHMsIHNlZSBbU3BlY2lmeWluZyBhIFByaW5jaXBhbF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19wcmluY2lwYWwuaHRtbCNQcmluY2lwYWxfc3BlY2lmeWluZykuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGFsd2F5cyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcHJpbmNpcGFsdHlwZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHByaW5jaXBhbCB0eXBlKHMpLiBBbnkgb2YgYEFjY291bnRgLCBgVXNlcmAsIGBGZWRlcmF0ZWRVc2VyYCwgYEFzc3VtZWRSb2xlYCwgYEFub255bW91c2BcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdFcXVhbHNgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NQcmluY2lwYWxUeXBlKFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6UHJpbmNpcGFsVHlwZScsIHZhbHVlLCBvcGVyYXRvcik7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB3aG8gcmVmZXJyZWQgdGhlIHJlcXVlc3QgaW4gdGhlIGNsaWVudCBicm93c2VyIHdpdGggdGhlIHJlZmVyZXIgdGhhdCB5b3Ugc3BlY2lmeS4gVGhlIGBhd3M6cmVmZXJlcmAgcmVxdWVzdCBjb250ZXh0IHZhbHVlIGlzIHByb3ZpZGVkIGJ5IHRoZSBjYWxsZXIgaW4gYW4gSFRUUCBoZWFkZXIuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQgb25seSBpZiB0aGUgcmVxdWVzdCB3YXMgaW52b2tlZCB1c2luZyBhIFVSTCBpbiB0aGUgYnJvd3Nlci5cbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIHlvdSBjYW4gY2FsbCBbQW1hem9uIFMzIEFQSSBvcGVyYXRpb25zIGRpcmVjdGx5IHVzaW5nIGEgd2ViIGJyb3dzZXJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2V4YW1wbGUtYnVja2V0LXBvbGljaWVzLmh0bWwjZXhhbXBsZS1idWNrZXQtcG9saWNpZXMtdXNlLWNhc2UtNCkuIFRoaXMgbWVhbnMgdGhhdCB5b3UgY2FuIHZpZXcgUzMgb2JqZWN0cywgc3VjaCBhcyBpbWFnZXMgYW5kIGRvY3VtZW50cywgZGlyZWN0bHkgdGhyb3VnaCBhIHdlYiBicm93c2VyLiBUaGUgYGF3czpyZWZlcmVyYCBjb25kaXRpb24gYWxsb3dzIHlvdSB0byByZXN0cmljdCBhY2Nlc3MgdG8gc3BlY2lmaWMgdmFsdWVzIGluIHRoZSBIVFRQIG9yIEhUVFBTIHJlcXVlc3QgYmFzZWQgb24gdGhlIHZhbHVlIG9mIHRoZSByZWZlcnJlciBoZWFkZXIuXG4gICAqXG4gICAqICoqV2FybmluZzoqKiBUaGlzIGNvbmRpdGlvbiBzaG91bGQgYmUgdXNlZCBjYXJlZnVsbHkuIEl0IGlzIGRhbmdlcm91cyB0byBpbmNsdWRlIGEgcHVibGljbHkga25vd24gcmVmZXJlciBoZWFkZXIgdmFsdWUuIFVuYXV0aG9yaXplZCBwYXJ0aWVzIGNhbiB1c2UgbW9kaWZpZWQgb3IgY3VzdG9tIGJyb3dzZXJzIHRvIHByb3ZpZGUgYW55IGBhd3M6cmVmZXJlcmAgdmFsdWUgdGhhdCB0aGV5IGNob29zZS4gQXMgYSByZXN1bHQsIGBhd3M6cmVmZXJlcmAgc2hvdWxkIG5vdCBiZSB1c2VkIHRvIHByZXZlbnQgdW5hdXRob3JpemVkIHBhcnRpZXMgZnJvbSBtYWtpbmcgZGlyZWN0IEFXUyByZXF1ZXN0cy4gSXQgaXMgb2ZmZXJlZCBvbmx5IHRvIGFsbG93IGN1c3RvbWVycyB0byBwcm90ZWN0IHRoZWlyIGRpZ2l0YWwgY29udGVudCwgc3VjaCBhcyBjb250ZW50IHN0b3JlZCBpbiBBbWF6b24gUzMsIGZyb20gYmVpbmcgcmVmZXJlbmNlZCBvbiB1bmF1dGhvcml6ZWQgdGhpcmQtcGFydHkgc2l0ZXMuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZWZlcmVyXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgcmVmZXJlciB1cmwocylcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVmZXJlcih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZignYXdzOlJlZmVyZXInLCB2YWx1ZSwgb3BlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIEFXUyBSZWdpb24gdGhhdCB3YXMgY2FsbGVkIGluIHRoZSByZXF1ZXN0IHdpdGggdGhlIFJlZ2lvbiB0aGF0IHlvdSBzcGVjaWZ5LiBZb3UgY2FuIHVzZSB0aGlzIGdsb2JhbCBjb25kaXRpb24ga2V5IHRvIGNvbnRyb2wgd2hpY2ggUmVnaW9ucyBjYW4gYmUgcmVxdWVzdGVkLiBUbyB2aWV3IHRoZSBBV1MgUmVnaW9ucyBmb3IgZWFjaCBzZXJ2aWNlLCBzZWUgW1NlcnZpY2UgZW5kcG9pbnRzIGFuZCBxdW90YXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9nZW5lcmFsL2xhdGVzdC9nci9hd3Mtc2VydmljZS1pbmZvcm1hdGlvbi5odG1sKSBpbiB0aGUgQW1hem9uIFdlYiBTZXJ2aWNlcyBHZW5lcmFsIFJlZmVyZW5jZS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgYWx3YXlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQuXG4gICAqXG4gICAqIFNvbWUgZ2xvYmFsIHNlcnZpY2VzLCBzdWNoIGFzIElBTSwgaGF2ZSBhIHNpbmdsZSBlbmRwb2ludC4gQmVjYXVzZSB0aGlzIGVuZHBvaW50IGlzIHBoeXNpY2FsbHkgbG9jYXRlZCBpbiB0aGUgVVMgRWFzdCAoTi4gVmlyZ2luaWEpIFJlZ2lvbiwgSUFNIGNhbGxzIGFyZSBhbHdheXMgbWFkZSB0byB0aGUgdXMtZWFzdC0xIFJlZ2lvbi4gRm9yIGV4YW1wbGUsIGlmIHlvdSBjcmVhdGUgYSBwb2xpY3kgdGhhdCBkZW5pZXMgYWNjZXNzIHRvIGFsbCBzZXJ2aWNlcyBpZiB0aGUgcmVxdWVzdGVkIFJlZ2lvbiBpcyBub3QgdXMtd2VzdC0yLCB0aGVuIElBTSBjYWxscyBhbHdheXMgZmFpbC4gVG8gdmlldyBhbiBleGFtcGxlIG9mIGhvdyB0byB3b3JrIGFyb3VuZCB0aGlzLCBzZWUgW05vdEFjdGlvbiB3aXRoIERlbnldKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfbm90YWN0aW9uLmh0bWwpLlxuICAgKlxuICAgKiAqKk5vdGU6KiogVGhlIGBhd3M6UmVxdWVzdGVkUmVnaW9uYCBjb25kaXRpb24ga2V5IGFsbG93cyB5b3UgdG8gY29udHJvbCB3aGljaCBlbmRwb2ludCBvZiBhIHNlcnZpY2UgaXMgaW52b2tlZCBidXQgZG9lcyBub3QgY29udHJvbCB0aGUgaW1wYWN0IG9mIHRoZSBvcGVyYXRpb24uIFNvbWUgc2VydmljZXMgaGF2ZSBjcm9zcy1SZWdpb24gaW1wYWN0cy4gRm9yIGV4YW1wbGUsIEFtYXpvbiBTMyBoYXMgQVBJIG9wZXJhdGlvbnMgdGhhdCBjb250cm9sIGNyb3NzLVJlZ2lvbiByZXBsaWNhdGlvbi4gWW91IGNhbiBpbnZva2UgYHMzOlB1dEJ1Y2tldFJlcGxpY2F0aW9uYCBpbiBvbmUgUmVnaW9uICh3aGljaCBpcyBhZmZlY3RlZCBieSB0aGUgYGF3czpSZXF1ZXN0ZWRSZWdpb25gIGNvbmRpdGlvbiBrZXkpLCBidXQgb3RoZXIgUmVnaW9ucyBhcmUgYWZmZWN0ZWQgYmFzZWQgb24gdGhlIHJlcGxpY2F0aW9ucyBjb25maWd1cmF0aW9uIHNldHRpbmdzLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdGVkcmVnaW9uXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgcmVnaW9uKHMpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdGVkUmVnaW9uKFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6UmVxdWVzdGVkUmVnaW9uJywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSB0YWcga2V5LXZhbHVlIHBhaXIgdGhhdCB3YXMgcGFzc2VkIGluIHRoZSByZXF1ZXN0IHdpdGggdGhlIHRhZyBwYWlyIHRoYXQgeW91IHNwZWNpZnkuIEZvciBleGFtcGxlLCB5b3UgY291bGQgY2hlY2sgd2hldGhlciB0aGUgcmVxdWVzdCBpbmNsdWRlcyB0aGUgdGFnIGtleSBgRGVwdGAgYW5kIHRoYXQgaXQgaGFzIHRoZSB2YWx1ZSBgQWNjb3VudGluZ2AuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgW0NvbnRyb2xsaW5nIEFjY2VzcyBEdXJpbmcgQVdTIFJlcXVlc3RzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvYWNjZXNzX3RhZ3MuaHRtbCNhY2Nlc3NfdGFnc19jb250cm9sLXJlcXVlc3RzKS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCB3aGVuIHRhZ3MgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdC4gV2hlbiBtdWx0aXBsZSB0YWdzIGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3QsIHRoZXJlIGlzIG9uZSBjb250ZXh0IGtleSBmb3IgZWFjaCB0YWcga2V5LXZhbHVlIHBhaXIuXG4gICAqXG4gICAqIEJlY2F1c2UgeW91IGNhbiBpbmNsdWRlIG11bHRpcGxlIHRhZyBrZXktdmFsdWUgcGFpcnMgaW4gYSByZXF1ZXN0LCB0aGUgcmVxdWVzdCBjb250ZW50IGNvdWxkIGJlIGEgbXVsdGl2YWx1ZWQgcmVxdWVzdC4gSW4gdGhpcyBjYXNlLCB5b3Ugc2hvdWxkIGNvbnNpZGVyIHVzaW5nIHRoZSBgRm9yQWxsVmFsdWVzYCBvciBgRm9yQW55VmFsdWVgIHNldCBvcGVyYXRvcnMuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgW1VzaW5nIE11bHRpcGxlIEtleXMgYW5kIFZhbHVlc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19tdWx0aS12YWx1ZS1jb25kaXRpb25zLmh0bWwjcmVmZXJlbmNlX3BvbGljaWVzX211bHRpLWtleS1vci12YWx1ZS1jb25kaXRpb25zKS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQHBhcmFtIGtleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHRhZyB2YWx1ZShzKSB0byBjaGVjayBhZ2FpbnN0XG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcoXG4gICAga2V5OiBzdHJpbmcsXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7a2V5fWAsIHZhbHVlLCBvcGVyYXRvcik7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgdGFnIGtleS12YWx1ZSBwYWlyIHRoYXQgeW91IHNwZWNpZnkgd2l0aCB0aGUga2V5LXZhbHVlIHBhaXIgdGhhdCBpcyBhdHRhY2hlZCB0byB0aGUgcmVzb3VyY2UuIEZvciBleGFtcGxlLCB5b3UgY291bGQgcmVxdWlyZSB0aGF0IGFjY2VzcyB0byBhIHJlc291cmNlIGlzIGFsbG93ZWQgb25seSBpZiB0aGUgcmVzb3VyY2UgaGFzIHRoZSBhdHRhY2hlZCB0YWcga2V5IGBEZXB0YCB3aXRoIHRoZSB2YWx1ZSBgTWFya2V0aW5nYC4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBbQ29udHJvbGxpbmcgQWNjZXNzIHRvIEFXUyBSZXNvdXJjZXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9hY2Nlc3NfdGFncy5odG1sI2FjY2Vzc190YWdzX2NvbnRyb2wtcmVzb3VyY2VzKS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCB3aGVuIHRoZSByZXF1ZXN0ZWQgcmVzb3VyY2UgYWxyZWFkeSBoYXMgYXR0YWNoZWQgdGFncy4gVGhpcyBrZXkgaXMgcmV0dXJuZWQgb25seSBmb3IgcmVzb3VyY2VzIHRoYXQgW3N1cHBvcnQgYXV0aG9yaXphdGlvbiBiYXNlZCBvbiB0YWdzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX2F3cy1zZXJ2aWNlcy10aGF0LXdvcmstd2l0aC1pYW0uaHRtbCkuIFRoZXJlIGlzIG9uZSBjb250ZXh0IGtleSBmb3IgZWFjaCB0YWcga2V5LXZhbHVlIHBhaXIuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBAcGFyYW0ga2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdGFnIHZhbHVlKHMpIHRvIGNoZWNrIGFnYWluc3RcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcoXG4gICAga2V5OiBzdHJpbmcsXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8ke2tleX1gLCB2YWx1ZSwgb3BlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIHdoZXRoZXIgdGhlIHJlcXVlc3Qgd2FzIHNlbnQgdXNpbmcgU1NMLiBUaGUgcmVxdWVzdCBjb250ZXh0IHJldHVybnMgYHRydWVgIG9yIGBmYWxzZWAuIEluIGEgcG9saWN5LCB5b3UgY2FuIGFsbG93IHNwZWNpZmljIGFjdGlvbnMgb25seSBpZiB0aGUgcmVxdWVzdCBpcyBzZW50IHVzaW5nIFNTTC5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgYWx3YXlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1zZWN1cmV0cmFuc3BvcnRcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFdlYXRoZXIgcmVxdWVzdCB3YXMgc2VudCB1c2luZyBTU0wuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1NlY3VyZVRyYW5zcG9ydCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihcbiAgICAgIGBhd3M6U2VjdXJlVHJhbnNwb3J0YCxcbiAgICAgIHR5cGVvZiB2YWx1ZSAhPT0gJ3VuZGVmaW5lZCcgPyB2YWx1ZSA6IHRydWUsXG4gICAgICBuZXcgT3BlcmF0b3IoKS5ib29sKClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIGFjY291bnQgSUQgb2YgdGhlIHJlc291cmNlIG1ha2luZyBhIHNlcnZpY2UtdG8tc2VydmljZSByZXF1ZXN0IHdpdGggdGhlIGFjY291bnQgSUQgdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBvbmx5IGlmIGFjY2Vzc2luZyBhIHJlc291cmNlIHRyaWdnZXJzIGFuIEFXUyBzZXJ2aWNlIHRvIGNhbGwgYW5vdGhlciBzZXJ2aWNlIG9uIGJlaGFsZiBvZiB0aGUgcmVzb3VyY2Ugb3duZXIuIFRoZSBjYWxsaW5nIHNlcnZpY2UgbXVzdCBwYXNzIHRoZSByZXNvdXJjZSBBUk4gb2YgdGhlIHNvdXJjZSB0byB0aGUgY2FsbGVkIHNlcnZpY2UuIFRoaXMgQVJOIGluY2x1ZGVzIHRoZSBzb3VyY2UgYWNjb3VudCBJRC5cbiAgICpcbiAgICogWW91IGNhbiB1c2UgdGhpcyBjb25kaXRpb24ga2V5IHRvIGNoZWNrIHRoYXQgQW1hem9uIFMzIGlzIG5vdCBiZWluZyB1c2VkIGFzIGEgW2NvbmZ1c2VkIGRlcHV0eV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX2NyZWF0ZV9mb3ItdXNlcl9leHRlcm5hbGlkLmh0bWwjY29uZnVzZWQtZGVwdXR5KS4gRm9yIGV4YW1wbGUsIHdoZW4gYW4gQW1hem9uIFMzIGJ1Y2tldCB1cGRhdGUgdHJpZ2dlcnMgYW4gQW1hem9uIFNOUyB0b3BpYyBwb3N0LCB0aGUgQW1hem9uIFMzIHNlcnZpY2UgaW52b2tlcyB0aGUgYHNuczpQdWJsaXNoYCBBUEkgb3BlcmF0aW9uLiBUaGUgYnVja2V0IGlzIGNvbnNpZGVyZWQgdGhlIHNvdXJjZSBvZiB0aGUgU05TIHJlcXVlc3QgYW5kIHRoZSB2YWx1ZSBvZiB0aGUga2V5IGlzIHRoZSBhY2NvdW50IElEIGZyb20gdGhlIGJ1Y2tldCdzIEFSTi5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXNvdXJjZWFjY291bnRcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSBhY2NvdW50IElEKHMpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1NvdXJjZUFjY291bnQoXG4gICAgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLFxuICAgIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmdcbiAgKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoJ2F3czpTb3VyY2VBY2NvdW50JywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSBbQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTildKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfaWRlbnRpZmllcnMuaHRtbCNpZGVudGlmaWVycy1hcm5zKSBvZiB0aGUgcmVzb3VyY2UgbWFraW5nIGEgc2VydmljZS10by1zZXJ2aWNlIHJlcXVlc3Qgd2l0aCB0aGUgQVJOIHRoYXQgeW91IHNwZWNpZnkuXG4gICAqXG4gICAqIFRoaXMga2V5IGRvZXMgbm90IHdvcmsgd2l0aCB0aGUgQVJOIG9mIHRoZSBwcmluY2lwYWwgbWFraW5nIHRoZSByZXF1ZXN0LiBJbnN0ZWFkLCB1c2UgW2F3czpQcmluY2lwYWxBcm5dKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1wcmluY2lwYWxhcm4pLiBUaGUgc291cmNlJ3MgQVJOIGluY2x1ZGVzIHRoZSBhY2NvdW50IElELCBzbyBpdCBpcyBub3QgbmVjZXNzYXJ5IHRvIHVzZSBgYXdzOlNvdXJjZUFjY291bnRgIHdpdGggYGF3czpTb3VyY2VBcm5gLlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IG9ubHkgaWYgYWNjZXNzaW5nIGEgcmVzb3VyY2UgdHJpZ2dlcnMgYW4gQVdTIHNlcnZpY2UgdG8gY2FsbCBhbm90aGVyIHNlcnZpY2Ugb24gYmVoYWxmIG9mIHRoZSByZXNvdXJjZSBvd25lci4gVGhlIGNhbGxpbmcgc2VydmljZSBtdXN0IHBhc3MgdGhlIEFSTiBvZiB0aGUgb3JpZ2luYWwgcmVzb3VyY2UgdG8gdGhlIGNhbGxlZCBzZXJ2aWNlLlxuICAgKlxuICAgKiBZb3UgY2FuIHVzZSB0aGlzIGNvbmRpdGlvbiBrZXkgdG8gY2hlY2sgdGhhdCBBbWF6b24gUzMgaXMgbm90IGJlaW5nIHVzZWQgYXMgYSBbY29uZnVzZWQgZGVwdXR5XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfY3JlYXRlX2Zvci11c2VyX2V4dGVybmFsaWQuaHRtbCNjb25mdXNlZC1kZXB1dHkpLiBGb3IgZXhhbXBsZSwgd2hlbiBhbiBBbWF6b24gUzMgYnVja2V0IHVwZGF0ZSB0cmlnZ2VycyBhbiBBbWF6b24gU05TIHRvcGljIHBvc3QsIHRoZSBBbWF6b24gUzMgc2VydmljZSBpbnZva2VzIHRoZSBgc25zOlB1Ymxpc2hgIEFQSSBvcGVyYXRpb24uIFRoZSBidWNrZXQgaXMgY29uc2lkZXJlZCB0aGUgc291cmNlIG9mIHRoZSBTTlMgcmVxdWVzdCBhbmQgdGhlIHZhbHVlIG9mIHRoZSBrZXkgaXMgdGhlIGJ1Y2tldCdzIEFSTi5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXNvdXJjZWFyblxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHNvdXJjZSBBUk4ocylcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW0FSTiBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfQVJOKSBhbmQgW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBBcm5MaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzU291cmNlQXJuKFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKFxuICAgICAgJ2F3czpTb3VyY2VBcm4nLFxuICAgICAgdmFsdWUsXG4gICAgICBvcGVyYXRvciB8fCBuZXcgT3BlcmF0b3IoKS5hcm5MaWtlKClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIHNvdXJjZSBpZGVudGl0eSB0aGF0IHdhcyBzZXQgYnkgdGhlIHByaW5jaXBhbCB3aXRoIHRoZSBzb3VyY2UgaWRlbnRpdHkgdGhhdCB5b3Ugc3BlY2lmeSBpbiB0aGUgcG9saWN5LlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IGFmdGVyIGEgc291cmNlIGlkZW50aXR5IGhhcyBiZWVuIHNldCB3aGVuIGEgcm9sZSBpcyBhc3N1bWVkIHVzaW5nIGFueSBBV1MgU1RTIGFzc3VtZS1yb2xlIENMSSBjb21tYW5kLCBvciBBV1MgU1RTIGBBc3N1bWVSb2xlYCBBUEkgb3BlcmF0aW9uLlxuICAgKlxuICAgKiBZb3UgY2FuIHVzZSB0aGlzIGtleSBpbiBhIHBvbGljeSB0byBhbGxvdyBhY3Rpb25zIGluIEFXUyBieSBwcmluY2lwYWxzIHRoYXQgaGF2ZSBzZXQgYSBzb3VyY2UgaWRlbnRpdHkgd2hlbiBhc3N1bWluZyBhIHJvbGUuIEFjdGl2aXR5IGZvciB0aGUgcm9sZSdzIHNwZWNpZmllZCBzb3VyY2UgaWRlbnRpdHkgYXBwZWFycyBpbiBbQVdTIENsb3VkVHJhaWxdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9jbG91ZHRyYWlsLWludGVncmF0aW9uLmh0bWwjY2xvdWR0cmFpbC1pbnRlZ3JhdGlvbl9zaWduaW4tdGVtcGNyZWRzKS4gVGhpcyBtYWtlcyBpdCBlYXNpZXIgZm9yIGFkbWluaXN0cmF0b3JzIHRvIGRldGVybWluZSB3aG8gb3Igd2hhdCBwZXJmb3JtZWQgYWN0aW9ucyB3aXRoIGEgcm9sZSBpbiBBV1MuXG4gICAqXG4gICAqIFVubGlrZSBbc3RzOlJvbGVTZXNzaW9uTmFtZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19pYW0tY29uZGl0aW9uLWtleXMuaHRtbCNja19yb2xlc2Vzc2lvbm5hbWUpLCBhZnRlciB0aGUgc291cmNlIGlkZW50aXR5IGlzIHNldCwgdGhlIHZhbHVlIGNhbm5vdCBiZSBjaGFuZ2VkLiBJdCBpcyBwcmVzZW50IGluIHRoZSByZXF1ZXN0IGNvbnRleHQgZm9yIGFsbCBhY3Rpb25zIHRha2VuIGJ5IHRoZSByb2xlLiBUaGUgdmFsdWUgcGVyc2lzdHMgaW50byBzdWJzZXF1ZW50IHJvbGUgc2Vzc2lvbnMgd2hlbiB5b3UgdXNlIHRoZSBzZXNzaW9uIGNyZWRlbnRpYWxzIHRvIGFzc3VtZSBhbm90aGVyIHJvbGUuIEFzc3VtaW5nIG9uZSByb2xlIGZyb20gYW5vdGhlciBpcyBjYWxsZWQgW3JvbGUgY2hhaW5pbmddKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1yb2xlLWNoYWluaW5nKS5cbiAgICpcbiAgICogVGhlIFtzdHM6U291cmNlSWRlbnRpdHldKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfaWFtLWNvbmRpdGlvbi1rZXlzLmh0bWwjY2tfc291cmNlaWRlbnRpdHkpIGtleSBpcyBwcmVzZW50IGluIHRoZSByZXF1ZXN0IHdoZW4gdGhlIHByaW5jaXBhbCBpbml0aWFsbHkgc2V0cyBhIHNvdXJjZSBpZGVudGl0eSB3aGlsZSBhc3N1bWluZyBhIHJvbGUgdXNpbmcgYW55IEFXUyBTVFMgYXNzdW1lLXJvbGUgQ0xJIGNvbW1hbmQsIG9yIEFXUyBTVFMgYEFzc3VtZVJvbGVgIEFQSSBvcGVyYXRpb24uIFRoZSBgYXdzOlNvdXJjZUlkZW50aXR5YCBrZXkgaXMgcHJlc2VudCBpbiB0aGUgcmVxdWVzdCBmb3IgYW55IGFjdGlvbnMgdGhhdCBhcmUgdGFrZW4gd2l0aCBhIHJvbGUgc2Vzc2lvbiB0aGF0IGhhcyBhIHNvdXJjZSBpZGVudGl0eSBzZXQuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1zb3VyY2VpZGVudGl0eVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHNvdXJjZSBpZGVudGl0eVxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NTb3VyY2VJZGVudGl0eShcbiAgICB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sXG4gICAgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZ1xuICApIHtcbiAgICByZXR1cm4gdGhpcy5pZignYXdzOlNvdXJjZUlkZW50aXR5JywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSByZXF1ZXN0ZXIncyBJUCBhZGRyZXNzIHdpdGggdGhlIElQIGFkZHJlc3MgdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCwgZXhjZXB0IHdoZW4gdGhlIHJlcXVlc3RlciB1c2VzIGEgVlBDIGVuZHBvaW50IHRvIG1ha2UgdGhlIHJlcXVlc3QuXG4gICAqXG4gICAqIFRoZSBgYXdzOlNvdXJjZUlwYCBjb25kaXRpb24ga2V5IGNhbiBiZSB1c2VkIGluIGEgcG9saWN5IHRvIGFsbG93IHByaW5jaXBhbHMgdG8gbWFrZSByZXF1ZXN0cyBvbmx5IGZyb20gd2l0aGluIGEgc3BlY2lmaWVkIElQIHJhbmdlLiBIb3dldmVyLCB0aGlzIHBvbGljeSBkZW5pZXMgYWNjZXNzIGlmIGFuIEFXUyBzZXJ2aWNlIG1ha2VzIGNhbGxzIG9uIHRoZSBwcmluY2lwYWwncyBiZWhhbGYuIEluIHRoaXMgY2FzZSwgeW91IGNhbiB1c2UgYGF3czpTb3VyY2VJcGAgd2l0aCB0aGUgW2F3czpWaWFBV1NTZXJ2aWNlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdmlhYXdzc2VydmljZSkga2V5IHRvIGVuc3VyZSB0aGF0IHRoZSBzb3VyY2UgSVAgcmVzdHJpY3Rpb24gYXBwbGllcyBvbmx5IHRvIHJlcXVlc3RzIG1hZGUgZGlyZWN0bHkgYnkgYSBwcmluY2lwYWwuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1zb3VyY2VpcFxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHNvdXJjZSBJUChzKVxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBJUCBbYWRkcmVzcyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfSVBBZGRyZXNzKS4gKipEZWZhdWx0OioqIGBJcEFkZHJlc3NgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NTb3VyY2VJcCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihcbiAgICAgICdhd3M6U291cmNlSXAnLFxuICAgICAgdmFsdWUsXG4gICAgICBvcGVyYXRvciB8fCBuZXcgT3BlcmF0b3IoKS5pcEFkZHJlc3MoKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgd2hldGhlciB0aGUgcmVxdWVzdCBjb21lcyBmcm9tIHRoZSBWUEMgdGhhdCB5b3Ugc3BlY2lmeS4gSW4gYSBwb2xpY3ksIHlvdSBjYW4gdXNlIHRoaXMgY29uZGl0aW9uIHRvIGFsbG93IGFjY2VzcyB0byBvbmx5IGEgc3BlY2lmaWMgVlBDLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFtSZXN0cmljdGluZyBBY2Nlc3MgdG8gYSBTcGVjaWZpYyBWUENdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2V4YW1wbGUtYnVja2V0LXBvbGljaWVzLXZwYy1lbmRwb2ludC5odG1sI2V4YW1wbGUtYnVja2V0LXBvbGljaWVzLXJlc3RyaWN0LWFjY2Vzcy12cGMpIGluIHRoZSAqQW1hem9uIFNpbXBsZSBTdG9yYWdlIFNlcnZpY2UgRGV2ZWxvcGVyIEd1aWRlKi5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBvbmx5IGlmIHRoZSByZXF1ZXN0ZXIgdXNlcyBhIFZQQyBlbmRwb2ludCB0byBtYWtlIHRoZSByZXF1ZXN0LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtc291cmNldnBjXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgVlBTIElEKHMpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmQXdzU291cmNlVnBjKFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6U291cmNlVnBjJywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSBWUEMgZW5kcG9pbnQgaWRlbnRpZmllciBvZiB0aGUgcmVxdWVzdCB3aXRoIHRoZSBlbmRwb2ludCBJRCB0aGF0IHlvdSBzcGVjaWZ5LiBJbiBhIHBvbGljeSwgeW91IGNhbiB1c2UgdGhpcyBjb25kaXRpb24gdG8gcmVzdHJpY3QgYWNjZXNzIHRvIGEgc3BlY2lmaWMgVlBDIGVuZHBvaW50LiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFtSZXN0cmljdGluZyBBY2Nlc3MgdG8gYSBTcGVjaWZpYyBWUEMgRW5kcG9pbnRdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2V4YW1wbGUtYnVja2V0LXBvbGljaWVzLXZwYy1lbmRwb2ludC5odG1sI2V4YW1wbGUtYnVja2V0LXBvbGljaWVzLXJlc3RyaWN0LWFjY2Vzcy12cGMtZW5kcG9pbnQpIGluIHRoZSAqQW1hem9uIFNpbXBsZSBTdG9yYWdlIFNlcnZpY2UgRGV2ZWxvcGVyIEd1aWRlKi5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBvbmx5IGlmIHRoZSByZXF1ZXN0ZXIgdXNlcyBhIFZQQyBlbmRwb2ludCB0byBtYWtlIHRoZSByZXF1ZXN0LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtc291cmNldnBjZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIFZQQyBFbmRwb2ludCBJRChzKVxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NTb3VyY2VWcGNlKFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6U291cmNlVnBjZScsIHZhbHVlLCBvcGVyYXRvcik7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgdGFnIGtleXMgaW4gYSByZXF1ZXN0IHdpdGggdGhlIGtleXMgdGhhdCB5b3Ugc3BlY2lmeS4gQXMgYSBiZXN0IHByYWN0aWNlIHdoZW4geW91IHVzZSBwb2xpY2llcyB0byBjb250cm9sIGFjY2VzcyB1c2luZyB0YWdzLCB1c2UgdGhlIGBhd3M6VGFnS2V5c2AgY29uZGl0aW9uIGtleSB0byBkZWZpbmUgd2hhdCB0YWcga2V5cyBhcmUgYWxsb3dlZC4gRm9yIGV4YW1wbGUgcG9saWNpZXMgYW5kIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBbQ29udHJvbGxpbmcgQWNjZXNzIEJhc2VkIG9uIFRhZyBLZXlzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvYWNjZXNzX3RhZ3MuaHRtbCNhY2Nlc3NfdGFnc19jb250cm9sLXRhZy1rZXlzKS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBvbmx5IGlmIHRoZSBvcGVyYXRpb24gc3VwcG9ydHMgYXR0YWNoaW5nIHRhZ3MgdG8gcmVzb3VyY2VzLlxuICAgKlxuICAgKiBCZWNhdXNlIHlvdSBjYW4gaW5jbHVkZSBtdWx0aXBsZSB0YWcga2V5LXZhbHVlIHBhaXJzIGluIGEgcmVxdWVzdCwgdGhlIHJlcXVlc3QgY29udGVudCBjb3VsZCBiZSBhIFttdWx0aXZhbHVlZF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19tdWx0aS12YWx1ZS1jb25kaXRpb25zLmh0bWwpIHJlcXVlc3QuIEluIHRoaXMgY2FzZSwgeW91IHNob3VsZCBjb25zaWRlciB1c2luZyB0aGUgYEZvckFsbFZhbHVlc2Agb3IgYEZvckFueVZhbHVlYCBzZXQgb3BlcmF0b3JzLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFVzaW5nIE11bHRpcGxlIEtleXMgYW5kIFZhbHVlcy5cbiAgICpcbiAgICogU29tZSBzZXJ2aWNlcyBzdXBwb3J0IHRhZ2dpbmcgd2l0aCByZXNvdXJjZSBvcGVyYXRpb25zLCBzdWNoIGFzIGNyZWF0aW5nLCBtb2RpZnlpbmcsIG9yIGRlbGV0aW5nIGEgcmVzb3VyY2UuIFRvIGFsbG93IHRhZ2dpbmcgYW5kIG9wZXJhdGlvbnMgYXMgYSBzaW5nbGUgY2FsbCwgeW91IG11c3QgY3JlYXRlIGEgcG9saWN5IHRoYXQgaW5jbHVkZXMgYm90aCB0aGUgdGFnZ2luZyBhY3Rpb24gYW5kIHRoZSByZXNvdXJjZS1tb2RpZnlpbmcgYWN0aW9uLiBZb3UgY2FuIHRoZW4gdXNlIHRoZSBgYXdzOlRhZ0tleXNgIGNvbmRpdGlvbiBrZXkgdG8gZW5mb3JjZSB1c2luZyBzcGVjaWZpYyB0YWcga2V5cyBpbiB0aGUgcmVxdWVzdC4gRm9yIGV4YW1wbGUsIHRvIGxpbWl0IHRhZ3Mgd2hlbiBzb21lb25lIGNyZWF0ZXMgYW4gQW1hem9uIEVDMiBzbmFwc2hvdCwgeW91IG11c3QgaW5jbHVkZSB0aGUgYGVjMjpDcmVhdGVTbmFwc2hvdGAgY3JlYXRpb24gYWN0aW9uICoqKmFuZCoqKiB0aGUgYGVjMjpDcmVhdGVUYWdzYCB0YWdnaW5nIGFjdGlvbiBpbiB0aGUgcG9saWN5LiBUbyB2aWV3IGEgcG9saWN5IGZvciB0aGlzIHNjZW5hcmlvIHRoYXQgdXNlcyBgYXdzOlRhZ0tleXNgLCBzZWUgW0NyZWF0aW5nIGEgU25hcHNob3Qgd2l0aCBUYWdzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTRUMyL2xhdGVzdC9Vc2VyR3VpZGUvRXhhbXBsZVBvbGljaWVzX0VDMi5odG1sI2lhbS1jcmVhdGluZy1zbmFwc2hvdC13aXRoLXRhZ3MpIGluIHRoZSAqQW1hem9uIEVDMiBVc2VyIEd1aWRlIGZvciBMaW51eCBJbnN0YW5jZXMqLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHRhZyBrZXkocylcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZignYXdzOlRhZ0tleXMnLCB2YWx1ZSwgb3BlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBhcmUgdGhlIGRhdGUgYW5kIHRpbWUgdGhhdCB0ZW1wb3Jhcnkgc2VjdXJpdHkgY3JlZGVudGlhbHMgd2VyZSBpc3N1ZWQgd2l0aCB0aGUgZGF0ZSBhbmQgdGltZSB0aGF0IHlvdSBzcGVjaWZ5LlxuICAgKlxuICAgKiAqKkF2YWlsYWJpbGl0eToqKiBUaGlzIGtleSBpcyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IG9ubHkgd2hlbiB0aGUgcHJpbmNpcGFsIHVzZXMgdGVtcG9yYXJ5IGNyZWRlbnRpYWxzIHRvIG1ha2UgdGhlIHJlcXVlc3QuIFRoZXkga2V5IGlzIG5vdCBwcmVzZW50IGluIEFXUyBDTEksIEFXUyBBUEksIG9yIEFXUyBTREsgcmVxdWVzdHMgdGhhdCBhcmUgbWFkZSB1c2luZyBhY2Nlc3Mga2V5cy5cbiAgICpcbiAgICogVG8gbGVhcm4gd2hpY2ggc2VydmljZXMgc3VwcG9ydCB1c2luZyB0ZW1wb3JhcnkgY3JlZGVudGlhbHMsIHNlZSBbQVdTIFNlcnZpY2VzIFRoYXQgV29yayB3aXRoIElBTV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9hd3Mtc2VydmljZXMtdGhhdC13b3JrLXdpdGgtaWFtLmh0bWwpLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdG9rZW5pc3N1ZXRpbWVcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSBkYXRlIGFuZCB0aW1lIHRvIGNoZWNrIGZvci4gQ2FuIGJlIGEgc3RyaW5nIGluIG9uZSBvZiB0aGUgW1czQyBpbXBsZW1lbnRhdGlvbnMgb2YgdGhlIElTTyA4NjAxIGRhdGVdKGh0dHBzOi8vd3d3LnczLm9yZy9UUi9OT1RFLWRhdGV0aW1lKSAoZS5nLiBgMjAyMC0wNC0wMVQwMDowMDowMFpgKSBvciBpbiBlcG9jaCAoVU5JWCkgdGltZSBvciBhIGBEYXRlKClgIG9iamVjdCAoSmF2YVNjcmlwdCBvbmx5KVxuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbZGF0ZSBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfRGF0ZSkuICoqRGVmYXVsdDoqKiBgRGF0ZUdyZWF0ZXJUaGFuRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVG9rZW5Jc3N1ZVRpbWUoXG4gICAgdmFsdWU6IHN0cmluZyB8IERhdGUsXG4gICAgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZ1xuICApIHtcbiAgICByZXR1cm4gdGhpcy5pZihcbiAgICAgICdhd3M6VG9rZW5Jc3N1ZVRpbWUnLFxuICAgICAgZGF0ZVRvU3RyaW5nKHZhbHVlKSxcbiAgICAgIG9wZXJhdG9yIHx8IG5ldyBPcGVyYXRvcigpLmRhdGVHcmVhdGVyVGhhbkVxdWFscygpXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSByZXF1ZXN0ZXIncyBjbGllbnQgYXBwbGljYXRpb24gd2l0aCB0aGUgYXBwbGljYXRpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgYWx3YXlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQuXG4gICAqXG4gICAqICoqV2FybmluZzoqKiBUaGlzIGtleSBzaG91bGQgYmUgdXNlZCBjYXJlZnVsbHkuIFNpbmNlIHRoZSBgYXdzOlVzZXJBZ2VudGAgdmFsdWUgaXMgcHJvdmlkZWQgYnkgdGhlIGNhbGxlciBpbiBhbiBIVFRQIGhlYWRlciwgdW5hdXRob3JpemVkIHBhcnRpZXMgY2FuIHVzZSBtb2RpZmllZCBvciBjdXN0b20gYnJvd3NlcnMgdG8gcHJvdmlkZSBhbnkgYGF3czpVc2VyQWdlbnRgIHZhbHVlIHRoYXQgdGhleSBjaG9vc2UuIEFzIGEgcmVzdWx0LCBgYXdzOlVzZXJBZ2VudGAgc2hvdWxkIG5vdCBiZSB1c2VkIHRvIHByZXZlbnQgdW5hdXRob3JpemVkIHBhcnRpZXMgZnJvbSBtYWtpbmcgZGlyZWN0IEFXUyByZXF1ZXN0cy4gWW91IGNhbiB1c2UgaXQgdG8gYWxsb3cgb25seSBzcGVjaWZpYyBjbGllbnQgYXBwbGljYXRpb25zLCBhbmQgb25seSBhZnRlciB0ZXN0aW5nIHlvdXIgcG9saWN5LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdXNlcmFnZW50XG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgVXNlciBBZ2VudCBzdHJpbmcocylcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVXNlckFnZW50KFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6VXNlckFnZW50JywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSByZXF1ZXN0ZXIncyBwcmluY2lwYWwgaWRlbnRpZmllciB3aXRoIHRoZSBJRCB0aGF0IHlvdSBzcGVjaWZ5LiBGb3IgSUFNIHVzZXJzLCB0aGUgcmVxdWVzdCBjb250ZXh0IHZhbHVlIGlzIHRoZSB1c2VyIElELiBGb3IgSUFNIHJvbGVzLCB0aGlzIHZhbHVlIGZvcm1hdCBjYW4gdmFyeS4gRm9yIGRldGFpbHMgYWJvdXQgaG93IHRoZSBpbmZvcm1hdGlvbiBhcHBlYXJzIGZvciBkaWZmZXJlbnQgcHJpbmNpcGFscywgc2VlIFtTcGVjaWZ5aW5nIGEgUHJpbmNpcGFsXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3ByaW5jaXBhbC5odG1sI1ByaW5jaXBhbF9zcGVjaWZ5aW5nKS5cbiAgICpcbiAgICogKipBdmFpbGFiaWxpdHk6KiogVGhpcyBrZXkgaXMgaW5jbHVkZWQgaW4gdGhlIHJlcXVlc3QgY29udGV4dCBmb3IgYWxsIHNpZ25lZCByZXF1ZXN0cy4gQW5vbnltb3VzIHJlcXVlc3RzIGRvIG5vdCBpbmNsdWRlIHRoaXMga2V5LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdXNlcmlkXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgcHJpbmNpcGFsIGlkZW50aWZpZXIocylcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVXNlcmlkKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6dXNlcmlkJywgdmFsdWUsIG9wZXJhdG9yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlIHRoZSByZXF1ZXN0ZXIncyB1c2VyIG5hbWUgd2l0aCB0aGUgdXNlciBuYW1lIHRoYXQgeW91IHNwZWNpZnkuIEZvciBkZXRhaWxzIGFib3V0IGhvdyB0aGUgaW5mb3JtYXRpb24gYXBwZWFycyBmb3IgZGlmZmVyZW50IHByaW5jaXBhbHMsIHNlZSBbU3BlY2lmeWluZyBhIFByaW5jaXBhbF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19wcmluY2lwYWwuaHRtbCNQcmluY2lwYWxfc3BlY2lmeWluZykuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGFsd2F5cyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IGZvciBJQU0gdXNlcnMuIEFub255bW91cyByZXF1ZXN0cyBhbmQgcmVxdWVzdHMgdGhhdCBhcmUgbWFkZSB1c2luZyB0aGUgQVdTIGFjY291bnQgcm9vdCB1c2VyIG9yIElBTSByb2xlcyBkbyBub3QgaW5jbHVkZSB0aGlzIGtleS5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXVzZXJuYW1lXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdXNlciBuYW1lKHMpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1VzZXJuYW1lKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKCdhd3M6dXNlcm5hbWUnLCB2YWx1ZSwgb3BlcmF0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIHdoZXRoZXIgYW4gQVdTIHNlcnZpY2UgbWFrZXMgYSByZXF1ZXN0IHRvIGFub3RoZXIgc2VydmljZSBvbiB5b3VyIGJlaGFsZi5cbiAgICpcbiAgICogVGhlIHJlcXVlc3QgY29udGV4dCBrZXkgcmV0dXJucyBgdHJ1ZWAgd2hlbiBhIHNlcnZpY2UgdXNlcyB0aGUgY3JlZGVudGlhbHMgb2YgYW4gSUFNIHByaW5jaXBhbCB0byBtYWtlIGEgcmVxdWVzdCBvbiBiZWhhbGYgb2YgdGhlIHByaW5jaXBhbC4gVGhlIGNvbnRleHQga2V5IHJldHVybnMgYGZhbHNlYCBpZiB0aGUgc2VydmljZSB1c2VzIGEgW3NlcnZpY2Ugcm9sZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Rlcm1zLWFuZC1jb25jZXB0cy5odG1sI2lhbS10ZXJtLXNlcnZpY2Utcm9sZSkgb3IgW3NlcnZpY2UtbGlua2VkIHJvbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc190ZXJtcy1hbmQtY29uY2VwdHMuaHRtbCNpYW0tdGVybS1zZXJ2aWNlLWxpbmtlZC1yb2xlKSB0byBtYWtlIGEgY2FsbCBvbiB0aGUgcHJpbmNpcGFsJ3MgYmVoYWxmLiBUaGUgcmVxdWVzdCBjb250ZXh0IGtleSBhbHNvIHJldHVybnMgYGZhbHNlYCB3aGVuIHRoZSBwcmluY2lwYWwgbWFrZXMgdGhlIGNhbGwgZGlyZWN0bHkuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGFsd2F5cyBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdCBjb250ZXh0IGZvciBtb3N0IHNlcnZpY2VzLlxuICAgKlxuICAgKlRoZSBmb2xsb3dpbmcgc2VydmljZXMgZG8gbm90IGN1cnJlbnRseSBzdXBwb3J0IGBhd3M6VmlhQVdTU2VydmljZWA6XG4gICAqLSBBbWF6b24gRUMyXG4gICAqLSBBV1MgR2x1ZVxuICAgKi0gQVdTIExha2UgRm9ybWF0aW9uXG4gICAqLSBBV1MgT3BzV29ya3NcbiAgICpcbiAgICogWW91IGNhbiB1c2UgdGhpcyBjb25kaXRpb24ga2V5IHRvIGFsbG93IG9yIGRlbnkgYWNjZXNzIGJhc2VkIG9uIHdoZXRoZXIgYSByZXF1ZXN0IHdhcyBtYWRlIGJ5IGEgc2VydmljZS4gVG8gdmlldyBhbiBleGFtcGxlIHBvbGljeSwgc2VlIFtBV1M6IERlbmllcyBBY2Nlc3MgdG8gQVdTIEJhc2VkIG9uIHRoZSBTb3VyY2UgSVBdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZXhhbXBsZXNfYXdzX2RlbnktaXAuaHRtbCkuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy12aWFhd3NzZXJ2aWNlXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBXaGV0aGVyIGEgcmVxdWVzdCB3YXMgbWFkZSBieSBhIHNlcnZpY2UuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ZpYUFXU1NlcnZpY2UodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoXG4gICAgICBgYXdzOlZpYUFXU1NlcnZpY2VgLFxuICAgICAgdHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSxcbiAgICAgIG5ldyBPcGVyYXRvcigpLmJvb2woKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQ29tcGFyZSB0aGUgSVAgYWRkcmVzcyBmcm9tIHdoaWNoIGEgcmVxdWVzdCB3YXMgbWFkZSB3aXRoIHRoZSBJUCBhZGRyZXNzIHRoYXQgeW91IHNwZWNpZnkuIEluIGEgcG9saWN5LCB0aGUga2V5IG1hdGNoZXMgb25seSBpZiB0aGUgcmVxdWVzdCBvcmlnaW5hdGVzIGZyb20gdGhlIHNwZWNpZmllZCBJUCBhZGRyZXNzIGFuZCBpdCBnb2VzIHRocm91Z2ggYSBWUEMgZW5kcG9pbnQuXG4gICAqXG4gICAqICoqQXZhaWxhYmlsaXR5OioqIFRoaXMga2V5IGlzIGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0IGNvbnRleHQgb25seSBpZiB0aGUgcmVxdWVzdCBpcyBtYWRlIHVzaW5nIGEgVlBDIGVuZHBvaW50LlxuICAgKlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFtDb250cm9sbGluZyBBY2Nlc3MgdG8gU2VydmljZXMgd2l0aCBWUEMgRW5kcG9pbnRzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdnBjL2xhdGVzdC91c2VyZ3VpZGUvdnBjLWVuZHBvaW50cy1hY2Nlc3MuaHRtbCkgaW4gdGhlICpBbWF6b24gVlBDIFVzZXIgR3VpZGUqLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdnBjc291cmNlaXBcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSBWUEMgc291cmNlIElQKHMpXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIElQIFthZGRyZXNzIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19JUEFkZHJlc3MpLiAqKkRlZmF1bHQ6KiogYElwQWRkcmVzc2BcbiAgICovXG4gIHB1YmxpYyBpZkF3c1ZwY1NvdXJjZUlwKFxuICAgIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSxcbiAgICBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmlmKFxuICAgICAgJ2F3czpWcGNTb3VyY2VJcCcsXG4gICAgICB2YWx1ZSxcbiAgICAgIG9wZXJhdG9yIHx8IG5ldyBPcGVyYXRvcigpLmlwQWRkcmVzcygpXG4gICAgKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBkYXRlVG9TdHJpbmcodmFsdWU6IERhdGUgfCBzdHJpbmcgfCBudW1iZXIpOiBzdHJpbmcgfCBudW1iZXIge1xuICBpZiAodHlwZW9mICh2YWx1ZSBhcyBEYXRlKS5nZXRNb250aCA9PT0gJ2Z1bmN0aW9uJykge1xuICAgIHZhbHVlID0gKHZhbHVlIGFzIERhdGUpLnRvSVNPU3RyaW5nKCk7XG4gIH1cbiAgcmV0dXJuIHZhbHVlIGFzIHN0cmluZztcbn1cbiJdfQ==