import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import { Duration } from '@aws-cdk/core';
import { IConstruct, Construct } from 'constructs';
import { BaseTargetGroupProps, ITargetGroup, LoadBalancerTargetProps, TargetGroupAttributes, TargetGroupBase, TargetGroupImportProps } from '../shared/base-target-group';
import { ApplicationProtocol } from '../shared/enums';
import { IApplicationListener } from './application-listener';
import { HttpCodeTarget } from './application-load-balancer';
/**
 * Properties for defining an Application Target Group.
 */
export interface ApplicationTargetGroupProps extends BaseTargetGroupProps {
    /**
     * The protocol to use.
     *
     * @default - Determined from port if known, optional for Lambda targets.
     */
    readonly protocol?: ApplicationProtocol;
    /**
     * The port on which the listener listens for requests.
     *
     * @default - Determined from protocol if known, optional for Lambda targets.
     */
    readonly port?: number;
    /**
     * The time period during which the load balancer sends a newly registered target a linearly increasing share of the traffic to the target group.
     *
     * The range is 30-900 seconds (15 minutes).
     *
     * @default 0
     */
    readonly slowStart?: Duration;
    /**
     * The stickiness cookie expiration period.
     *
     * Setting this value enables load balancer stickiness.
     *
     * After this period, the cookie is considered stale. The minimum value is
     * 1 second and the maximum value is 7 days (604800 seconds).
     *
     * @default Duration.days(1)
     */
    readonly stickinessCookieDuration?: Duration;
    /**
     * The targets to add to this target group.
     *
     * Can be `Instance`, `IPAddress`, or any self-registering load balancing
     * target. If you use either `Instance` or `IPAddress` as targets, all
     * target must be of the same type.
     *
     * @default - No targets.
     */
    readonly targets?: IApplicationLoadBalancerTarget[];
}
/**
 * Define an Application Target Group.
 */
export declare class ApplicationTargetGroup extends TargetGroupBase implements IApplicationTargetGroup {
    /**
     * Import an existing target group.
     */
    static fromTargetGroupAttributes(scope: Construct, id: string, attrs: TargetGroupAttributes): IApplicationTargetGroup;
    /**
     * (deprecated) Import an existing target group.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope: Construct, id: string, props: TargetGroupImportProps): IApplicationTargetGroup;
    private readonly connectableMembers;
    private readonly listeners;
    private readonly protocol?;
    private readonly port?;
    /**
     *
     */
    constructor(scope: Construct, id: string, props?: ApplicationTargetGroupProps);
    /**
     * Add a load balancing target to this target group.
     */
    addTarget(...targets: IApplicationLoadBalancerTarget[]): void;
    /**
     * Enable sticky routing via a cookie to members of this target group.
     */
    enableCookieStickiness(duration: Duration): void;
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     */
    registerConnectable(connectable: ec2.IConnectable, portRange?: ec2.Port): void;
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener: IApplicationListener, associatingConstruct?: IConstruct): void;
    /**
     * Full name of first load balancer.
     */
    get firstLoadBalancerFullName(): string;
    /**
     * Return the given named metric for this Application Load Balancer Target Group.
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of IPv6 requests received by the target group.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of healthy hosts in the target group.
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of unhealthy hosts in the target group.
     *
     * @default Average over 5 minutes
     */
    metricUnhealthyHostCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code: HttpCodeTarget, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCountPerTarget(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    protected validate(): string[];
}
/**
 * A Target Group for Application Load Balancers.
 */
export interface IApplicationTargetGroup extends ITargetGroup {
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener: IApplicationListener, associatingConstruct?: IConstruct): void;
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     */
    registerConnectable(connectable: ec2.IConnectable, portRange?: ec2.Port): void;
    /**
     * Add a load balancing target to this target group.
     */
    addTarget(...targets: IApplicationLoadBalancerTarget[]): void;
}
/**
 * Interface for constructs that can be targets of an application load balancer.
 */
export interface IApplicationLoadBalancerTarget {
    /**
     * Attach load-balanced target to a TargetGroup.
     *
     * May return JSON to directly add to the [Targets] list, or return undefined
     * if the target will register itself with the load balancer.
     */
    attachToApplicationTargetGroup(targetGroup: IApplicationTargetGroup): LoadBalancerTargetProps;
}
