# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['metaDMG', 'metaDMG.fit', 'metaDMG.loggers']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=5.4.1,<6.0.0',
 'click-help-colors>=0.9.1,<0.10.0',
 'iminuit>=2.8.3,<3.0.0',
 'joblib>=1.0.1,<2.0.0',
 'logger-tt>=1.6.1,<2.0.0',
 'numpyro>=0.7.2,<0.8.0',
 'pandas>=1.3.2,<2.0.0',
 'pyarrow>=5.0.0,<6.0.0',
 'typer>=0.3.2,<0.4.0']

extras_require = \
{':python_version >= "3.8" and python_version < "3.10"': ['numba>=0.54.0,<0.55.0',
                                                          'scipy>=1.7.1,<2.0.0'],
 'all': ['metaDMG-viz>=0.5.1,<0.6.0'],
 'viz': ['metaDMG-viz>=0.5.1,<0.6.0']}

entry_points = \
{'console_scripts': ['metaDMG = metaDMG.cli:cli_main']}

setup_kwargs = {
    'name': 'metadmg',
    'version': '0.10.1',
    'description': 'metaDMG: Estimating ancient damage in (meta)genomic DNA rapidly',
    'long_description': '# metaDMG: Estimating ancient damage in (meta)genomic DNA rapidly\n\n---\n\n#### Work in progress. Please contact christianmichelsen@gmail.com for further information.\n\n---\n\n\n## Installation:\n\n```\nconda env create --file environment.yaml\n```\nor, if you have mamba installed (faster)\n```\nmamba env create --file environment.yaml\n```\n\nor, by using pip:\n```\npip install "metaDMG[all]"\n```\nor, with Poetry:\n```\npoetry add "metaDMG[all]"\n```\n\n---\n\n## Workflow:\n\n\nCreate `config.yaml` file:\n```console\n$ metaDMG config ./raw_data/example.bam \\\n    --names raw_data/names.dmp.gz \\\n    --nodes raw_data/nodes.dmp.gz \\\n    --acc2tax raw_data/combined_taxid_accssionNO_20200425.gz\n```\nRun actual program:\n```console\n$ metaDMG compute\n```\nSee the results in the dashboard:\n```console\n$ metaDMG dashboard\n```\n\n\n---\n\n## Usage:\n\nmetaDMG works by first creating a config file using the `config` command. This file contains all of the information related to metaDMG such that you only have to type this once. The config file is saved in the current directory as `config.yaml` and can subsequently be edited in any text editor of your like.\n\nAfter the config has been created, we run the actual program using the `compute` command. This can take a while depending on the number (and size) of the files.\n\nFinally the results are saved in `{storage-dir}/results` directory (`data/results` by default). These can be viewed with the interactive dashboard using the `dashboard` command.\n\n\n---\n\n# `config`\n\n#### CLI options:\n\n`metaDMG config` takes a single argument, `samples`, and a bunch of additional options. The `samples` refer to a single or multiple alignment-files (or a directory containing them), all with the file extensions: `.bam`, `.sam`, and `.sam.gz`.\n\nThe options are listed below:\n\n- Input files:\n  - `--names`: Path to the (NCBI) `names.dmp.gz`. Mandatory.\n  - `--nodes`: Path to the (NCBI) `nodes.dmp.gz`. Mandatory.\n  - `--acc2tax`: Path to the (NCBI) `acc2tax.gz`. Mandatory.\n\n- LCA parameters:\n  - `--simscorelow`: Normalised edit distance (read to reference similarity) minimum. Number between 0-1. Default: 0.95.\n  - `--simscorehigh`: Normalised edit distance (read to reference similarity) maximum. Number between 0-1 Default: 1.0.\n  - `--editdistmin`: Minimum edit distance (read to reference similarity). Number between 0-10. Default: 0.\n  - `--editdistmax`: Maximum edit distance (read to reference similarity). Number between 0-10. Default: 10.\n  - `--minmapq`: Minimum mapping quality. Default: 0.\n  - `--max-position`: Maximum position in the sequence to include. Default is (+/-) 15 (forward/reverse).\n  - `--weighttype`: Method for calculating weights. Default is 1.\n  - `--fix-ncbi`: Fix the (ncbi) database. Disable (0) if using a custom database. Default is 1.\n  - `--lca-rank`: The LCA rank used in ngsLCA. Can be either `family`, `genus`, `species` or `""` (everything). Default is `""`.\n  <!-- - `--forward-only`: Only fit the forward strand. -->\n\n\n- General parameters:\n  - `--storage-dir`: Path where the generated output files and folders are stored. Default: `./data/`.\n  - `--cores`: The maximum number of cores to use. Default is 1.\n  - `--sample-prefix`: Prefix for the sample names.\n  - `--sample-suffix`: Suffix for the sample names.\n  - `--config-path`: The name of the generated config file. Default: `config.yaml`.\n\n- Boolean flags (does not take any values, only the flag). Default is false.\n  - `--bayesian`: Include a fully Bayesian model (probably better, but also _a lot_ slower, about a factor of 100).\n\n\n\n```console\n$ metaDMG config ./raw_data/example.bam \\\n    --names raw_data/names.dmp.gz \\\n    --nodes raw_data/nodes.dmp.gz \\\n    --acc2tax raw_data/combined_taxid_accssionNO_20200425.gz \\\n    --cores 4\n```\n\n\nmetaDMG is pretty versatile regarding its input argument and also accepts multiple alignment files:\n```console\n$ metaDMG config ./raw_data/*.bam [...]\n```\nor even an entire directory containing alignment files (`.bam`, `.sam`, and `.sam.gz`):\n```console\n$ metaDMG config ./raw_data/ [...]\n```\n\n---\n\n\n# `compute`\n\n\nThe `metaDMG compute` command takes an optional config-file as argument\n(defaults to `config.yaml` if not specified).\n\n#### Example:\n\n```console\n$ metaDMG compute\n```\n\n```console\n$ metaDMG compute non-default-config.yaml\n```\n\n---\n\n# `dashboard`\n\n\nThe `metaDMG dashboard` command takes first an optional config-file as argument\n(defaults to `config.yaml` if not specified) followed by the following CLI options:\n\n#### CLI options:\n\n- `--port`: The port to be used for the dashboard. Default is `8050`.\n- `--host`: The dashboard host adress. Default is `0.0.0.0`.\n- `--debug`: Boolean flag that allows for debugging the dashboard. Only for internal usage.\n\n#### Example:\n\n```console\n$ metaDMG dashboard\n```\n\n```console\n$ metaDMG dashboard non-default-config.yaml --port 8050 --host 0.0.0.0\n```\n\n---\n\n# Results\n\nThe column names in the results and their explanation:\n\n- General parameters:\n  - `tax_id`: The tax ID. int64.\n  - `tax_name`: The tax name. string.\n  - `tax_rank`: The tax rank. string.\n  - `sample`: The name of the original sample. string.\n  - `N_reads`: The number of reads. int64.\n  - `N_alignments`: The number of alignments. int64.\n\n- Fit related parameters:\n  - `lambda_LR`: The likelihood ratio between the null model and the ancient damage model. This can be interpreted as the fit certainty, where higher values means higher certainty. float32.\n  - `lambda_LR_P`: The likelihood ratio expressed as a probability. float32.\n  - `lambda_LR_z`: The likelihood ratio expressed as number of ![](https://latex.codecogs.com/svg.image?%5Csigma). float32.\n  - `D_max`: The estimated damage. This can be interpreted as the amount of damage in the specific taxa. float32.\n  - `q`: The damage decay rate. float32.\n  - `A`: The background independent damage. float32.\n  - `c`: The background. float32.\n  - `phi`: The concentration for a beta binomial distribution (parametrised by ![](https://latex.codecogs.com/svg.image?%5Cmu) and ![](https://latex.codecogs.com/svg.image?%5Cphi)). float32.\n  - `rho_Ac`: The correlation between `A` and `c`. High values of this are often a sign of a bad fit. float32.\n  - `valid`: Wether or not the fit is valid (defined by [iminuit](https://iminuit.readthedocs.io/en/stable/)). bool.\n  - `asymmetry`: An estimate of the asymmetry of the forward and reverse fits. See below for more information. float32.\n  - `XXX_std`: the uncertainty (standard deviation) of the variable `XXX` for `D_max`, `A`, `q`, `c`, and `phi`.\n  - `forward__XXX`: The same description as above for variable `XXX`, but only for the forward read.\n  - `reverse__XXX`: The same description as above for variable `XXX`, but only for the reverse read.\n\n- Read related parameters\n  - `mean_L`: The mean read length of all the individual, unique reads that map to the specific taxa. float64.\n  - `std_L`: The standard deviation of the above. float64.\n  - `mean_GC`: The mean GC content of all the individual, unique reads that map to the specific taxa. float64.\n  - `std_GC`: The standard deviation of the above. float64.\n  - `tax_path`: The taxanomic path from the LCA to the root through the phylogenetic tree. string.\n\n- Count related paramters:\n  - `N_x=1_forward`: The total number of _"trials"_, ![](https://latex.codecogs.com/svg.image?N), at position ![](https://latex.codecogs.com/svg.image?x=1): ![](https://latex.codecogs.com/svg.image?N(x=1)) in the forward direction. int64.\n  - `N_x=1_reverse`:  Same as above, but for the reverse direction. int64.\n  - `N_sum_forward`:  The sum of ![](https://latex.codecogs.com/svg.image?N) over all positions in the forward direction. int64.\n  - `N_sum_reverse`: Same as above, but for the reverse direction. int64.\n  - `N_sum_total`:  The total sum `N_sum_forward` and `N_sum_reverse`. int64.\n  - `N_min`: The minimum of ![](https://latex.codecogs.com/svg.image?N) for all positions (forward and reverse alike). int64.\n  - `k_sum_forward`:  The total number of _"successes"_, ![](https://latex.codecogs.com/svg.image?k), summed over all positions in the forward direction. int64.\n  - `k_sum_reverse`: Same as above, but for the reverse direction. int64..\n  - `k_sum_total`: The total sum `k_sum_forward` and `k_sum_reverse`. int64.\n  - `k+i`: The number of _"successes"_, ![](https://latex.codecogs.com/svg.image?k) at position ![](https://latex.codecogs.com/svg.image?z=i): ![](https://latex.codecogs.com/svg.image?k(x=1)) in the forward direction. int64.\n  - `k-i`: Same as above, but for the reverse direction. int64.\n  - `N+i`: The number of _"trials"_, ![](https://latex.codecogs.com/svg.image?N) at position ![](https://latex.codecogs.com/svg.image?z=i): ![](https://latex.codecogs.com/svg.image?N(x=1)) in the forward direction. int64.\n  - `N-i`: Same as above, but for the reverse direction. int64.\n  - `f+i`: The fraction between ![](https://latex.codecogs.com/svg.image?k) and ![](https://latex.codecogs.com/svg.image?N) at position ![](https://latex.codecogs.com/svg.image?z=i) in the forward direction. int64.\n  - `f-i`: Same as above, but for the reverse direction. int64.\n\n\n---\n\n# `convert`\n\nThe `metaDMG convert` command takes first an optional config-file as argument\n(defaults to `config.yaml` if not specified) used to infer the results directory\n followed by the following CLI options:\n\n#### CLI options:\n\n- `--output`: Mandatory output path.\n- `--results_dir`: Direct path to the results directory.\n\nNote that neither the config-file nor `--results_dir` have to be specified\n(in which just the default `config.yaml` is used), however,\nboth cannot be set at the same time.\n\n#### Example:\n\n```console\n$ metaDMG convert --output ./directory/to/contain/results.csv\n```\n\n```console\n$ metaDMG convert non-default-config.yaml --output ./directory/to/contain/results.csv\n```\n\n---\n\n# `filter`\n\nThe `metaDMG filter` command takes first an optional config-file as argument\n(defaults to `config.yaml` if not specified) used to infer the results directory\n followed by the following CLI options:\n\n#### CLI options:\n\n- `--output`: Mandatory output path.\n- `--query`: The query string to use for filtering. Follows the [Pandas Query()]([www.link.dk](https://pandas.pydata.org/pandas-docs/stable/user_guide/indexing.html#the-query-method)) syntax.\nDefault is `""` which applies no filtering and is thus similar to the `metaDMG convert` command.\n- `--results_dir`: Direct path to the results directory.\n\nNote that neither the config-file nor `--results_dir` have to be specified\n(in which just the default `config.yaml` is used), however,\nboth cannot be set at the same time.\n\n\n\n#### Example:\n\n```console\n$ metaDMG filter --output convert-no-query.csv # similar to metaDMG convert\n```\n\n```console\n$ metaDMG filter --output convert-test.csv --query "N_reads > 5_000 & sample in [\'subs\', \'SPL_195_9299\'] & tax_name == \'root\'"\n```\n\n---\n\n\n---\n\nIf you only want to install some of the tools, you can run:\n```console\npip install "metaDMG[fit]"\n```\nto only install the fitting part of the tool, or:\n```console\npip install "metaDMG[viz]"\n```\nto only install the interactive plotting tool (requires you to have gotten the results from somewhere else).\n\n---\n\n',
    'author': 'Christian Michelsen',
    'author_email': 'christianmichelsen@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/metaDMG/metaDMG/',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
