# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['confme', 'confme.core', 'confme.source_backend', 'confme.utils']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.4,<2.0', 'pyyaml>=5.3,<6.0', 'tabulate>=0.8.9,<0.9.0']

setup_kwargs = {
    'name': 'confme',
    'version': '0.1.0',
    'description': 'Easy configuration management in python',
    'long_description': '# ConfMe: Configuration Made Easy 💖\n\n[![image](https://img.shields.io/pypi/v/confme?color=blue)](https://pypi.org/project/confme/) [![image](https://img.shields.io/pypi/l/confme)](https://pypi.org/project/confme/) [![image](https://github.com/iwanbolzern/ConfMe/workflows/Test/badge.svg?branch=master)](https://pypi.org/project/confme/)\n[![image](https://img.shields.io/pypi/pyversions/confme?color=blue)](https://pypi.org/project/confme/)\n\nConfMe is a simple to use, production ready application configuration management library, which takes into consideration the following three thoughts:\n1. Access to configuration values must be safe at runtime. **No ```myconfig[\'value1\'][\'subvalue\']``` anymore!**\n2. The configuration must be checked for consistency at startup e.g. type check, range check, ...\n3. Secrets shall be injectable from environment variables\n\nConfMe makes all these features possible with just a few type annotations on plain Python objects.\n\n## Installation\nConfMe can be installed from the official python package repository [pypi](https://pypi.org/project/confme/)\n\n```\npip install confme\n```\n\nOr, if you\'re using pipenv:\n\n```\npipenv install confme\n```\n\nOr, if you\'re using poetry:\n\n```\npoetry add confme\n```\n\n## Basic Usage of confme\nDefine your config structure as plain python objects with type annotations:\n\n```python\nfrom confme import BaseConfig\n\nclass DatabaseConfig(BaseConfig):\n    host: str\n    port: int\n    user: str\n\nclass MyConfig(BaseConfig):\n    name: str\n    database: DatabaseConfig\n```\n\nCreate a configuration yaml file with the same structure as your configuration classes have:\n\n```yaml\nname: "Database Application"\ndatabase:\n    host: "localhost"\n    port: 5000\n    user: "any-db-user"\n```\n\nLoad the yaml file into your Python object structure and access it in a secure manner:\n\n```python\nmy_config = MyConfig.load(\'config.yaml\')\n\nprint(f\'Using database connection {my_config.database.host} \'\n      f\'on port {my_config.database.port}\')\n```\n\nIn the background the yaml file is parsed and mapped to the defined object structure. While mapping the values to object properties, type checks are performed. If a value is not available or is not of the correct type, an error is generated already when the configuration is loaded.\n\n## Supported Annotations\nConfMe is based on pydantic and supports all annotations provided by pydantic. The most important annotations are listed and explain bellow. For the whole list, please checkout [Field Types](https://pydantic-docs.helpmanual.io/usage/types/):\n- str\n- int\n- float\n- bool\n- typing.List[x]\n- typing.Optional[x]\n- [Secret](#secret)\n- [Range](#range)\n- [Enum](#enum)\n\n### Secret\nWith the Secret annotation you can inject secrets from environment variables directly into your configuration structure. This is especially handy when you\'re deploying applications by using docker. Therefore, let\'s extend the previous example with a Secret annotation:\n\n```python\nfrom confme import BaseConfig\nfrom confme.annotation import Secret\n\nclass DatabaseConfig(BaseConfig):\n    ...\n    password: str = Secret(\'highSecurePassword\')\n```\n\nNow set the password to the defined environment variable:\n\n```bash\nexport highSecurePassword="This is my password"\n```\n\nLoad your config and check for the injected password.\n\n```\nmy_config = MyConfig.load(\'config.yaml\')\nprint(f\'My password is: {my_config.database.password}\')\n```\n\n### Range\n\nConfME supports OpenRange, ClosedRange and MixedRange values. The terms open and close are similar to open and closed intervals in mathematics. This means, if you want to include the lower and upper range use ClosedRange otherwise OpenRange:\n* ```ClosedRange(2, 3)``` will include 2 and 3\n* ```OpenRange(2, 3)``` will not include 2 and 3\n\nIf you want to have a mixture of both, e.g. include 2 but exclude 3 use MixedRange:\n* ```MixedRange(ge=2, lt=3)``` will include 2 but exclude 3\n\n```python\nfrom confme import BaseConfig\nfrom confme.annotation import ClosedRange\n\nclass DatabaseConfig(BaseConfig):\n    ...\n    password: int = ClosedRange(2, 3)\n```\n\n### Enum\n\nIf a Python Enum is set as type annotation, ConfMe expect to find the enum value in the configuration file.\n\n```python\nfrom confme import BaseConfig\nfrom enum import Enum\n\nclass DatabaseConnection(Enum):\n    TCP = \'tcp\'\n    UDP = \'udp\'\n\nclass DatabaseConfig(BaseConfig):\n    ...\n    connection_type: DatabaseConnection\n```\n\n## Switching configuration based on Environment\nA very common situation is that configurations must be changed based on the execution environment (dev, test, prod). This can be accomplished \nby registering a folder with one .yaml file per environment and seting the `ENV` environment variable to the value you need. An example could look \nlike this:  \n\nLet\'s assume we have three environments (dev, test, prod) and one configuration file per environment in the following folder structure:\n```\nproject\n│\n└───config\n│   │   my_prod_config.yaml\n│   │   my_test_config.yaml\n│   │   my_dev_config.yaml\n│   \n└───src\n│   │   app.py\n│   │   my_config.py\n```\nThe definition of `my_config.py` is equivalent to the one used in the basic introduction section and `app.py` uses our configuration the \nfollowing way:\n```python\n# we register the folder where ConfME can find the configuration files\nMyConfig.register_folder(Path(__file__).parent / \'../config\')\n...\n\n# we access the instance of the corresponding configuration file anywhere in our project. \nmy_config = MyConfig.get()\nprint(f\'Using database connection {my_config.database.host} \'\n      f\'on port {my_config.database.port}\')\n```\nIf now one of the following environment variables (precedence in descending order): `[\'env\', \'environment\', \'environ\', \'stage\']` is \nset e.g. `export ENV=prod` it will load the configuration file with `prod` in its name.\n\n## Parameter overwrite\nIn addition to loading configuration parameters from the configuration file, they can be passed/overwritten from the command line or environment variables. Thereby, the following precedences apply (lower number means higher precedence):\n1. **Command Line Arguments**: Check if parameter is set as command line argument. If not go one line done...\n2. **Environment Variables**: Check if parameter is set as environment variable. If not go one line done...\n3. **Configuration File**: If value was not found in one of the previous sources, it will check in the configuration file.\n\n### Overwrite Parameters from Command Line\nEspecially in the Data Science and Machine Learning area it is useful to pass certain parameters for experimental purposes as command line arguments. Therefore, all properties defined in the configuration classes are automatically offered as command line arguments in the following format:\n\n**my_program.py:**\n\n```python\nfrom confme import BaseConfig\n\nclass DatabaseConfig(BaseConfig):\n    host: str\n    port: int\n    user: str\n\nclass MyConfig(BaseConfig):\n    name: int\n    database: DatabaseConfig\n\nconfig = MyConfig.load(\'test.yaml\')\n```\n\nWhen you now start your program from the command line with the ```--help``` argument, you get the full list of all configuration options:\n\n```shell\n$ python my_program.py --help\nusage: my_program.py [-h] [--name NAME] [--database.host DATABASE.HOST] [--database.port DATABASE.PORT] [--database.user DATABASE.USER]\n\noptional arguments:\n  -h, --help            show this help message and exit\n\nConfiguration Parameters:\n  With the parameters specified bellow, the configuration values from the config file can be overwritten.\n\n  --name NAME\n  --database.host DATABASE.HOST\n  --database.port DATABASE.PORT\n  --database.user DATABASE.USER\n```\n\n### Overwrite Parameters with Environment Variables\nLikewise to overwriting parameters from the commandline you can also overwrite by passing environment variables. Therefore, simply set the environment variable in the same format as it would be passed as command line arguments and run your application:\n```shell\n$ export database.host=localhost\n$ python my_programm.py\n```\n\n## LICENSE\n\nConfMe is released under the [MIT](LICENSE) license.\n\n',
    'author': 'Iwan Silvan Bolzern',
    'author_email': 'iwan.bolzern@zuehlke.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/iwanbolzern/confme',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
