""" Helper class for managing a database used by PhotoInfo.export for tracking state of exports and updates """


import datetime
import logging
import os
import pathlib
import sqlite3
import sys
from abc import ABC, abstractmethod
from io import StringIO
from sqlite3 import Error
from typing import Union

from ._constants import OSXPHOTOS_EXPORT_DB
from ._version import __version__
from .utils import normalize_fs_path

__all__ = ["ExportDB_ABC", "ExportDBNoOp", "ExportDB", "ExportDBInMemory"]

OSXPHOTOS_EXPORTDB_VERSION = "4.3"
OSXPHOTOS_EXPORTDB_VERSION_MIGRATE_FILEPATH = "4.3"

OSXPHOTOS_ABOUT_STRING = f"Created by osxphotos version {__version__} (https://github.com/RhetTbull/osxphotos) on {datetime.datetime.now()}"


class ExportDB_ABC(ABC):
    """abstract base class for ExportDB"""

    @abstractmethod
    def get_uuid_for_file(self, filename):
        pass

    @abstractmethod
    def set_uuid_for_file(self, filename, uuid):
        pass

    @abstractmethod
    def set_stat_orig_for_file(self, filename, stats):
        pass

    @abstractmethod
    def get_stat_orig_for_file(self, filename):
        pass

    @abstractmethod
    def set_stat_edited_for_file(self, filename, stats):
        pass

    @abstractmethod
    def get_stat_edited_for_file(self, filename):
        pass

    @abstractmethod
    def set_stat_converted_for_file(self, filename, stats):
        pass

    @abstractmethod
    def get_stat_converted_for_file(self, filename):
        pass

    @abstractmethod
    def set_stat_exif_for_file(self, filename, stats):
        pass

    @abstractmethod
    def get_stat_exif_for_file(self, filename):
        pass

    @abstractmethod
    def get_info_for_uuid(self, uuid):
        pass

    @abstractmethod
    def set_info_for_uuid(self, uuid, info):
        pass

    @abstractmethod
    def get_exifdata_for_file(self, uuid):
        pass

    @abstractmethod
    def set_exifdata_for_file(self, uuid, exifdata):
        pass

    @abstractmethod
    def set_sidecar_for_file(self, filename, sidecar_data, sidecar_sig):
        pass

    @abstractmethod
    def get_sidecar_for_file(self, filename):
        pass

    @abstractmethod
    def get_previous_uuids(self):
        pass

    @abstractmethod
    def get_detected_text_for_uuid(self, uuid):
        pass

    @abstractmethod
    def set_detected_text_for_uuid(self, uuid, json_text):
        pass

    @abstractmethod
    def set_data(
        self,
        filename,
        uuid,
        orig_stat=None,
        exif_stat=None,
        converted_stat=None,
        edited_stat=None,
        info_json=None,
        exif_json=None,
    ):
        pass


class ExportDBNoOp(ExportDB_ABC):
    """An ExportDB with NoOp methods"""

    def __init__(self):
        self.was_created = True
        self.was_upgraded = False
        self.version = OSXPHOTOS_EXPORTDB_VERSION

    def get_uuid_for_file(self, filename):
        pass

    def set_uuid_for_file(self, filename, uuid):
        pass

    def set_stat_orig_for_file(self, filename, stats):
        pass

    def get_stat_orig_for_file(self, filename):
        pass

    def set_stat_edited_for_file(self, filename, stats):
        pass

    def get_stat_edited_for_file(self, filename):
        pass

    def set_stat_converted_for_file(self, filename, stats):
        pass

    def get_stat_converted_for_file(self, filename):
        pass

    def set_stat_exif_for_file(self, filename, stats):
        pass

    def get_stat_exif_for_file(self, filename):
        pass

    def get_info_for_uuid(self, uuid):
        pass

    def set_info_for_uuid(self, uuid, info):
        pass

    def get_exifdata_for_file(self, uuid):
        pass

    def set_exifdata_for_file(self, uuid, exifdata):
        pass

    def set_sidecar_for_file(self, filename, sidecar_data, sidecar_sig):
        pass

    def get_sidecar_for_file(self, filename):
        return None, (None, None, None)

    def get_previous_uuids(self):
        return []

    def get_detected_text_for_uuid(self, uuid):
        return None

    def set_detected_text_for_uuid(self, uuid, json_text):
        pass

    def set_data(
        self,
        filename,
        uuid,
        orig_stat=None,
        exif_stat=None,
        converted_stat=None,
        edited_stat=None,
        info_json=None,
        exif_json=None,
    ):
        pass


class ExportDB(ExportDB_ABC):
    """Interface to sqlite3 database used to store state information for osxphotos export command"""

    def __init__(self, dbfile, export_dir):
        """dbfile: path to osxphotos export database file"""
        self._dbfile = dbfile
        # export_dir is required as all files referenced by get_/set_uuid_for_file will be converted to
        # relative paths to this path
        # this allows the entire export tree to be moved to a new disk/location
        # whilst preserving the UUID to filename mapping
        self._path = export_dir
        self._conn = self._open_export_db(dbfile)
        self._insert_run_info()

    def get_uuid_for_file(self, filename):
        """query database for filename and return UUID
        returns None if filename not found in database
        """
        filepath_normalized = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "SELECT uuid FROM files WHERE filepath_normalized = ?",
                (filepath_normalized,),
            )
            results = c.fetchone()
            uuid = results[0] if results else None
        except Error as e:
            logging.warning(e)
            uuid = None
        return uuid

    def set_uuid_for_file(self, filename, uuid):
        """set UUID of filename to uuid in the database"""
        filename = str(pathlib.Path(filename).relative_to(self._path))
        filename_normalized = self._normalize_filepath(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "INSERT OR REPLACE INTO files(filepath, filepath_normalized, uuid) VALUES (?, ?, ?);",
                (filename, filename_normalized, uuid),
            )

            conn.commit()
        except Error as e:
            logging.warning(e)

    def set_stat_orig_for_file(self, filename, stats):
        """set stat info for filename
        filename: filename to set the stat info for
        stat: a tuple of length 3: mode, size, mtime"""
        filename = self._normalize_filepath_relative(filename)
        if len(stats) != 3:
            raise ValueError(f"expected 3 elements for stat, got {len(stats)}")

        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "UPDATE files "
                + "SET orig_mode = ?, orig_size = ?, orig_mtime = ? "
                + "WHERE filepath_normalized = ?;",
                (*stats, filename),
            )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def get_stat_orig_for_file(self, filename):
        """get stat info for filename
        returns: tuple of (mode, size, mtime)
        """
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "SELECT orig_mode, orig_size, orig_mtime FROM files WHERE filepath_normalized = ?",
                (filename,),
            )
            results = c.fetchone()
            if results:
                stats = results[:3]
                mtime = int(stats[2]) if stats[2] is not None else None
                stats = (stats[0], stats[1], mtime)
            else:
                stats = (None, None, None)
        except Error as e:
            logging.warning(e)
            stats = None, None, None
        return stats

    def set_stat_edited_for_file(self, filename, stats):
        """set stat info for edited version of image (in Photos' library)
        filename: filename to set the stat info for
        stat: a tuple of length 3: mode, size, mtime"""
        return self._set_stat_for_file("edited", filename, stats)

    def get_stat_edited_for_file(self, filename):
        """get stat info for edited version of image (in Photos' library)
        filename: filename to set the stat info for
        stat: a tuple of length 3: mode, size, mtime"""
        return self._get_stat_for_file("edited", filename)

    def set_stat_exif_for_file(self, filename, stats):
        """set stat info for filename (after exiftool has updated it)
        filename: filename to set the stat info for
        stat: a tuple of length 3: mode, size, mtime"""
        filename = self._normalize_filepath_relative(filename)
        if len(stats) != 3:
            raise ValueError(f"expected 3 elements for stat, got {len(stats)}")

        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "UPDATE files "
                + "SET exif_mode = ?, exif_size = ?, exif_mtime = ? "
                + "WHERE filepath_normalized = ?;",
                (*stats, filename),
            )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def get_stat_exif_for_file(self, filename):
        """get stat info for filename (after exiftool has updated it)
        returns: tuple of (mode, size, mtime)
        """
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "SELECT exif_mode, exif_size, exif_mtime FROM files WHERE filepath_normalized = ?",
                (filename,),
            )
            results = c.fetchone()
            if results:
                stats = results[:3]
                mtime = int(stats[2]) if stats[2] is not None else None
                stats = (stats[0], stats[1], mtime)
            else:
                stats = (None, None, None)
        except Error as e:
            logging.warning(e)
            stats = None, None, None
        return stats

    def set_stat_converted_for_file(self, filename, stats):
        """set stat info for filename (after image converted to jpeg)
        filename: filename to set the stat info for
        stat: a tuple of length 3: mode, size, mtime"""
        return self._set_stat_for_file("converted", filename, stats)

    def get_stat_converted_for_file(self, filename):
        """get stat info for filename (after jpeg conversion)
        returns: tuple of (mode, size, mtime)
        """
        return self._get_stat_for_file("converted", filename)

    def get_info_for_uuid(self, uuid):
        """returns the info JSON struct for a UUID"""
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute("SELECT json_info FROM info WHERE uuid = ?", (uuid,))
            results = c.fetchone()
            info = results[0] if results else None
        except Error as e:
            logging.warning(e)
            info = None

        return info

    def set_info_for_uuid(self, uuid, info):
        """sets the info JSON struct for a UUID"""
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "INSERT OR REPLACE INTO info(uuid, json_info) VALUES (?, ?);",
                (uuid, info),
            )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def get_exifdata_for_file(self, filename):
        """returns the exifdata JSON struct for a file"""
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "SELECT json_exifdata FROM exifdata WHERE filepath_normalized = ?",
                (filename,),
            )
            results = c.fetchone()
            exifdata = results[0] if results else None
        except Error as e:
            logging.warning(e)
            exifdata = None

        return exifdata

    def set_exifdata_for_file(self, filename, exifdata):
        """sets the exifdata JSON struct for a file"""
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "INSERT OR REPLACE INTO exifdata(filepath_normalized, json_exifdata) VALUES (?, ?);",
                (filename, exifdata),
            )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def get_sidecar_for_file(self, filename):
        """returns the sidecar data and signature for a file"""
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "SELECT sidecar_data, mode, size, mtime FROM sidecar WHERE filepath_normalized = ?",
                (filename,),
            )
            results = c.fetchone()
            if results:
                sidecar_data = results[0]
                sidecar_sig = (
                    results[1],
                    results[2],
                    int(results[3]) if results[3] is not None else None,
                )
            else:
                sidecar_data = None
                sidecar_sig = (None, None, None)
        except Error as e:
            logging.warning(e)
            sidecar_data = None
            sidecar_sig = (None, None, None)

        return sidecar_data, sidecar_sig

    def set_sidecar_for_file(self, filename, sidecar_data, sidecar_sig):
        """sets the sidecar data and signature for a file"""
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "INSERT OR REPLACE INTO sidecar(filepath_normalized, sidecar_data, mode, size, mtime) VALUES (?, ?, ?, ?, ?);",
                (filename, sidecar_data, *sidecar_sig),
            )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def get_previous_uuids(self):
        """returns list of UUIDs of previously exported photos found in export database"""
        conn = self._conn
        previous_uuids = []
        try:
            c = conn.cursor()
            c.execute("SELECT DISTINCT uuid FROM files")
            results = c.fetchall()
            previous_uuids = [row[0] for row in results]
        except Error as e:
            logging.warning(e)
        return previous_uuids

    def get_detected_text_for_uuid(self, uuid):
        """Get the detected_text for a uuid"""
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "SELECT text_data FROM detected_text WHERE uuid = ?",
                (uuid,),
            )
            results = c.fetchone()
            detected_text = results[0] if results else None
        except Error as e:
            logging.warning(e)
            detected_text = None

        return detected_text

    def set_detected_text_for_uuid(self, uuid, text_json):
        """Set the detected text for uuid"""
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "INSERT OR REPLACE INTO detected_text(uuid, text_data) VALUES (?, ?);",
                (
                    uuid,
                    text_json,
                ),
            )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def set_data(
        self,
        filename,
        uuid,
        orig_stat=None,
        exif_stat=None,
        converted_stat=None,
        edited_stat=None,
        info_json=None,
        exif_json=None,
    ):
        """sets all the data for file and uuid at once; if any value is None, does not set it"""
        filename = str(pathlib.Path(filename).relative_to(self._path))
        filename_normalized = self._normalize_filepath(filename)
        conn = self._conn
        try:
            c = conn.cursor()
            # update files table (if needed);
            # this statement works around fact that there was no unique constraint on files.filepath_normalized
            c.execute(
                """INSERT OR IGNORE INTO files(filepath, filepath_normalized, uuid) VALUES (?, ?, ?);""",
                (filename, filename_normalized, uuid),
            )

            if orig_stat is not None:
                c.execute(
                    "UPDATE files "
                    + "SET orig_mode = ?, orig_size = ?, orig_mtime = ? "
                    + "WHERE filepath_normalized = ?;",
                    (*orig_stat, filename_normalized),
                )

            if exif_stat is not None:
                c.execute(
                    "UPDATE files "
                    + "SET exif_mode = ?, exif_size = ?, exif_mtime = ? "
                    + "WHERE filepath_normalized = ?;",
                    (*exif_stat, filename_normalized),
                )

            if converted_stat is not None:
                c.execute(
                    "INSERT OR REPLACE INTO converted(filepath_normalized, mode, size, mtime) VALUES (?, ?, ?, ?);",
                    (filename_normalized, *converted_stat),
                )

            if edited_stat is not None:
                c.execute(
                    "INSERT OR REPLACE INTO edited(filepath_normalized, mode, size, mtime) VALUES (?, ?, ?, ?);",
                    (filename_normalized, *edited_stat),
                )

            if info_json is not None:
                c.execute(
                    "INSERT OR REPLACE INTO info(uuid, json_info) VALUES (?, ?);",
                    (uuid, info_json),
                )

            if exif_json is not None:
                c.execute(
                    "INSERT OR REPLACE INTO exifdata(filepath_normalized, json_exifdata) VALUES (?, ?);",
                    (filename_normalized, exif_json),
                )
            conn.commit()
        except Error as e:
            logging.warning(e)

    def close(self):
        """close the database connection"""
        try:
            self._conn.close()
        except Error as e:
            logging.warning(e)

    def _set_stat_for_file(self, table, filename, stats):
        filename = self._normalize_filepath_relative(filename)
        if len(stats) != 3:
            raise ValueError(f"expected 3 elements for stat, got {len(stats)}")

        conn = self._conn
        c = conn.cursor()
        c.execute(
            f"INSERT OR REPLACE INTO {table}(filepath_normalized, mode, size, mtime) VALUES (?, ?, ?, ?);",
            (filename, *stats),
        )
        conn.commit()

    def _get_stat_for_file(self, table, filename):
        filename = self._normalize_filepath_relative(filename)
        conn = self._conn
        c = conn.cursor()
        c.execute(
            f"SELECT mode, size, mtime FROM {table} WHERE filepath_normalized = ?",
            (filename,),
        )
        results = c.fetchone()
        if results:
            stats = results[:3]
            mtime = int(stats[2]) if stats[2] is not None else None
            stats = (stats[0], stats[1], mtime)
        else:
            stats = (None, None, None)

        return stats

    def _open_export_db(self, dbfile):
        """open export database and return a db connection
        if dbfile does not exist, will create and initialize the database
        returns: connection to the database
        """

        if not os.path.isfile(dbfile):
            conn = self._get_db_connection(dbfile)
            if not conn:
                raise Exception("Error getting connection to database {dbfile}")
            self._create_db_tables(conn)
            self.was_created = True
            self.was_upgraded = ()
        else:
            conn = self._get_db_connection(dbfile)
            self.was_created = False
            version_info = self._get_database_version(conn)
            if version_info[1] < OSXPHOTOS_EXPORTDB_VERSION:
                self._create_db_tables(conn)
                if version_info[1] < OSXPHOTOS_EXPORTDB_VERSION_MIGRATE_FILEPATH:
                    self._migrate_normalized_filepath(conn)
                self.was_upgraded = (version_info[1], OSXPHOTOS_EXPORTDB_VERSION)
            else:
                self.was_upgraded = ()
        self.version = OSXPHOTOS_EXPORTDB_VERSION
        return conn

    def _get_db_connection(self, dbfile):
        """return db connection to dbname"""
        try:
            conn = sqlite3.connect(dbfile)
        except Error as e:
            logging.warning(e)
            conn = None

        return conn

    def _get_database_version(self, conn):
        """return tuple of (osxphotos, exportdb) versions for database connection conn"""
        version_info = conn.execute(
            "SELECT osxphotos, exportdb, max(id) FROM version"
        ).fetchone()
        return (version_info[0], version_info[1])

    def _create_db_tables(self, conn):
        """create (if not already created) the necessary db tables for the export database
        conn: sqlite3 db connection
        """
        sql_commands = {
            "sql_version_table": """ CREATE TABLE IF NOT EXISTS version (
                                id INTEGER PRIMARY KEY,
                                osxphotos TEXT,
                                exportdb TEXT 
                                ); """,
            "sql_about_table": """ CREATE TABLE IF NOT EXISTS about (
                                id INTEGER PRIMARY KEY,
                                about TEXT
                                );""",
            "sql_files_table": """ CREATE TABLE IF NOT EXISTS files (
                              id INTEGER PRIMARY KEY,
                              filepath TEXT NOT NULL,
                              filepath_normalized TEXT NOT NULL,
                              uuid TEXT,
                              orig_mode INTEGER,
                              orig_size INTEGER,
                              orig_mtime REAL,
                              exif_mode INTEGER,
                              exif_size INTEGER,
                              exif_mtime REAL
                              ); """,
            "sql_files_table_migrate": """ CREATE TABLE IF NOT EXISTS files_migrate (
                              id INTEGER PRIMARY KEY,
                              filepath TEXT NOT NULL,
                              filepath_normalized TEXT NOT NULL,
                              uuid TEXT,
                              orig_mode INTEGER,
                              orig_size INTEGER,
                              orig_mtime REAL,
                              exif_mode INTEGER,
                              exif_size INTEGER,
                              exif_mtime REAL,
                              UNIQUE(filepath_normalized)
                              ); """,
            "sql_files_migrate": """ INSERT INTO files_migrate SELECT * FROM files;""",
            "sql_files_drop_tables": """ DROP TABLE files;""",
            "sql_files_alter": """ ALTER TABLE files_migrate RENAME TO files;""",
            "sql_runs_table": """ CREATE TABLE IF NOT EXISTS runs (
                             id INTEGER PRIMARY KEY,
                             datetime TEXT,
                             python_path TEXT,
                             script_name TEXT,
                             args TEXT,
                             cwd TEXT 
                             ); """,
            "sql_info_table": """ CREATE TABLE IF NOT EXISTS info (
                             id INTEGER PRIMARY KEY,
                             uuid text NOT NULL,
                             json_info JSON 
                             ); """,
            "sql_exifdata_table": """ CREATE TABLE IF NOT EXISTS exifdata (
                             id INTEGER PRIMARY KEY,
                             filepath_normalized TEXT NOT NULL,
                             json_exifdata JSON 
                             ); """,
            "sql_edited_table": """ CREATE TABLE IF NOT EXISTS edited (
                              id INTEGER PRIMARY KEY,
                              filepath_normalized TEXT NOT NULL,
                              mode INTEGER,
                              size INTEGER,
                              mtime REAL
                              ); """,
            "sql_converted_table": """ CREATE TABLE IF NOT EXISTS converted (
                              id INTEGER PRIMARY KEY,
                              filepath_normalized TEXT NOT NULL,
                              mode INTEGER,
                              size INTEGER,
                              mtime REAL
                              ); """,
            "sql_sidecar_table": """ CREATE TABLE IF NOT EXISTS sidecar (
                              id INTEGER PRIMARY KEY,
                              filepath_normalized TEXT NOT NULL,
                              sidecar_data TEXT,
                              mode INTEGER,
                              size INTEGER,
                              mtime REAL
                              ); """,
            "sql_detected_text_table": """ CREATE TABLE IF NOT EXISTS detected_text (
                              id INTEGER PRIMARY KEY,
                              uuid TEXT NOT NULL,
                              text_data JSON
                              ); """,
            "sql_files_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_files_filepath_normalized on files (filepath_normalized); """,
            "sql_info_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_info_uuid on info (uuid); """,
            "sql_exifdata_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_exifdata_filename on exifdata (filepath_normalized); """,
            "sql_edited_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_edited_filename on edited (filepath_normalized);""",
            "sql_converted_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_converted_filename on converted (filepath_normalized);""",
            "sql_sidecar_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_sidecar_filename on sidecar (filepath_normalized);""",
            "sql_detected_text_idx": """ CREATE UNIQUE INDEX IF NOT EXISTS idx_detected_text on detected_text (uuid);""",
        }
        try:
            c = conn.cursor()
            for cmd in sql_commands.values():
                c.execute(cmd)
            c.execute(
                "INSERT INTO version(osxphotos, exportdb) VALUES (?, ?);",
                (__version__, OSXPHOTOS_EXPORTDB_VERSION),
            )
            c.execute("INSERT INTO about(about) VALUES (?);", (OSXPHOTOS_ABOUT_STRING,))
            conn.commit()
        except Error as e:
            logging.warning(e)

    def __del__(self):
        """ensure the database connection is closed"""
        try:
            self._conn.close()
        except:
            pass

    def _insert_run_info(self):
        dt = datetime.datetime.utcnow().isoformat()
        python_path = sys.executable
        cmd = sys.argv[0]
        args = " ".join(sys.argv[1:]) if len(sys.argv) > 1 else ""
        cwd = os.getcwd()
        conn = self._conn
        try:
            c = conn.cursor()
            c.execute(
                "INSERT INTO runs (datetime, python_path, script_name, args, cwd) VALUES (?, ?, ?, ?, ?)",
                (dt, python_path, cmd, args, cwd),
            )

            conn.commit()
        except Error as e:
            logging.warning(e)

    def _normalize_filepath(self, filepath: Union[str, pathlib.Path]) -> str:
        """normalize filepath for unicode, lower case"""
        return normalize_fs_path(str(filepath)).lower()

    def _normalize_filepath_relative(self, filepath: Union[str, pathlib.Path]) -> str:
        """normalize filepath for unicode, relative path (to export dir), lower case"""
        filepath = str(pathlib.Path(filepath).relative_to(self._path))
        return normalize_fs_path(str(filepath)).lower()

    def _migrate_normalized_filepath(self, conn):
        """Fix all filepath_normalized columns for unicode normalization"""
        # Prior to database version 4.3, filepath_normalized was not normalized for unicode
        c = conn.cursor()
        for table in ["converted", "edited", "exifdata", "files", "sidecar"]:
            old_values = c.execute(
                f"SELECT filepath_normalized, id FROM {table}"
            ).fetchall()
            new_values = [
                (self._normalize_filepath(filepath_normalized), id_)
                for filepath_normalized, id_ in old_values
            ]
            c.executemany(
                f"UPDATE {table} SET filepath_normalized=? WHERE id=?", new_values
            )
        conn.commit()


class ExportDBInMemory(ExportDB):
    """In memory version of ExportDB
    Copies the on-disk database into memory so it may be operated on without
    modifying the on-disk version
    """

    def __init__(self, dbfile, export_dir):
        self._dbfile = dbfile or f"./{OSXPHOTOS_EXPORT_DB}"
        # export_dir is required as all files referenced by get_/set_uuid_for_file will be converted to
        # relative paths to this path
        # this allows the entire export tree to be moved to a new disk/location
        # whilst preserving the UUID to filename mapping
        self._path = export_dir
        self._conn = self._open_export_db(self._dbfile)
        self._insert_run_info()

    def _open_export_db(self, dbfile):
        """open export database and return a db connection
        returns: connection to the database
        """
        if not os.path.isfile(dbfile):
            conn = self._get_db_connection()
            if not conn:
                raise Exception("Error getting connection to in-memory database")
            self._create_db_tables(conn)
            self.was_created = True
            self.was_upgraded = ()
        else:
            try:
                conn = sqlite3.connect(dbfile)
            except Error as e:
                logging.warning(e)
                raise e

            tempfile = StringIO()
            for line in conn.iterdump():
                tempfile.write("%s\n" % line)
            conn.close()
            tempfile.seek(0)

            # Create a database in memory and import from tempfile
            conn = sqlite3.connect(":memory:")
            conn.cursor().executescript(tempfile.read())
            conn.commit()
            self.was_created = False
            _, exportdb_ver = self._get_database_version(conn)
            if exportdb_ver < OSXPHOTOS_EXPORTDB_VERSION:
                self._create_db_tables(conn)
                self.was_upgraded = (exportdb_ver, OSXPHOTOS_EXPORTDB_VERSION)
            else:
                self.was_upgraded = ()
        self.version = OSXPHOTOS_EXPORTDB_VERSION
        return conn

    def _get_db_connection(self):
        """return db connection to in memory database"""
        try:
            conn = sqlite3.connect(":memory:")
        except Error as e:
            logging.warning(e)
            conn = None

        return conn
