"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
        if (props.api instanceof restapi_1.RestApiBase) {
            props.api._attachDeployment(this);
        }
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Quoting from CloudFormation's docs:
     *
     *   If you create an AWS::ApiGateway::RestApi resource and its methods (using
     *   AWS::ApiGateway::Method) in the same template as your deployment, the
     *   deployment must depend on the RestApi's methods. To create a dependency,
     *   add a DependsOn attribute to the deployment. If you don't, AWS
     *   CloudFormation creates the deployment right after it creates the RestApi
     *   resource that doesn't contain any methods, and AWS CloudFormation
     *   encounters the following error: The REST API doesn't contain any methods.
     *
     * @param method The method to add as a dependency of the deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
     * @see https://github.com/aws/aws-cdk/pull/6165
     * @internal
     */
    _addMethodDependency(method) {
        // adding a dependency between the constructs using `node.addDependency()`
        // will create additional dependencies between `AWS::ApiGateway::Deployment`
        // and the `AWS::Lambda::Permission` resources (children under Method),
        // causing cyclic dependency errors. Hence, falling back to declaring
        // dependencies between the underlying CfnResources.
        this.node.addDependency(method.node.defaultChild);
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = this.stack.getLogicalId(this);
        this.overrideLogicalId(core_1.Lazy.stringValue({ produce: () => this.calculateLogicalId() }));
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    calculateLogicalId() {
        const hash = [...this.hashComponents];
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            hash.push(this.stack.resolve(cfnRestApiCF));
        }
        let lid = this.originalLogicalId;
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (hash.length > 0) {
            const md5 = crypto.createHash('md5');
            hash.map(x => this.stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            lid += md5.digest('hex');
        }
        return lid;
    }
}
//# sourceMappingURL=data:application/json;base64,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