"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNhZ2UtcGxhbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzYWdlLXBsYW4udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQWlFO0FBRWpFLGlFQUF1RTtBQUl2RSxpQ0FBeUM7QUFvQnpDOztHQUVHO0FBQ0gsSUFBWSxNQUlYO0FBSkQsV0FBWSxNQUFNO0lBQ2hCLHFCQUFXLENBQUE7SUFDWCx1QkFBYSxDQUFBO0lBQ2IseUJBQWUsQ0FBQTtBQUNqQixDQUFDLEVBSlcsTUFBTSxHQUFOLGNBQU0sS0FBTixjQUFNLFFBSWpCO0FBK0dELE1BQWEsU0FBVSxTQUFRLGVBQVE7SUFRckMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUF3QixFQUFHO1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFIRixjQUFTLEdBQUcsSUFBSSxLQUFLLEVBQXdCLENBQUM7UUFJN0QsSUFBSSxRQUFzQixDQUFDO1FBRTNCLFFBQVEsR0FBRyxJQUFJLG1DQUFZLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM1QyxTQUFTLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDO1lBQ2pGLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUM7WUFDOUIsUUFBUSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQztZQUM3QyxhQUFhLEVBQUUsS0FBSyxDQUFDLElBQUk7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUVoRCxJQUFJLENBQUMsV0FBVyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFFaEMsa0JBQWtCO1FBQ2xCLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNoQixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUM5QjtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksU0FBUyxDQUFDLE1BQWU7UUFDOUIsTUFBTSxNQUFNLEdBQUcsc0JBQXNCLENBQUM7UUFFdEMscUhBQXFIO1FBQ3JILE1BQU0sRUFBRSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLE1BQU0sSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFFekYsSUFBSSxzQ0FBZSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDNUIsS0FBSyxFQUFFLE1BQU0sQ0FBQyxLQUFLO1lBQ25CLE9BQU8seUJBQTBCO1lBQ2pDLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztTQUM5QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksV0FBVyxDQUFDLFFBQThCO1FBQy9DLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7O09BR0c7SUFDSyxlQUFlLENBQUMsU0FBNkM7UUFDbkUsSUFBSSxTQUFTLElBQUksU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDckMsTUFBTSxNQUFNLEdBQW9DLEVBQUUsQ0FBQztZQUNuRCxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBOEIsRUFBRSxFQUFFO2dCQUNuRCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUMxQyxDQUFDLENBQUMsQ0FBQztZQUNILE9BQU8sTUFBTSxDQUFDO1NBQ2Y7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU8sV0FBVyxDQUFDLFFBQThCO1FBQ2hELE1BQU0sS0FBSyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDL0UsTUFBTSxLQUFLLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDNUUsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNqRSxPQUFPO1lBQ0wsS0FBSztZQUNMLEtBQUs7WUFDTCxRQUFRO1NBQ1QsQ0FBQztJQUNKLENBQUM7SUFFTyxXQUFXLENBQUMsS0FBcUI7UUFDdkMsSUFBSSxLQUFLLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUM3QixPQUFPLFNBQVMsQ0FBQztTQUNsQjthQUFNO1lBQ0wsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUMxRCxzQkFBZSxDQUFDLEtBQUssRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQy9DLE1BQU0sR0FBRyxHQUFHO2dCQUNWLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDaEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNwRCxNQUFNLEVBQUUsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDckQsQ0FBQztZQUNGLE9BQU8sR0FBRyxDQUFDO1NBQ1o7SUFDSCxDQUFDO0lBRU8sY0FBYyxDQUFDLEtBQW1DO1FBQ3hELElBQUksR0FBa0QsQ0FBQztRQUN2RCxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDdkIsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztZQUNwQyxzQkFBZSxDQUFDLFVBQVUsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQ3BELE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDbEMsc0JBQWUsQ0FBQyxTQUFTLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUVsRCxHQUFHLEdBQUc7Z0JBQ0osVUFBVSxFQUFFLFVBQVU7Z0JBQ3RCLFNBQVMsRUFBRSxTQUFTO2FBQ3JCLENBQUM7U0FDSDtRQUNELE9BQU8sR0FBSSxDQUFDO0lBQ2QsQ0FBQztJQUVPLHVCQUF1QixDQUFDLGlCQUF5QztRQUN2RSxNQUFNLEdBQUcsR0FBdUUsRUFBRSxDQUFDO1FBQ25GLElBQUksaUJBQWlCLElBQUksaUJBQWlCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyRCxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUEwQixFQUFFLEVBQUU7Z0JBQ3ZELE1BQU0sTUFBTSxHQUFXLEtBQUssQ0FBQyxNQUFNLENBQUM7Z0JBQ3BDLDBFQUEwRTtnQkFDMUUsTUFBTSxRQUFRLEdBQUcsR0FBRyxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksSUFBSSxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUM7Z0JBQ2hFLEdBQUcsQ0FBQyxRQUFRLENBQUMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUN0RCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUE3SEQsOEJBNkhDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBMYXp5LCBSZXNvdXJjZSwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IElBcGlLZXkgfSBmcm9tICcuL2FwaS1rZXknO1xuaW1wb3J0IHsgQ2ZuVXNhZ2VQbGFuLCBDZm5Vc2FnZVBsYW5LZXkgfSBmcm9tICcuL2FwaWdhdGV3YXkuZ2VuZXJhdGVkJztcbmltcG9ydCB7IE1ldGhvZCB9IGZyb20gJy4vbWV0aG9kJztcbmltcG9ydCB7IElSZXN0QXBpIH0gZnJvbSAnLi9yZXN0YXBpJztcbmltcG9ydCB7IFN0YWdlIH0gZnJvbSAnLi9zdGFnZSc7XG5pbXBvcnQgeyB2YWxpZGF0ZUludGVnZXIgfSBmcm9tICcuL3V0aWwnO1xuXG4vKipcbiAqIENvbnRhaW5lciBmb3IgZGVmaW5pbmcgdGhyb3R0bGluZyBwYXJhbWV0ZXJzIHRvIEFQSSBzdGFnZXMgb3IgbWV0aG9kcy5cbiAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1yZXF1ZXN0LXRocm90dGxpbmcuaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRocm90dGxlU2V0dGluZ3Mge1xuICAvKipcbiAgICogVGhlIEFQSSByZXF1ZXN0IHN0ZWFkeS1zdGF0ZSByYXRlIGxpbWl0IChhdmVyYWdlIHJlcXVlc3RzIHBlciBzZWNvbmQgb3ZlciBhbiBleHRlbmRlZCBwZXJpb2Qgb2YgdGltZSlcbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcmF0ZUxpbWl0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBBUEkgcmVxdWVzdCByYXRlIGxpbWl0IG92ZXIgYSB0aW1lIHJhbmdpbmcgZnJvbSBvbmUgdG8gYSBmZXcgc2Vjb25kcy5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYnVyc3RMaW1pdD86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBUaW1lIHBlcmlvZCBmb3Igd2hpY2ggcXVvdGEgc2V0dGluZ3MgYXBwbHkuXG4gKi9cbmV4cG9ydCBlbnVtIFBlcmlvZCB7XG4gIERBWSA9ICdEQVknLFxuICBXRUVLID0gJ1dFRUsnLFxuICBNT05USCA9ICdNT05USCdcbn1cblxuLyoqXG4gKiBTcGVjaWZpZXMgdGhlIG1heGltdW0gbnVtYmVyIG9mIHJlcXVlc3RzIHRoYXQgY2xpZW50cyBjYW4gbWFrZSB0byBBUEkgR2F0ZXdheSBBUElzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFF1b3RhU2V0dGluZ3Mge1xuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIHJlcXVlc3RzIHRoYXQgdXNlcnMgY2FuIG1ha2Ugd2l0aGluIHRoZSBzcGVjaWZpZWQgdGltZSBwZXJpb2QuXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGxpbWl0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBGb3IgdGhlIGluaXRpYWwgdGltZSBwZXJpb2QsIHRoZSBudW1iZXIgb2YgcmVxdWVzdHMgdG8gc3VidHJhY3QgZnJvbSB0aGUgc3BlY2lmaWVkIGxpbWl0LlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBvZmZzZXQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIHBlcmlvZCBmb3Igd2hpY2ggdGhlIG1heGltdW0gbGltaXQgb2YgcmVxdWVzdHMgYXBwbGllcy5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcGVyaW9kPzogUGVyaW9kO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgcGVyLW1ldGhvZCB0aHJvdHRsaW5nIGZvciBhIHJlc291cmNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRocm90dGxpbmdQZXJNZXRob2Qge1xuICAvKipcbiAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV1cbiAgICogVGhlIG1ldGhvZCBmb3Igd2hpY2ggeW91IHNwZWNpZnkgdGhlIHRocm90dGxpbmcgc2V0dGluZ3MuXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IG1ldGhvZDogTWV0aG9kO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIG92ZXJhbGwgcmVxdWVzdCByYXRlIChhdmVyYWdlIHJlcXVlc3RzIHBlciBzZWNvbmQpIGFuZCBidXJzdCBjYXBhY2l0eS5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgdGhyb3R0bGU6IFRocm90dGxlU2V0dGluZ3M7XG59XG5cbi8qKlxuICogVHlwZSBvZiBVc2FnZSBQbGFuIEtleS4gQ3VycmVudGx5IHRoZSBvbmx5IHN1cHBvcnRlZCB0eXBlIGlzICdBcGlLZXknXG4gKi9cbmNvbnN0IGVudW0gVXNhZ2VQbGFuS2V5VHlwZSB7XG4gIEFQSV9LRVkgPSAnQVBJX0tFWSdcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBBUEkgc3RhZ2VzIHRoYXQgYSB1c2FnZSBwbGFuIGFwcGxpZXMgdG8uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVXNhZ2VQbGFuUGVyQXBpU3RhZ2Uge1xuXG4gIC8qKlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBhcGk/OiBJUmVzdEFwaTtcblxuICAvKipcbiAgICpcbiAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV1cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgc3RhZ2U/OiBTdGFnZTtcblxuICAvKipcbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgdGhyb3R0bGU/OiBUaHJvdHRsaW5nUGVyTWV0aG9kW107XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVXNhZ2VQbGFuUHJvcHMge1xuICAvKipcbiAgICogQVBJIFN0YWdlcyB0byBiZSBhc3NvY2lhdGVkIHdpdGggdGhlIHVzYWdlIHBsYW4uXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFwaVN0YWdlcz86IFVzYWdlUGxhblBlckFwaVN0YWdlW107XG5cbiAgLyoqXG4gICAqIFJlcHJlc2VudHMgdXNhZ2UgcGxhbiBwdXJwb3NlLlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHJlcXVlc3RzIGNsaWVudHMgY2FuIG1ha2UgaW4gYSBnaXZlbiB0aW1lIHBlcmlvZC5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcXVvdGE/OiBRdW90YVNldHRpbmdzO1xuXG4gIC8qKlxuICAgKiBPdmVyYWxsIHRocm90dGxlIHNldHRpbmdzIGZvciB0aGUgQVBJLlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSB0aHJvdHRsZT86IFRocm90dGxlU2V0dGluZ3M7XG5cbiAgLyoqXG4gICAqIE5hbWUgZm9yIHRoaXMgdXNhZ2UgcGxhbi5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogQXBpS2V5IHRvIGJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgdXNhZ2UgcGxhbi5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYXBpS2V5PzogSUFwaUtleTtcbn1cblxuZXhwb3J0IGNsYXNzIFVzYWdlUGxhbiBleHRlbmRzIFJlc291cmNlIHtcbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB1c2FnZVBsYW5JZDogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgYXBpU3RhZ2VzID0gbmV3IEFycmF5PFVzYWdlUGxhblBlckFwaVN0YWdlPigpO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBVc2FnZVBsYW5Qcm9wcyA9IHsgfSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgbGV0IHJlc291cmNlOiBDZm5Vc2FnZVBsYW47XG5cbiAgICByZXNvdXJjZSA9IG5ldyBDZm5Vc2FnZVBsYW4odGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYXBpU3RhZ2VzOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5yZW5kZXJBcGlTdGFnZXModGhpcy5hcGlTdGFnZXMpIH0pLFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgcXVvdGE6IHRoaXMucmVuZGVyUXVvdGEocHJvcHMpLFxuICAgICAgdGhyb3R0bGU6IHRoaXMucmVuZGVyVGhyb3R0bGUocHJvcHMudGhyb3R0bGUpLFxuICAgICAgdXNhZ2VQbGFuTmFtZTogcHJvcHMubmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXBpU3RhZ2VzLnB1c2goLi4uKHByb3BzLmFwaVN0YWdlcyB8fCBbXSkpO1xuXG4gICAgdGhpcy51c2FnZVBsYW5JZCA9IHJlc291cmNlLnJlZjtcblxuICAgIC8vIEFkZCBBcGlLZXkgd2hlblxuICAgIGlmIChwcm9wcy5hcGlLZXkpIHtcbiAgICAgIHRoaXMuYWRkQXBpS2V5KHByb3BzLmFwaUtleSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gQXBpS2V5LlxuICAgKlxuICAgKiBAcGFyYW0gYXBpS2V5XG4gICAqL1xuICBwdWJsaWMgYWRkQXBpS2V5KGFwaUtleTogSUFwaUtleSk6IHZvaWQge1xuICAgIGNvbnN0IHByZWZpeCA9ICdVc2FnZVBsYW5LZXlSZXNvdXJjZSc7XG5cbiAgICAvLyBQb3N0Zml4aW5nIGFwaWtleSBpZCBvbmx5IGZyb20gdGhlIDJuZCBjaGlsZCwgdG8ga2VlcCBwaHlzaWNhbElkcyBvZiBVc2FnZVBsYW5LZXkgZm9yIGV4aXN0aW5nIENESyBhcHBzIHVubW9kaWZlZC5cbiAgICBjb25zdCBpZCA9IHRoaXMubm9kZS50cnlGaW5kQ2hpbGQocHJlZml4KSA/IGAke3ByZWZpeH06JHthcGlLZXkubm9kZS51bmlxdWVJZH1gIDogcHJlZml4O1xuXG4gICAgbmV3IENmblVzYWdlUGxhbktleSh0aGlzLCBpZCwge1xuICAgICAga2V5SWQ6IGFwaUtleS5rZXlJZCxcbiAgICAgIGtleVR5cGU6IFVzYWdlUGxhbktleVR5cGUuQVBJX0tFWSxcbiAgICAgIHVzYWdlUGxhbklkOiB0aGlzLnVzYWdlUGxhbklkLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gYXBpU3RhZ2UuXG4gICAqIEBwYXJhbSBhcGlTdGFnZVxuICAgKi9cbiAgcHVibGljIGFkZEFwaVN0YWdlKGFwaVN0YWdlOiBVc2FnZVBsYW5QZXJBcGlTdGFnZSkge1xuICAgIHRoaXMuYXBpU3RhZ2VzLnB1c2goYXBpU3RhZ2UpO1xuICB9XG5cbiAgLyoqXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSByZW5kZXJBcGlTdGFnZXMoYXBpU3RhZ2VzOiBVc2FnZVBsYW5QZXJBcGlTdGFnZVtdIHwgdW5kZWZpbmVkKTogQ2ZuVXNhZ2VQbGFuLkFwaVN0YWdlUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKGFwaVN0YWdlcyAmJiBhcGlTdGFnZXMubGVuZ3RoID4gMCkge1xuICAgICAgY29uc3Qgc3RhZ2VzOiBDZm5Vc2FnZVBsYW4uQXBpU3RhZ2VQcm9wZXJ0eVtdID0gW107XG4gICAgICBhcGlTdGFnZXMuZm9yRWFjaCgoYXBpU3RhZ2U6IFVzYWdlUGxhblBlckFwaVN0YWdlKSA9PiB7XG4gICAgICAgIHN0YWdlcy5wdXNoKHRoaXMuY3JlYXRlU3RhZ2UoYXBpU3RhZ2UpKTtcbiAgICAgIH0pO1xuICAgICAgcmV0dXJuIHN0YWdlcztcbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlU3RhZ2UoYXBpU3RhZ2U6IFVzYWdlUGxhblBlckFwaVN0YWdlKTogQ2ZuVXNhZ2VQbGFuLkFwaVN0YWdlUHJvcGVydHkge1xuICAgIGNvbnN0IHN0YWdlID0gYXBpU3RhZ2Uuc3RhZ2UgPyBhcGlTdGFnZS5zdGFnZS5zdGFnZU5hbWUudG9TdHJpbmcoKSA6IHVuZGVmaW5lZDtcbiAgICBjb25zdCBhcGlJZCA9IGFwaVN0YWdlLnN0YWdlID8gYXBpU3RhZ2Uuc3RhZ2UucmVzdEFwaS5yZXN0QXBpSWQgOiB1bmRlZmluZWQ7XG4gICAgY29uc3QgdGhyb3R0bGUgPSB0aGlzLnJlbmRlclRocm90dGxlUGVyTWV0aG9kKGFwaVN0YWdlLnRocm90dGxlKTtcbiAgICByZXR1cm4ge1xuICAgICAgYXBpSWQsXG4gICAgICBzdGFnZSxcbiAgICAgIHRocm90dGxlLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclF1b3RhKHByb3BzOiBVc2FnZVBsYW5Qcm9wcykge1xuICAgIGlmIChwcm9wcy5xdW90YSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBsaW1pdCA9IHByb3BzLnF1b3RhID8gcHJvcHMucXVvdGEubGltaXQgOiB1bmRlZmluZWQ7XG4gICAgICB2YWxpZGF0ZUludGVnZXIobGltaXQsICdUaHJvdHRsZSBxdW90YSBsaW1pdCcpO1xuICAgICAgY29uc3QgcmV0ID0ge1xuICAgICAgICBsaW1pdDogbGltaXQgPyBsaW1pdCA6IHVuZGVmaW5lZCxcbiAgICAgICAgb2Zmc2V0OiBwcm9wcy5xdW90YSA/IHByb3BzLnF1b3RhLm9mZnNldCA6IHVuZGVmaW5lZCxcbiAgICAgICAgcGVyaW9kOiBwcm9wcy5xdW90YSA/IHByb3BzLnF1b3RhLnBlcmlvZCA6IHVuZGVmaW5lZCxcbiAgICAgIH07XG4gICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVGhyb3R0bGUocHJvcHM6IFRocm90dGxlU2V0dGluZ3MgfCB1bmRlZmluZWQpOiAoQ2ZuVXNhZ2VQbGFuLlRocm90dGxlU2V0dGluZ3NQcm9wZXJ0eSB8IFRva2VuKSB7XG4gICAgbGV0IHJldDogQ2ZuVXNhZ2VQbGFuLlRocm90dGxlU2V0dGluZ3NQcm9wZXJ0eSB8IFRva2VuO1xuICAgIGlmIChwcm9wcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBjb25zdCBidXJzdExpbWl0ID0gcHJvcHMuYnVyc3RMaW1pdDtcbiAgICAgIHZhbGlkYXRlSW50ZWdlcihidXJzdExpbWl0LCAnVGhyb3R0bGUgYnVyc3QgbGltaXQnKTtcbiAgICAgIGNvbnN0IHJhdGVMaW1pdCA9IHByb3BzLnJhdGVMaW1pdDtcbiAgICAgIHZhbGlkYXRlSW50ZWdlcihyYXRlTGltaXQsICdUaHJvdHRsZSByYXRlIGxpbWl0Jyk7XG5cbiAgICAgIHJldCA9IHtcbiAgICAgICAgYnVyc3RMaW1pdDogYnVyc3RMaW1pdCxcbiAgICAgICAgcmF0ZUxpbWl0OiByYXRlTGltaXQsXG4gICAgICB9O1xuICAgIH1cbiAgICByZXR1cm4gcmV0ITtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVGhyb3R0bGVQZXJNZXRob2QodGhyb3R0bGVQZXJNZXRob2Q/OiBUaHJvdHRsaW5nUGVyTWV0aG9kW10pIHtcbiAgICBjb25zdCByZXQ6IHsgW2tleTogc3RyaW5nXTogKENmblVzYWdlUGxhbi5UaHJvdHRsZVNldHRpbmdzUHJvcGVydHkgfCBUb2tlbikgfSA9IHt9O1xuICAgIGlmICh0aHJvdHRsZVBlck1ldGhvZCAmJiB0aHJvdHRsZVBlck1ldGhvZC5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdHRsZVBlck1ldGhvZC5mb3JFYWNoKCh2YWx1ZTogVGhyb3R0bGluZ1Blck1ldGhvZCkgPT4ge1xuICAgICAgICBjb25zdCBtZXRob2Q6IE1ldGhvZCA9IHZhbHVlLm1ldGhvZDtcbiAgICAgICAgLy8gdGhpcyBtZXRob2RJZCBpcyByZXNvdXJjZSBwYXRoIGFuZCBtZXRob2QgZm9yIGV4YW1wbGUgL0dFVCBvciAvcGV0cy9HRVRcbiAgICAgICAgY29uc3QgbWV0aG9kSWQgPSBgJHttZXRob2QucmVzb3VyY2UucGF0aH0vJHttZXRob2QuaHR0cE1ldGhvZH1gO1xuICAgICAgICByZXRbbWV0aG9kSWRdID0gdGhpcy5yZW5kZXJUaHJvdHRsZSh2YWx1ZS50aHJvdHRsZSk7XG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxufVxuIl19