from __future__ import annotations

import json
import os
import shutil
import tempfile
import typing
from typing import Any, Optional, Union, List, Dict

import requests
from requests.models import Response
from supermarket_connector import utils
from supermarket_connector.models.category import Category
from supermarket_connector.models.image import Image
from supermarket_connector.models.product import Product


class Client:
    BASE_URL = "https://pls-sprmrkt-mw.prd.vdc1.plus.nl/api/v3/"
    DEFAULT_HEADERS = {
        "Cache-Control": "no-cache",
        "Content-Type": "application/json",
        "Accept-Encoding": "gzip, deflate, br",
        "Accept": "*/*",
        "Connection": "keep-alive",
    }
    TEMP_DIR = os.path.join(tempfile.gettempdir(), "Supermarket-Connector", "Debug", "PLUS")
    AUTH_COOKIE_KEY = "reese84"

    access_token: Optional[str] = None

    def request(
        self,
        method: str,
        end_point: str,
        headers: Optional[Dict[str, Any]] = None,
        params: Optional[Dict[str, Any]] = None,
        request_data: Optional[Dict[str, Any]] = None,
        timeout: int = 10,
        authorized: bool = True,
        json_: bool = True,
        debug_key: Optional[str] = None,
    ) -> Union[str, List[Any], Dict[Any, Any]]:
        if headers is None:
            headers = {}

        if params is None:
            params = {}

        headers.update(self.DEFAULT_HEADERS)
        counter_tries = 0
        s = requests.Session()

        cookies = {}

        if authorized:
            if self.access_token is None:
                raise Exception("Need token to make authorized requests")
            cookies[self.AUTH_COOKIE_KEY] = self.access_token

        s.cookies.update(cookies)

        while True:
            try:
                counter_tries += 1
                if not request_data is None:
                    response: Response = s.request(method, f"{self.BASE_URL}{end_point}", params=params, headers=headers, data=json.dumps(request_data), timeout=timeout)
                else:
                    response: Response = s.request(method, f"{self.BASE_URL}{end_point}", params=params, headers=headers, timeout=timeout)

                if not response.ok:
                    print(f"Connection error: {response.status_code} try: {counter_tries} page: {end_point}", end="\r")
                    if response.status_code == 401:
                        self.login()
                        continue
                    if counter_tries > 20:
                        raise Exception(f"Connection error: {response.status_code}")
                        break
                    continue

            except Exception:
                continue
            else:
                break

        if json_:
            try:
                response_json: Union[List[Any], Dict[Any, Any]] = response.json()

                if self.debug:
                    if self.debug_fn is None:
                        print("To debug response also give a filename")
                    elif not self.debug_fn.endswith(".json"):
                        print("Currently only json format is supported")
                    else:
                        debug_path = os.path.join(self.TEMP_DIR, self.debug_fn)
                        debug_path_temp = os.path.join(self.TEMP_DIR, self.debug_fn.replace(".json", "_old.json"))
                        if os.path.isfile(debug_path):
                            with open(debug_path, "r") as f:
                                try:
                                    data: Dict[str, Any] = json.load(f)
                                    shutil.copyfile(debug_path, debug_path_temp)
                                except ValueError:
                                    data = {}
                                    pass
                        else:
                            data = {}

                        if not debug_key in data.keys() and not debug_key is None:
                            data[debug_key] = {}

                        if not end_point in data.keys() and debug_key is None:
                            data[end_point] = {}

                        if not debug_key is None:
                            key = debug_key
                        else:
                            key = end_point

                        with open(debug_path, "w") as f:
                            if isinstance(response_json, list):
                                data[key] = utils.process_type(response_json, data[key], self.debug_value)
                                json.dump(data, f)
                            else:
                                data[key] = utils.type_def_dict(response_json, data[key], self.debug_value)
                                json.dump(data, f)

                return response_json
            except ValueError:
                raise ValueError("Response is not in JSON format")
        else:
            return response.text

    def login(self):
        cur_dir = os.path.dirname(os.path.realpath(__file__))
        with open(os.path.join(cur_dir, "auth_data.json"), "r") as f:
            response: Response = requests.request("POST", "https://pls-sprmrkt-mw.prd.vdc1.plus.nl/Due-away-are-Fight-Banq-Though-theere-Prayers-On?d=pls-sprmrkt-mw.prd.vdc1.plus.nl", data=json.dumps(json.load(f)))
            if not response.ok:
                raise Exception("Login failed")

            self.access_token = response.json().get("token")
            return

    def __init__(
        self,
        debug: bool = False,
        debug_fn: Optional[str] = None,
        debug_value: bool = True,
    ) -> None:
        if not os.path.isdir(self.TEMP_DIR):
            os.makedirs(self.TEMP_DIR)

        self.products = self.Products(self)
        self.categories = self.Categories(self)
        self.images = self.Images(self)
        self.debug = debug
        self.debug_fn = debug_fn
        self.debug_value = debug_value

        self.login()

    class Categories:
        def __init__(self, client: Client) -> None:
            self.__client = client
            self.data: Dict[Union[int, str], Client.Category] = {}

        def list(self):
            response = self.__client.request("GET", "categorytree")

            if not isinstance(response, dict):
                raise ValueError("Reponse is not in right format")

            data: List[Dict[str, Any]] = response.get("categories", [])

            for elem in data:
                category = self.__client.Category(self.__client, data=elem)
                self.data[category.id] = category

                sub_categories = elem.get("children", [])

                def get_sub_categories(sub_categories: List[Dict[str, Any]]) -> None:
                    for sub_category in sub_categories:
                        sub_category_elem = self.__client.Category(self.__client, data=sub_category)
                        self.data[sub_category_elem.id] = sub_category_elem
                        get_sub_categories(sub_category.get("children", []))

                get_sub_categories(sub_categories)

            return self.data

        def get(self, id: Optional[int] = None, name: Optional[str] = None):
            if not id is None:
                if id in self.data.keys():
                    self.data[id]

                self.list()

                if id in self.data.keys():
                    self.data[id]

            elif not name is None:
                for category in self.data.values():
                    lookup = category.lookup(name=name)
                    if not lookup is None:
                        return lookup

                for category in self.list().values():
                    lookup = category.lookup(name=name)
                    if not lookup is None:
                        return lookup

            return None

    class Products:
        def __init__(self, client: Client) -> None:
            self.__client = client
            self.data: Dict[Union[int, str], Dict[int, Client.Product]] = {}

        @typing.overload
        def list(self) -> Dict[int, Client.Product]:
            ...

        @typing.overload
        def list(self, category: Client.Category) -> Dict[int, Client.Product]:
            ...

        def list(self, category: Optional[Client.Category] = None):
            category_id = 333333

            self.data[category_id] = {}

            page = 1
            total_pages = 1

            while True:
                response = self.__client.request("GET", f"navigation", params={"tn_cid": category_id, "tn_ps": 1000, "tn_p": page})

                if not isinstance(response, dict):
                    raise ValueError("Expected response to be dict")

                total_pages = response.get("properties", {}).get("nrofpages")

                print(f"{page + 1}/{total_pages}", end="\r")

                products = response.get("items", [])

                for product in products:
                    temp_ = self.__client.Product(self.__client, data=product)
                    if not temp_ is None:
                        if not temp_.id in self.data[category_id].keys():
                            self.data[category_id][temp_.id] = temp_

                if page == total_pages:
                    break

                page += 1

            return self.data[category_id]

    class Images:
        def __init__(self, client: Client) -> None:
            self.__client = client

        def process(self, data: List[Dict[str, Any]]):
            temp: List[Client.Image] = []
            for elem in data:
                temp.append(self.__client.Image(self.__client, data=elem))

            return temp

    class Product(Product):
        def __init__(self, client: Client, id: Optional[Union[int, str]] = None, data: Optional[Dict[str, Any]] = None) -> None:
            self.__client = client

            if data is None and id is None:
                raise ValueError("When initilizing category need to have data or id")

            if not data is None:
                id = data.get("itemno")

                if id is None:
                    raise ValueError("Expected data to have ID")

                if not isinstance(id, int) and id.isdigit():
                    id = int(id)

            super().__init__(id)

            if not data is None:
                self.name = data.get("title")
                self.price_current = data.get("price")
                self.brand = data.get("brand")



        def details(self):
            response = self.__client.request("GET", f"product/{self.id}", debug_key="product_details")

            if not isinstance(response, dict):
                raise ValueError("Expected value to be dict")

            data: Dict[str, Any] = response

            self.unit = data.get("baseUnit")
            self.brand = data.get("merk")
            self.description = data.get("wettelijke_naam")
            self.price_current = data.get("salePrice")
            self.price_raw = data.get("listPrice")
            self.quantity = data.get("ratioBasePackagingUnit")
            self.category_id = data.get("mainCategoryId")
            self.category_id = int(self.category_id) if not self.category_id is None and not isinstance(self.category_id, int) and self.category_id.isdigit() else None
            self.category = data.get("mainCategoryName")

            if not self.price_current is None and not self.price_raw is None:
                self.bonus = True if self.price_current < self.price_raw else False

            return self

        def price(self): # type: ignore
            if not self.price_current is None:
                return self.price_current
            else:
                return self.price_raw

    class Category(Category):
        subs: List[Client.Category]
        images: List[Client.Image]

        def __init__(
            self,
            client: Client,
            id: Optional[int] = None,
            slug_name: Optional[str] = None,
            name: Optional[str] = None,
            nix18: bool = False,
            images: List[Client.Image] = [],
            data: Optional[Dict[str, Any]] = None,
        ) -> None:
            self.__client = client

            if data is None and id is None:
                raise ValueError("When initilizing category need to have data or id")

            if not data is None:
                id = data.get("categoryid")

                slug_name = data.get("categorypath")
                name = data.get("title")
                nix18 = data.get("nix18", False)
                images = self.__client.images.process(data.get("images", []))


            if id is None:
                raise ValueError("Expected data to have ID")

            super().__init__(id, slug_name, name, nix18, True, images, [])

        def list_subs(self, recursive: bool = True):
            response = self.__client.request("GET", f"mobile-services/v1/product-shelves/categories/{self.id}/sub-categories", debug_key="list_subcategories")

            if not isinstance(response, dict):
                raise ValueError("Expected response to be dict")

            children: List[Dict[str, Any]] = response.get("children", [])

            for elem in children:
                cat = self.__client.Category(self.__client, data=elem)
                if not cat is None:
                    if recursive:
                        cat.list_subs()
                    self.subs.append(cat)

            return self.subs

        def lookup(self, id: Optional[int] = None, name: Optional[str] = None) -> Optional[Client.Category]:
            if not id is None:
                if self.id == id:
                    return self
                else:
                    for sub in self.subs:
                        lookup = sub.lookup(id=id)
                        if not lookup is None:
                            return lookup

                    for sub in self.list_subs(False):
                        lookup = sub.lookup(id=id)
                        if not lookup is None:
                            return lookup

                    return None
            elif not name is None:
                if self.name == name:
                    return self
                else:
                    for sub in self.subs:
                        lookup = sub.lookup(name=name)
                        if not lookup is None:
                            return lookup

                    for sub in self.list_subs(False):
                        lookup = sub.lookup(name=name)
                        if not lookup is None:
                            return lookup

                    return None
            else:
                return None

    class Image(Image):
        def __init__(
            self,
            client: Client,
            url: Optional[str] = None,
            data: Optional[Dict[str, Any]] = None,
        ) -> None:
            self.__client = client

            height = None
            width = None

            if not data is None:
                url = data.get("effectiveUrl")
                height = data.get("imageActualHeight")
                width = data.get("imageActualWidth")

            if url is None:
                raise ValueError("Expected image url not to be None")

            super().__init__(url, height, width)
