# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['dbt_osmosis', 'dbt_osmosis.exceptions', 'dbt_osmosis.utils']

package_data = \
{'': ['*']}

install_requires = \
['Flask>=1.1.2',
 'click>7',
 'dbt-client>=0.1.2,<0.2.0',
 'dbt-core>=0.20.0',
 'graphviz>=0.17',
 'itsdangerous==2.0.1',
 'pydot>=1.4.2',
 'rich>=10',
 'ruamel.yaml>=0.17',
 'streamlit-ace>=0.1.0',
 'streamlit-aggrid>=0.2.2',
 'streamlit-agraph>=0.0.35',
 'streamlit-pandas-profiling>=0.1.2',
 'streamlit>=1.0.0']

entry_points = \
{'console_scripts': ['dbt-osmosis = dbt_osmosis.main:cli',
                     'dbt-osmosis-proxy = dbt_osmosis.server:cli']}

setup_kwargs = {
    'name': 'dbt-osmosis',
    'version': '0.4.9',
    'description': 'This package exposes an interactive workbench for dbt modelling and serves to cascadingly populate column level documentation, build & conform schema files, and audit coverage.',
    'long_description': '# dbt-osmosis\n\n<!--![GitHub Actions](https://github.com/z3z1ma/dbt-osmosis/actions/workflows/master.yml/badge.svg)-->\n\n![PyPI](https://img.shields.io/pypi/v/dbt-osmosis)\n![Downloads](https://pepy.tech/badge/dbt-osmosis)\n![License: Apache 2.0](https://img.shields.io/badge/License-Apache_2.0-green.svg)\n![black](https://img.shields.io/badge/code%20style-black-000000.svg)\n\n## Primary Objectives\n\nHello and welcome to the project! [dbt-osmosis](https://github.com/z3z1ma/dbt-osmosis) 🌊 serves to enhance the developer experience significantly. We do this by automating the most of the management of schema yml files, we synchronize inheritable column level documentation which permits a write-it-once principle in a DAG oriented way, we enforce user-defined organization for schema yml files in your dbt project automatically making it super tidy, we automatically inject models which are undocumented into the appropriate schema right where you expect it, and we expose a **workbench** which allows you to interactively develop in dbt. The workbench allows you to develop and instantly compile models side by side (extremely performant compilation), document model columns, test the query against your data warehouse, inspect row level diffs and diff metric as you modify SQL, run dbt tests interactively & download results, and more. \n\n\n[Workbench Reference](#Workbench)\n\n[CLI Reference](#CLI)\n\n____\n\n\n## Workbench\n\nThe workbench is under active development. Feel free to open issues or discuss additions. \n\n```sh\n# Command to start server\ndbt-osmosis workbench\n```\n\nPress "r" to reload the workbench at any time.\n\n\n\n✔️ dbt Model Editor\n\n✔️ Full control over model and workbench theme, light and dark mode\n\n✔️ Create or delete models from the workbench without switching context\n\n✔️ Materialize Active Model in Warehouse\n\n✔️ Query Tester, test the model you are working on for instant feedback\n\n✔️ SQL Model Data Diffs, modify models with confidence like never before\n\n  - Adding pandas engine and support for `MODIFIED` rows in addition to `ADDED` and `REMOVED`\n\n  - Adding scorecards which show the sum of each of the 3 diff categories\n\n✔️ Data Profiler (leverages pandas-profiling)\n\n⚠️ Doc Editor (resolves basic column level lineage)\n\n  - View only, modifications aren\'t committed yet\n\n✔️ Test Runner, run dbt tests interactively and on the fly with the ability to download or inspect results and action\n\n✔️ Manifest View\n\n\n\n**Editor** \n\nThe editor is able to compile models with control+enter or dynamically as you type. Its speedy!\n\n![editor](/screenshots/osmosis_editor.png?raw=true "dbt-osmosis Workbench")\n\n**Profile Selection**\n\nSelect a target, models can also be materialized by executing the SQL against the target using dbt as a wrapper.\n\n![profiles](/screenshots/osmosis_profile_selection.png?raw=true "dbt-osmosis Profile Selection")\n\n\n**Edit and Save Models**\n\nSee when there are uncommitted changes and commit them to file when ready, or revert to initial state. Pivot the layout if you prefer a larger editor context or pivot it back to get side by side instant dbt jinja compilation to accelerate your learning\n\n![pivot-uncommitted](/screenshots/osmosis_pivot_layout_uncommitted_changes.png?raw=true "dbt-osmosis Pivot Layout")\n\n\n**Test Query**\n\nTest dbt models as you work against whatever profile you have selected and inspect the results.\n\n![test-model](/screenshots/osmosis_test_dbt_model.png?raw=true "dbt-osmosis Test Model")\n\n**Row Level Diffs**\n\nAs you develop and modify a model with uncommitted changes, you can calculate the diff. This allows you instant feedback on if the changes you make are safe.\n\n![diff-model](/screenshots/osmosis_row_level_diff.png?raw=true "dbt-osmosis Diff Model")\n\n**Profile Model Results**\n\nProfile your datasets on the fly while you develop without switching context. Allows for more refined interactive data modelling when dataset fits in memory.\n\n![profile-data](/screenshots/osmosis_profile_data.png?raw=true "dbt-osmosis Profile Data")\n\n**Run dbt Tests**\n\nRun declared dbt data tests interactively with the ability to download the results to CSV.\n\n![data-tests](/screenshots/osmosis_data_tests.png?raw=true "dbt-osmosis Data Tests")\n\n\n____\n\n\n## CLI\n\ndbt-osmosis is ready to use as-is. To get familiar, you should run it on a fresh branch and ensure everything is backed in source control. Enjoy!\n\nYou should set a base config in your dbt_project.yml and ensure any models within the scope of your execution plan will inherit a config/preference. Example below.\n\n```yaml\nmodels:\n\n    your_dbt_project:\n\n        # This config will apply to your whole project\n        +dbt-osmosis: "schema/model.yml"\n\n        staging:\n\n            # This config will apply to your staging directory\n            +dbt-osmosis: "folder.yml"\n\n            +tags: \n                - "staged"\n\n            +materialized: view\n\n            monday:\n                intermediate:\n                    +materialized: ephemeral\n\n        marts:\n\n            +tags: \n                - "mart"\n\n            supply_chain: \n```\n\nTo use dbt-osmosis, simply run the following:\n\n```bash\n# Install\npip install dbt-osmosis\n# Alternatively\npipx install dbt-osmosis\n\n\n# This command executes all tasks in preferred order and is usually all you need\n\ndbt-osmosis run --project-dir /path/to/dbt/project --target prod\n\n\n# Inherit documentation in staging/salesforce/ & sync \n# schema yaml columns with database columns\n\ndbt-osmosis document --project-dir /path/to/dbt/project --target prod --fqn staging.salesforce\n\n\n# Reorganize marts/operations/ & inject undocumented models \n# into schema files or create new schema files as needed\n\ndbt-osmosis compose --project-dir /path/to/dbt/project --target prod --fqn marts.operations\n\n\n# Open the dbt-osmosis workbench\n\ndbt-osmosis workbench\n```\n\n## Roadmap\n\nThese features are being actively developed and will be merged into the next few minor releases\n\n1. Complete build out of `sources` tools.\n2. Add `--min-cov` flag to audit task and to workbench\n3. Add interactive documentation flag that engages user to documents ONLY progenitors and novel columns for a subset of models (the most optimized path to full documentation coverage feasible)\n4. Add `impact` command that allows us to leverage our resolved column level progenitors for ad hoc impact analysis\n\n## Features\n\n### Standardize organization of schema files (and provide ability to define and conform with code)\n\n- Config can be set on per directory basis if desired utilizing `dbt_project.yml`, all models which are processed require direct or inherited config `+dbt-osmosis:`. If even one dir is missing the config, we close gracefully and inform user to update dbt_project.yml. No assumed defaults. Placing our config under your dbt project name in `models:` is enough to set a default for the project since the config applies to all subdirectories. \n\n    Note: You can **change these configs as often as you like** or try them all, dbt-osmosis will take care of restructuring your project schema files-- _no human effort required_. \n\n    A directory can be configured to conform to any one of the following standards:\n\n    - Can be one schema file to one model file sharing the same name and directory ie. \n\n            staging/\n                stg_order.sql\n                stg_order.yml\n                stg_customer.sql\n                stg_customer.yml\n\n        - `+dbt-osmosis: "model.yml"`\n\n    - Can be one schema file per directory wherever model files reside named schema.yml, ie.\n\n            staging/\n                stg_order.sql\n                stg_customer.sql\n                schema.yml\n\n        - `+dbt-osmosis: "schema.yml"`\n    - Can be one schema file per directory wherever model files reside named after its containing folder, ie. \n\n            staging/\n                stg_order.sql\n                stg_customer.sql\n                staging.yml\n\n        - `+dbt-osmosis: "folder.yml"`\n\n    - Can be one schema file to one model file sharing the same name _nested_ in a schema subdir wherever model files reside, ie. \n\n            staging/\n                stg_order.sql\n                stg_customer.sql\n                schema/\n                    stg_order.yml\n                    stg_customer.yml\n\n        - `+dbt-osmosis: "schema/model.yml"`\n\n### Build and Inject Non-documented models\n\n- Injected models will automatically conform to above config per directory based on location of model file. \n\n- This means you can focus fully on modelling; and documentation, including yaml updates or creation, will automatically follow at any time with simple invocation of dbt-osmosis\n\n### Propagate existing column level documentation downward to children\n\n- Build column level knowledge graph accumulated and updated from furthest identifiable origin (ancestors) to immediate parents\n\n- Will automatically populate undocumented columns of the same name with passed down knowledge accumulated within the context of the models upstream dependency tree\n\n- This means you can freely generate models and all columns you pull into the models SQL that already have been documented will be automatically learned/propagated. Again the focus for analysts is almost fully on modelling and yaml work is an afterthought / less heavy of a manual lift.\n\n### Order Matters\n\nIn a full run [ `dbt-osmosis run` ] we will:\n\n1. Conform dbt project\n    - Configuration lives in `dbt_project.yml` --> we require our config to run, can be at root level of `models:` to apply a default convention to a project \n    or can be folder by folder, follows dbt config resolution where config is overridden by scope. \n    Config is called `+dbt-osmosis: "folder.yml" | "schema.yml" | "model.yml" | "schema/model.yml"`\n2. Bootstrap models to ensure all models exist\n3. Recompile Manifest\n4. Propagate definitions downstream to undocumented models solely within the context of each models dependency tree\n\n\n#### Here are some of the original foundational pillars:\n\nFirst and foremost, we want dbt documentation to retain a DRY principle. Every time we repeat ourselves, we waste our time. 80% of documentation is often a matter of inheritance and continued passing down of columns from parent models to children. They need not be redocumented if there has been no mutation. \n\nSecond, we want to standardize ways that we all organize our schema files which hold the fruits of our documentation. We should be able to enforce a standard on a per directory basis and jump between layouts at will as certain folders scale up the number of models or scale down. \n\nLastly, and tangential to the first objective, we want to understand column level lineage, streamline impact analysis, and audit our documentation.\n\n\n## New workflows enabled!\n\n1. Build one dbt model or a __bunch__ of them without documenting anything (gasp)\n\n    Run `dbt-osmosis run` or `dbt-osmosis compose && dbt-osmosis document`\n    \n    Sit back and watch as:\n\n    Automatically constructed/updated schema yamls are built with as much of the definitions pre-populated as possible from upstream dependencies \n    \n    Schema yaml(s) are automatically organized in exactly the right directories / style that conform to the easily configurable standard upheld and enforced across your dbt project on a directory by directory basis \n    \n    boom, mic drop\n\n2. Problem reported by stakeholder with data **(WIP)**\n    \n    Identify column\n    \n    Run `dbt-osmosis impact --model orders --column price`\n    \n    Find the originating model and action\n\n3. Need to score our documentation **(WIP)**\n\n    Run `dbt-osmosis audit --docs --min-cov 80`\n\n    Get a curated list of all the documentation to update in your pre-bootstrapped dbt project\n\n    Sip coffee and engage in documentation\n\n4. Add dbt-osmosis to a pre-commit hook to ensure all your analysts are passing down column level documentation & reaching your designated min-coverage',
    'author': 'z3z1ma',
    'author_email': 'butler.alex2010@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/z3z1ma/dbt-osmosis',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
