"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    deployment.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W45',
                    reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
                }]
        }
    };
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    if (params.awsIntegrationProps) {
        const overridenProps = utils_1.overrideProps(baseProps, params.awsIntegrationProps);
        apiGatewayIntegration = new api.AwsIntegration(overridenProps);
    }
    else {
        apiGatewayIntegration = new api.AwsIntegration(baseProps);
    }
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    if (params.methodOptions) {
        const overridenProps = utils_1.overrideProps(defaultMethodOptions, params.methodOptions);
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    }
    else {
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, defaultMethodOptions);
    }
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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