"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${core_1.Aws.PARTITION}:kms:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps.command, _glueSecurityConfigName, _jobArgumentsList, glueJobProps.glueVersion), glueJobProps);
    let _scriptLocation;
    if (isJobCommandProperty(_newGlueJobProps.command)) {
        if (_newGlueJobProps.command.scriptLocation) {
            _scriptLocation = _newGlueJobProps.command.scriptLocation;
        }
        else {
            throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
        }
    }
    const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
    _scriptBucketLocation.grantRead(_jobRole);
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    const splitString = s3Url.slice('s3://'.length);
    return `arn:${core_1.Aws.PARTITION}:s3:::${splitString}`;
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name ||
        command.pythonVersion ||
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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