"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        });
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_10_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_12_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    cfnLambdafunction.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
                },
                {
                    id: 'W89',
                    reason: `This is not a rule for the general case, just for specific use cases/industries`
                },
                {
                    id: 'W92',
                    reason: `Impossible for us to define the correct concurrency for clients`
                }]
        }
    };
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        cfnLambdafunctionDefPolicy.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
                    }]
            }
        };
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
//# sourceMappingURL=data:application/json;base64,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