"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.createLoggingBucket = exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const rule_1 = require("@aws-cdk/aws-s3/lib/rule");
const duration_1 = require("@aws-cdk/core/lib/duration");
function buildS3Bucket(scope, props, bucketId) {
    if (props.bucketProps) {
        return s3BucketWithLogging(scope, props.bucketProps, bucketId);
    }
    else {
        return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId, removalPolicy) {
    let loggingBucketProps;
    if (removalPolicy) {
        loggingBucketProps = utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), { removalPolicy });
    }
    else {
        loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
    }
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Turn off Versioning for the logging bucket as objects will be written only ONCE
    loggingBucketResource.addPropertyDeletionOverride('VersioningConfiguration.Status');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: _reason
                }]
        }
    };
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: rule_1.StorageClass.GLACIER,
                    transitionAfter: duration_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if (s3BucketProps.versioned === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(undefined, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props();
        }
    }
    else {
        // Create the Logging Bucket
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.removalPolicy);
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if ((s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.versioned) === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
        }
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    fnResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
                },
                {
                    id: 'W89',
                    reason: `This is not a rule for the general case, just for specific use cases/industries`
                },
                {
                    id: 'W92',
                    reason: `Impossible for us to define the correct concurrency for clients`
                }]
        }
    };
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    policyResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                }]
        }
    };
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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