"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("@aws-cdk/core");
const smDefaults = require("./step-function-defaults");
const sfn = require("@aws-cdk/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    let consolidatedLogGroupProps = logGroupProps;
    // Three possibilities
    // 1) logGroupProps not provided - create logGroupProps with just logGroupName
    // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
    // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
    if (!consolidatedLogGroupProps) {
        consolidatedLogGroupProps = {};
    }
    if (!(consolidatedLogGroupProps === null || consolidatedLogGroupProps === void 0 ? void 0 : consolidatedLogGroupProps.logGroupName)) {
        const logGroupPrefix = '/aws/vendedlogs/states/';
        const maxResourceNameLength = 255 - logGroupPrefix.length;
        const nameParts = [
            cdk.Stack.of(scope).stackName,
            scope.node.id,
            'StateMachineLog' // Literal string for log group name portion
        ];
        const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength);
        consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
    }
    // Configure Cloudwatch log group for Step function State Machine
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
    // Override the defaults with the user provided props
    const _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    cfnDefaultPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `The 'LogDelivery' actions do not support resource-level authorizations`
                }]
        }
    };
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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