"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateResourceName = exports.printWarning = exports.overrideProps = void 0;
const deepmerge = require("deepmerge");
const override_warning_service_1 = require("./override-warning-service");
const log = require("npmlog");
const crypto = require("crypto");
function isObject(val) {
    return val != null && typeof val === 'object'
        && Object.prototype.toString.call(val) === '[object Object]';
}
function isPlainObject(o) {
    if (Array.isArray(o) === true) {
        return true;
    }
    if (isObject(o) === false) {
        return false;
    }
    // If has modified constructor
    const ctor = o.constructor;
    if (typeof ctor !== 'function') {
        return false;
    }
    // If has modified prototype
    const prot = ctor.prototype;
    if (isObject(prot) === false) {
        return false;
    }
    // If constructor does not have an Object-specific method
    if (prot.hasOwnProperty('isPrototypeOf') === false) {
        return false;
    }
    // Most likely a plain Object
    return true;
}
function combineMerge(target, source) {
    return target.concat(source);
}
function overwriteMerge(target, source) {
    target = source;
    return target;
}
function overrideProps(DefaultProps, userProps, concatArray = false) {
    // Notify the user via console output if defaults are overridden
    const overrideWarningsEnabled = (process.env.overrideWarningsEnabled !== 'false');
    if (overrideWarningsEnabled) {
        override_warning_service_1.flagOverriddenDefaults(DefaultProps, userProps);
    }
    // Override the sensible defaults with user provided props
    if (concatArray) {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: combineMerge,
            isMergeableObject: isPlainObject
        });
    }
    else {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: overwriteMerge,
            isMergeableObject: isPlainObject
        });
    }
}
exports.overrideProps = overrideProps;
function printWarning(message) {
    // Style the log output
    log.prefixStyle.bold = true;
    log.prefixStyle.fg = 'red';
    log.enableColor();
    log.warn('AWS_SOLUTIONS_CONSTRUCTS_WARNING: ', message);
}
exports.printWarning = printWarning;
/**
 * @summary Creates a resource name in the style of the CDK (string+hash)
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if neccessary) + a hash of the full concatenated parts
 *
 * This is based upon this discussion - https://github.com/aws/aws-cdk/issues/1424
 */
function generateResourceName(parts, maxLength) {
    const hashLength = 12;
    const maxPartLength = Math.floor((maxLength - hashLength) / parts.length);
    const sha256 = crypto.createHash("sha256");
    let finalName = '';
    parts.forEach((part) => {
        sha256.update(part);
        finalName += removeNonAlphanumeric(part.slice(0, maxPartLength));
    });
    const hash = sha256.digest("hex").slice(0, hashLength);
    finalName += hash;
    return finalName.toLowerCase();
}
exports.generateResourceName = generateResourceName;
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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