"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployRegionalApiGateway(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    return defaults.RegionalLambdaRestApi(stack, fn);
}
function setupRestApi(stack, apiProps) {
    const [restApi] = defaults.GlobalRestApi(stack, apiProps);
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    // Setup the API Gateway Integration
    const apiGatewayIntegration = new api.AwsIntegration({
        service: "sqs",
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            requestParameters: {
                "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
            },
            requestTemplates: {
                "application/x-www-form-urlencoded": "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")&MessageAttribute.1.Name=queryParam1&MessageAttribute.1.Value.StringValue=$input.params(\"query_param_1\")&MessageAttribute.1.Value.DataType=String"
            },
            integrationResponses: [
                {
                    statusCode: "200",
                    responseTemplates: {
                        "text/html": "Success"
                    }
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        },
        path: '11112222' + "/" + 'thisqueuequeueName'
    });
    // Setup the API Gateway method(s)
    apiGatewayResource.addMethod('POST', apiGatewayIntegration, {
        requestParameters: {
            "method.request.querystring.query_param_1": true
        },
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ]
    });
}
test('snapshot test RegionalApiGateway default params', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        description: 'Hello World'
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            Description: "Hello World",
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "LambdaRestApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test override for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn, {
        handler: fn,
        restApiName: "HelloWorld"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "EDGE"
                ]
            },
            Name: "HelloWorld"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test ApiGateway::Account resource for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test ApiGateway::Account resource for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test default RestApi deployment w/o ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test default RestApi deployment w/ ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        restApiName: "customRestApi"
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Name: "customRestApi"
    });
});
test('Test default RestApi deployment for Cloudwatch loggroup', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        UpdateReplacePolicy: "Retain",
        DeletionPolicy: "Retain"
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::ApiGateway::Stage', {
        AccessLogSetting: {
            DestinationArn: {
                "Fn::GetAtt": [
                    "ApiAccessLogGroupCEA70788",
                    "Arn"
                ]
            },
            Format: "{\"requestId\":\"$context.requestId\",\"ip\":\"$context.identity.sourceIp\",\"user\":\"$context.identity.user\",\"caller\":\"$context.identity.caller\",\"requestTime\":\"$context.requestTime\",\"httpMethod\":\"$context.httpMethod\",\"resourcePath\":\"$context.resourcePath\",\"status\":\"$context.status\",\"protocol\":\"$context.protocol\",\"responseLength\":\"$context.responseLength\"}",
        },
    });
});
test('Test addMethodToApiResource with action', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const getRequestTemplate = "{}";
    // Add Method
    defaults.addProxyMethodToApiResource({
        action: "Query",
        service: "dynamodb",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "GET",
        requestTemplate: getRequestTemplate
    });
    // Add Method
    defaults.addProxyMethodToApiResource({
        path: '11112222' + "/" + 'thisqueuequeueName',
        service: "sqs",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "PUT",
        requestTemplate: getRequestTemplate
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Error scenario: missing action and path
    try {
        defaults.addProxyMethodToApiResource({
            service: "sqs",
            apiResource: apiGatewayResource,
            apiGatewayRole,
            apiMethod: "DELETE",
            requestTemplate: getRequestTemplate
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Test default RestApi w/ request model and validator', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const validator = restApi.addRequestValidator('default-validator', {
        requestValidatorName: 'default-validator',
        validateRequestBody: true
    });
    defaults.addProxyMethodToApiResource({
        service: "kinesis",
        action: "PutRecord",
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "{}",
        contentType: "'x-amz-json-1.1'",
        requestValidator: validator,
        requestModel: { "application/json": api.Model.EMPTY_MODEL }
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource('AWS::ApiGateway::RequestValidator', {
        Name: "default-validator",
        ValidateRequestBody: true
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        RequestModels: { "application/json": "Empty" }
    });
});
// -----------------------------------------------------------------------
// Test for Regional ApiGateway Creation
// -----------------------------------------------------------------------
test('Test for RegionalRestApiGateway', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack, {
        restApiName: "HelloWorld-RegionalApi"
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "HelloWorld-RegionalApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// -----------------------------------------------------------------------
// Tests for exception while overriding restApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding restApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    try {
        defaults.RegionalRestApi(stack, {
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Tests for exception while overriding LambdaRestApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding LambdaRestApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    try {
        defaults.GlobalLambdaRestApi(stack, fn, {
            handler: fn,
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Test for Integration Request Props Override
// -----------------------------------------------------------------------
test('Test for Integration Request Props Override', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    const integReqParams = { 'integration.request.path.topic-level-1': "'method.request.path.topic-level-1'" };
    const integResp = [
        {
            statusCode: "200",
            selectionPattern: "2\\d{2}",
            responseTemplates: {
                "application/json": "$input.json('$')"
            }
        }
    ];
    // Override the default Integration Request Props
    const integrationReqProps = {
        subdomain: 'abcdefgh12345',
        options: {
            requestParameters: integReqParams,
            integrationResponses: integResp,
            passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
        }
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        awsIntegrationProps: integrationReqProps
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
                "integration.request.path.topic-level-1": "'method.request.path.topic-level-1'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:abcdefgh12345.iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200",
                ResponseParameters: {
                    "method.response.header.Content-Type": true
                }
            }
        ]
    });
});
// -----------------------------------------------------------------------
// Test for Method Options Override
// -----------------------------------------------------------------------
test('Test for Method Request Props Override', () => {
    const stack = new core_1.Stack();
    const [globalApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalApi.root.addResource('hello');
    const methodReqParams = { 'method.request.path.topic-level-1': true };
    const methodResp = [
        {
            statusCode: "403"
        }
    ];
    const resourceMethodOptions = {
        requestParameters: methodReqParams,
        methodResponses: methodResp,
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        methodOptions: resourceMethodOptions
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    StatusCode: "200"
                },
                {
                    StatusCode: "500",
                    ResponseTemplates: {
                        "text/html": "Error"
                    },
                    SelectionPattern: "500"
                }
            ],
            PassthroughBehavior: "NEVER",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true
        }
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of RestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using restApiProps', () => {
    const stack = new core_1.Stack();
    const [globalRestApi] = defaults.GlobalRestApi(stack, {
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalRestApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of LambdaRestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using lambdaApiProps', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
//# sourceMappingURL=data:application/json;base64,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