"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
const dynamodb_table_helper_1 = require("../lib/dynamodb-table-helper");
test('snapshot test TableProps default params', () => {
    const stack = new core_1.Stack();
    new dynamodb.Table(stack, 'test-dynamo-defaults', defaults.DefaultTableProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('snapshot test TableWithStream default params', () => {
    const stack = new core_1.Stack();
    new dynamodb.Table(stack, 'test-dynamo-stream-defaults', defaults.DefaultTableWithStreamProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test TableProps change billing mode', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        readCapacity: 3,
        writeCapacity: 3,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 3,
            WriteCapacityUnits: 3
        },
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('test TableProps override add sort key', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
        sortKey: {
            name: 'sort_key',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            },
            {
                AttributeName: "sort_key",
                KeyType: "RANGE"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            },
            {
                AttributeName: "sort_key",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('test TableWithStreamProps override stream view type', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableWithStreamProps;
    const inProps = {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        },
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    });
});
test('test buildDynamoDBTable with existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        }
    };
    const existingTableObj = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTable(stack, {
        existingTableObj
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        ProvisionedThroughput: {
            ReadCapacityUnits: 5,
            WriteCapacityUnits: 5
        }
    }));
});
test('test buildDynamoDBTable without any arguments', () => {
    const stack = new core_1.Stack();
    defaults.buildDynamoDBTable(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        BillingMode: "PAY_PER_REQUEST"
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        SSESpecification: {
            SSEEnabled: true
        }
    }));
});
test('test buildDynamoDBTable with TableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        }
    };
    defaults.buildDynamoDBTable(stack, {
        dynamoTableProps
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        ProvisionedThroughput: {
            ReadCapacityUnits: 5,
            WriteCapacityUnits: 5
        }
    }));
});
test('test buildDynamoDBTableWithStream with TableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    defaults.buildDynamoDBTableWithStream(stack, {
        dynamoTableProps
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
test('test buildDynamoDBTableWithStream without any arguments', () => {
    const stack = new core_1.Stack();
    defaults.buildDynamoDBTableWithStream(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        BillingMode: "PAY_PER_REQUEST"
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        SSESpecification: {
            SSEEnabled: true
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_AND_OLD_IMAGES"
        }
    }));
});
test('test buildDynamoDBTableWithStream with existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableObj = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTableWithStream(stack, {
        existingTableObj
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
test('test getPartitionKeyNameFromTable()', () => {
    const partitionKeyName = 'testPartitionKey';
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        partitionKey: {
            name: partitionKeyName,
            type: dynamodb.AttributeType.STRING
        },
        sortKey: {
            name: 'sort_key',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    const newTable = new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    const testKeyName = dynamodb_table_helper_1.getPartitionKeyNameFromTable(newTable);
    expect(testKeyName).toEqual(partitionKeyName);
});
//# sourceMappingURL=data:application/json;base64,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