"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * Define an ApplicationListener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.anyValue({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * Import an existing listener
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        if (additionalCerts.length > 0) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
                listener: this,
                certificates: additionalCerts,
            });
        }
    }
    /**
     * Perform the given default action on incoming requests
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     */
    addAction(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                action: props.action,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: props.protocol,
            slowStart: props.slowStart,
            stickinessCookieDuration: props.stickinessCookieDuration,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, {
            conditions: props.conditions,
            hostHeader: props.hostHeader,
            pathPattern: props.pathPattern,
            pathPatterns: props.pathPatterns,
            priority: props.priority,
            targetGroups: [group],
        });
        return group;
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * Validate this listener.
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
class ImportedApplicationListener extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificateArns(id, arns) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificateArns: arns,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
}
function checkAddRuleProps(props) {
    var _a;
    const conditionsCount = ((_a = props.conditions) === null || _a === void 0 ? void 0 : _a.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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