"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpCodeTarget = exports.HttpCodeElb = exports.ApplicationLoadBalancer = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const base_load_balancer_1 = require("../shared/base-load-balancer");
const enums_1 = require("../shared/enums");
const application_listener_1 = require("./application-listener");
/**
 * Define an Application Load Balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class ApplicationLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props, {
            type: 'application',
            securityGroups: core_1.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            ipAddressType: props.ipAddressType,
        });
        this.ipAddressType = (_a = props.ipAddressType) !== null && _a !== void 0 ? _a : enums_1.IpAddressType.IPV4;
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            description: `Automatically created Security Group for ELB ${this.node.uniqueId}`,
            allowAllOutbound: false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        if (props.http2Enabled === false) {
            this.setAttribute('routing.http2.enabled', 'false');
        }
        if (props.idleTimeout !== undefined) {
            this.setAttribute('idle_timeout.timeout_seconds', props.idleTimeout.toSeconds().toString());
        }
    }
    /**
     * Import an existing Application Load Balancer
     */
    static fromApplicationLoadBalancerAttributes(scope, id, attrs) {
        return new ImportedApplicationLoadBalancer(scope, id, attrs);
    }
    /**
     * Add a new listener to this load balancer
     */
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Return the given named metric for this Application Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        });
    }
    /**
     * The total number of concurrent TCP connections active from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricActiveConnectionCount(props) {
        return this.metric('ActiveConnectionCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the client that did not
     * establish a session with the load balancer. Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricClientTlsNegotiationErrorCount(props) {
        return this.metric('ClientTLSNegotiationErrorCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpFixedResponseCount(props) {
        return this.metric('HTTP_Fixed_Response_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectCount(props) {
        return this.metric('HTTP_Redirect_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that couldn't be completed because the URL
     * in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectUrlLimitExceededCount(props) {
        return this.metric('HTTP_Redirect_Url_Limit_Exceeded_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeElb(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets
     * in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6ProcessedBytes(props) {
        return this.metric('IPv6ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of new TCP connections established from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricNewConnectionCount(props) {
        return this.metric('NewConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were rejected because the load balancer had
     * reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     */
    metricRejectedConnectionCount(props) {
        return this.metric('RejectedConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     */
    metricRuleEvaluations(props) {
        return this.metric('RuleEvaluations', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthError(props) {
        return this.metric('ELBAuthError', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed because the
     * IdP denied access to the user or an authorization code was used more than
     * once.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthFailure(props) {
        return this.metric('ELBAuthFailure', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     */
    metricElbAuthLatency(props) {
        return this.metric('ELBAuthLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthSuccess(props) {
        return this.metric('ELBAuthSuccess', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.ApplicationLoadBalancer = ApplicationLoadBalancer;
/**
 * Count of HTTP status originating from the load balancer
 *
 * This count does not include any response codes generated by the targets.
 */
var HttpCodeElb;
(function (HttpCodeElb) {
    /**
     * The number of HTTP 3XX redirection codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_3XX_COUNT"] = "HTTPCode_ELB_3XX_Count";
    /**
     * The number of HTTP 4XX client error codes that originate from the load balancer.
     *
     * Client errors are generated when requests are malformed or incomplete.
     * These requests have not been received by the target. This count does not
     * include any response codes generated by the targets.
     */
    HttpCodeElb["ELB_4XX_COUNT"] = "HTTPCode_ELB_4XX_Count";
    /**
     * The number of HTTP 5XX server error codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_5XX_COUNT"] = "HTTPCode_ELB_5XX_Count";
})(HttpCodeElb = exports.HttpCodeElb || (exports.HttpCodeElb = {}));
/**
 * Count of HTTP status originating from the targets
 */
var HttpCodeTarget;
(function (HttpCodeTarget) {
    /**
     * The number of 2xx response codes from targets
     */
    HttpCodeTarget["TARGET_2XX_COUNT"] = "HTTPCode_Target_2XX_Count";
    /**
     * The number of 3xx response codes from targets
     */
    HttpCodeTarget["TARGET_3XX_COUNT"] = "HTTPCode_Target_3XX_Count";
    /**
     * The number of 4xx response codes from targets
     */
    HttpCodeTarget["TARGET_4XX_COUNT"] = "HTTPCode_Target_4XX_Count";
    /**
     * The number of 5xx response codes from targets
     */
    HttpCodeTarget["TARGET_5XX_COUNT"] = "HTTPCode_Target_5XX_Count";
})(HttpCodeTarget = exports.HttpCodeTarget || (exports.HttpCodeTarget = {}));
/**
 * An ApplicationLoadBalancer that has been defined elsewhere
 */
class ImportedApplicationLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.vpc = props.vpc;
        this.loadBalancerArn = props.loadBalancerArn;
        this.connections = new ec2.Connections({
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                    allowAllOutbound: props.securityGroupAllowsAllOutbound,
                })],
        });
    }
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    get loadBalancerCanonicalHostedZoneId() {
        if (this.props.loadBalancerCanonicalHostedZoneId) {
            return this.props.loadBalancerCanonicalHostedZoneId;
        }
        // eslint-disable-next-line max-len
        throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
    get loadBalancerDnsName() {
        if (this.props.loadBalancerDnsName) {
            return this.props.loadBalancerDnsName;
        }
        // eslint-disable-next-line max-len
        throw new Error(`'loadBalancerDnsName' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
}
//# sourceMappingURL=data:application/json;base64,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