"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancer = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const base_load_balancer_1 = require("../shared/base-load-balancer");
const network_listener_1 = require("./network-listener");
/**
 * Define a new network load balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class NetworkLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    static fromNetworkLoadBalancerAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.loadBalancerArn = attrs.loadBalancerArn;
                this.vpc = attrs.vpc;
            }
            addListener(lid, props) {
                return new network_listener_1.NetworkListener(this, lid, {
                    loadBalancer: this,
                    ...props,
                });
            }
            get loadBalancerCanonicalHostedZoneId() {
                if (attrs.loadBalancerCanonicalHostedZoneId) {
                    return attrs.loadBalancerCanonicalHostedZoneId;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
            get loadBalancerDnsName() {
                if (attrs.loadBalancerDnsName) {
                    return attrs.loadBalancerDnsName;
                }
                // eslint-disable-next-line max-len
                throw new Error(`'loadBalancerDnsName' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'network',
        });
        if (props.crossZoneEnabled) {
            this.setAttribute('load_balancing.cross_zone.enabled', 'true');
        }
    }
    /**
     * Add a listener to this load balancer
     *
     * @returns The newly created listener
     */
    addListener(id, props) {
        return new network_listener_1.NetworkListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     *
     * This is extending the BaseLoadBalancer.logAccessLogs method to match the bucket permissions described
     * at https://docs.aws.amazon.com/elasticloadbalancing/latest/network/load-balancer-access-logs.html#access-logging-bucket-requirements
     */
    logAccessLogs(bucket, prefix) {
        super.logAccessLogs(bucket, prefix);
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
    }
    /**
     * Return the given named metric for this Network Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The total number of concurrent TCP flows (or connections) from clients to targets.
     *
     * This metric includes connections in the SYN_SENT and ESTABLISHED states.
     * TCP connections are not terminated at the load balancer, so a client
     * opening a TCP connection to a target counts as a single flow.
     *
     * @default Average over 5 minutes
     */
    metricActiveFlowCount(props) {
        return this.metric('ActiveFlowCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The total number of new TCP flows (or connections) established from clients to targets in the time period.
     *
     * @default Sum over 5 minutes
     */
    metricNewFlowCount(props) {
        return this.metric('NewFlowCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer, including TCP/IP headers.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of reset (RST) packets sent from a client to a target.
     *
     * These resets are generated by the client and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpClientResetCount(props) {
        return this.metric('TCP_Client_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of reset (RST) packets generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpElbResetCount(props) {
        return this.metric('TCP_ELB_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of reset (RST) packets sent from a target to a client.
     *
     * These resets are generated by the target and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpTargetResetCount(props) {
        return this.metric('TCP_Target_Reset_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.NetworkLoadBalancer = NetworkLoadBalancer;
//# sourceMappingURL=data:application/json;base64,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