"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const cdk = require("@aws-cdk/core");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define a Network Target Group
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props) {
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing listener
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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