"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// eslint-disable-next-line max-len
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
//# sourceMappingURL=data:application/json;base64,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