"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseScalableAttribute = void 0;
const cdk = require("@aws-cdk/core");
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFzZS1zY2FsYWJsZS1hdHRyaWJ1dGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJiYXNlLXNjYWxhYmxlLWF0dHJpYnV0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxxQ0FBcUM7QUFDckMsdURBQXNGO0FBNkJ0Rjs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ0gsTUFBc0IscUJBQXNCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFHL0QsWUFBbUIsS0FBb0IsRUFBRSxFQUFVLEVBQXFCLEtBQWlDO1FBQ3ZHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFEcUQsVUFBSyxHQUFMLEtBQUssQ0FBNEI7UUFHdkcsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLGdDQUFjLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUMvQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQjtZQUM3QyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7WUFDdkMsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVTtZQUNqQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJO1lBQ3JCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNwRSxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7U0FDL0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ08saUJBQWlCLENBQUMsRUFBVSxFQUFFLEtBQXNCO1FBQzVELElBQUksQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7O09BRUc7SUFDTyxlQUFlLENBQUMsRUFBVSxFQUFFLEtBQWtDO1FBQ3RFLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7O09BRUc7SUFDTyxvQkFBb0IsQ0FBQyxFQUFVLEVBQUUsS0FBNEM7UUFDckYsSUFBSSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDNUMsQ0FBQztDQUNGO0FBcENELHNEQW9DQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFNjYWxhYmxlVGFyZ2V0LCBTY2FsaW5nU2NoZWR1bGUsIFNlcnZpY2VOYW1lc3BhY2UgfSBmcm9tICcuL3NjYWxhYmxlLXRhcmdldCc7XG5pbXBvcnQgeyBCYXNpY1N0ZXBTY2FsaW5nUG9saWN5UHJvcHMgfSBmcm9tICcuL3N0ZXAtc2NhbGluZy1wb2xpY3knO1xuaW1wb3J0IHsgQmFzaWNUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lQcm9wcyB9IGZyb20gJy4vdGFyZ2V0LXRyYWNraW5nLXNjYWxpbmctcG9saWN5JztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIFNjYWxhYmxlVGFibGVBdHRyaWJ1dGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCYXNlU2NhbGFibGVBdHRyaWJ1dGVQcm9wcyBleHRlbmRzIEVuYWJsZVNjYWxpbmdQcm9wcyB7XG4gIC8qKlxuICAgKiBTZXJ2aWNlIG5hbWVzcGFjZSBvZiB0aGUgc2NhbGFibGUgYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBzZXJ2aWNlTmFtZXNwYWNlOiBTZXJ2aWNlTmFtZXNwYWNlO1xuXG4gIC8qKlxuICAgKiBSZXNvdXJjZSBJRCBvZiB0aGUgYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNjYWxhYmxlIGRpbWVuc2lvbiBvZiB0aGUgYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBkaW1lbnNpb246IHN0cmluZztcblxuICAvKipcbiAgICogUm9sZSB0byB1c2UgZm9yIHNjYWxpbmdcbiAgICovXG4gIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnQgYW4gYXR0cmlidXRlIGZvciB3aGljaCBhdXRvc2NhbGluZyBjYW4gYmUgY29uZmlndXJlZFxuICpcbiAqIFRoaXMgY2xhc3MgaXMgYmFzaWNhbGx5IGEgbGlnaHQgd3JhcHBlciBhcm91bmQgU2NhbGFibGVUYXJnZXQsIGJ1dCB3aXRoXG4gKiBhbGwgbWV0aG9kcyBwcm90ZWN0ZWQgaW5zdGVhZCBvZiBwdWJsaWMgc28gdGhleSBjYW4gYmUgc2VsZWN0aXZlbHlcbiAqIGV4cG9zZWQgYW5kL29yIG1vcmUgc3BlY2lmaWMgdmVyc2lvbnMgb2YgdGhlbSBjYW4gYmUgZXhwb3NlZCBieSBkZXJpdmVkXG4gKiBjbGFzc2VzIGZvciBpbmRpdmlkdWFsIHNlcnZpY2VzIHN1cHBvcnQgYXV0b3NjYWxpbmcuXG4gKlxuICogVHlwaWNhbCB1c2UgY2FzZXM6XG4gKlxuICogLSBIaWRlIGF3YXkgdGhlIFByZWRlZmluZWRNZXRyaWMgZW51bSBmb3IgdGFyZ2V0IHRyYWNraW5nIHBvbGljaWVzLlxuICogLSBEb24ndCBleHBvc2UgYWxsIHNjYWxpbmcgbWV0aG9kcyAoZm9yIGV4YW1wbGUgRHluYW1vIHRhYmxlcyBkb24ndCBzdXBwb3J0XG4gKiAgIFN0ZXAgU2NhbGluZywgc28gdGhlIER5bmFtbyBzdWJjbGFzcyB3b24ndCBleHBvc2UgdGhpcyBtZXRob2QpLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQmFzZVNjYWxhYmxlQXR0cmlidXRlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHByaXZhdGUgdGFyZ2V0OiBTY2FsYWJsZVRhcmdldDtcblxuICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3RlY3RlZCByZWFkb25seSBwcm9wczogQmFzZVNjYWxhYmxlQXR0cmlidXRlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy50YXJnZXQgPSBuZXcgU2NhbGFibGVUYXJnZXQodGhpcywgJ1RhcmdldCcsIHtcbiAgICAgIHNlcnZpY2VOYW1lc3BhY2U6IHRoaXMucHJvcHMuc2VydmljZU5hbWVzcGFjZSxcbiAgICAgIHNjYWxhYmxlRGltZW5zaW9uOiB0aGlzLnByb3BzLmRpbWVuc2lvbixcbiAgICAgIHJlc291cmNlSWQ6IHRoaXMucHJvcHMucmVzb3VyY2VJZCxcbiAgICAgIHJvbGU6IHRoaXMucHJvcHMucm9sZSxcbiAgICAgIG1pbkNhcGFjaXR5OiBwcm9wcy5taW5DYXBhY2l0eSAhPT0gdW5kZWZpbmVkID8gcHJvcHMubWluQ2FwYWNpdHkgOiAxLFxuICAgICAgbWF4Q2FwYWNpdHk6IHByb3BzLm1heENhcGFjaXR5LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFNjYWxlIG91dCBvciBpbiBiYXNlZCBvbiB0aW1lXG4gICAqL1xuICBwcm90ZWN0ZWQgZG9TY2FsZU9uU2NoZWR1bGUoaWQ6IHN0cmluZywgcHJvcHM6IFNjYWxpbmdTY2hlZHVsZSkge1xuICAgIHRoaXMudGFyZ2V0LnNjYWxlT25TY2hlZHVsZShpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFNjYWxlIG91dCBvciBpbiBiYXNlZCBvbiBhIG1ldHJpYyB2YWx1ZVxuICAgKi9cbiAgcHJvdGVjdGVkIGRvU2NhbGVPbk1ldHJpYyhpZDogc3RyaW5nLCBwcm9wczogQmFzaWNTdGVwU2NhbGluZ1BvbGljeVByb3BzKSB7XG4gICAgdGhpcy50YXJnZXQuc2NhbGVPbk1ldHJpYyhpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFNjYWxlIG91dCBvciBpbiBpbiBvcmRlciB0byBrZWVwIGEgbWV0cmljIGFyb3VuZCBhIHRhcmdldCB2YWx1ZVxuICAgKi9cbiAgcHJvdGVjdGVkIGRvU2NhbGVUb1RyYWNrTWV0cmljKGlkOiBzdHJpbmcsIHByb3BzOiBCYXNpY1RhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeVByb3BzKSB7XG4gICAgdGhpcy50YXJnZXQuc2NhbGVUb1RyYWNrTWV0cmljKGlkLCBwcm9wcyk7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBlbmFibGluZyBEeW5hbW9EQiBjYXBhY2l0eSBzY2FsaW5nXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRW5hYmxlU2NhbGluZ1Byb3BzIHtcbiAgLyoqXG4gICAqIE1pbmltdW0gY2FwYWNpdHkgdG8gc2NhbGUgdG9cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgbWluQ2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE1heGltdW0gY2FwYWNpdHkgdG8gc2NhbGUgdG9cbiAgICovXG4gIHJlYWRvbmx5IG1heENhcGFjaXR5OiBudW1iZXI7XG59XG4iXX0=