"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../lib/index");
const sfn = require("@aws-cdk/aws-stepfunctions");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const aws_s3_1 = require("@aws-cdk/aws-s3");
function deployNewStateMachine(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        }
    };
    return new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
}
test('check deployCloudTrail = false', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        deployCloudTrail: false
    };
    const construct = new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
    expect(construct.cloudtrail === undefined);
});
test('override eventRuleProps', () => {
    const stack = new cdk.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket');
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        existingBucketObj: mybucket,
        eventRuleProps: {
            eventPattern: {
                source: ['aws.s3'],
                detailType: ['AWS API Call via CloudTrail'],
                detail: {
                    eventSource: [
                        "s3.amazonaws.com"
                    ],
                    eventName: [
                        "GetObject"
                    ],
                    requestParameters: {
                        bucketName: [
                            mybucket.bucketName
                        ]
                    }
                }
            }
        }
    };
    new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        EventPattern: {
            "source": [
                "aws.s3"
            ],
            "detail-type": [
                "AWS API Call via CloudTrail"
            ],
            "detail": {
                eventSource: [
                    "s3.amazonaws.com"
                ],
                eventName: [
                    "GetObject"
                ],
                requestParameters: {
                    bucketName: [
                        {
                            Ref: "mybucket160F8132"
                        }
                    ]
                }
            }
        },
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "tests3stepfunctiontests3stepfunctionWtests3stepfunctionWeventrulestepfunctionconstructStateMachineAAE00FFE"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "tests3stepfunctiontests3stepfunctionWtests3stepfunctionWeventrulestepfunctionconstructEventsRuleRole1B233B12",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.cloudtrail !== null);
    expect(construct.stateMachine !== null);
    expect(construct.s3Bucket !== null);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachineLogGroup !== null);
    expect(construct.s3LoggingBucket !== null);
    expect(construct.cloudtrail !== null);
    expect(construct.cloudtrailBucket !== null);
    expect(construct.cloudtrailLoggingBucket !== null);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new aws_s3_1.Bucket(stack, 'test-bucket', {});
    const startState = new sfn.Pass(stack, 'StartState');
    const app = () => {
        // Helper declaration
        new index_1.S3ToStepFunction(stack, "bad-s3-args", {
            stateMachineProps: {
                definition: startState
            },
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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