"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy
            },
            ruleSetName: props.ruleSet.receiptRuleSetName
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16'
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        }
                    })
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log("SES Notification:\n", JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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