"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
// Imports
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:aws:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    deployment.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W45',
                    reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
                }]
        }
    };
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    configureCloudwatchRoleForApi(scope, _api);
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    });
    // Return the API object
    return _api;
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    configureCloudwatchRoleForApi(scope, _api);
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    });
    // Return the API
    return _api;
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    return configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    return configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    return configureRestApi(scope, defaultProps, apiGatewayProps);
}
exports.GlobalRestApi = GlobalRestApi;
//# sourceMappingURL=data:application/json;base64,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