"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const lambda_helper_1 = require("./lambda-helper");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    const cfnCfDistribution = cfDistribution.node.defaultChild;
    cfnCfDistribution.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W70',
                    reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
                }]
        }
    };
    return cfDistribution;
}
function createCloudfrontLoggingBucket(scope) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, 'CloudfrontLoggingBucket', s3_bucket_defaults_1.DefaultS3Props());
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                }, {
                    id: 'W51',
                    reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                }]
        }
    };
    return loggingBucket;
}
// Lambda@Edge function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultLambdaEdgeFunction(scope) {
    const edgeLambdaFunc = lambda_helper_1.deployLambdaFunction(scope, {
        code: new lambda.InlineCode("exports.handler = (event, context, callback) => { \
          const response = event.Records[0].cf.response; \
          const headers = response.headers; \
          headers['x-xss-protection'] = [ \
            { \
              key: 'X-XSS-Protection', \
              value: '1; mode=block' \
            } \
          ]; \
          headers['x-frame-options'] = [ \
            { \
              key: 'X-Frame-Options', \
              value: 'DENY' \
            } \
          ]; \
          headers['x-content-type-options'] = [ \
            { \
              key: 'X-Content-Type-Options', \
              value: 'nosniff' \
            } \
          ]; \
          headers['strict-transport-security'] = [ \
            { \
              key: 'Strict-Transport-Security', \
              value: 'max-age=63072000; includeSubdomains; preload' \
            } \
          ]; \
          headers['referrer-policy'] = [ \
            { \
              key: 'Referrer-Policy', \
              value: 'same-origin' \
            } \
          ]; \
          headers['content-security-policy'] = [ \
            { \
              key: 'Content-Security-Policy', \
              value: \"default-src 'none'; base-uri 'self'; img-src 'self'; script-src 'self'; style-src 'self' https:; object-src 'none'; frame-ancestors 'none'; font-src 'self' https:; form-action 'self'; manifest-src 'self'; connect-src 'self'\" \
             } \
          ]; \
          callback(null, response); \
        };"),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    }, 'SetHttpSecurityHeaders');
    // Remove any environment variables as it is not supported by the Lambda@Edge functions :(
    const cfnEdgeLambdaFunction = edgeLambdaFunc.node.defaultChild;
    cfnEdgeLambdaFunction.addDeletionOverride('Properties.Environment');
    return edgeLambdaFunc;
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders) {
    const _httpSecurityHeaders = httpSecurityHeaders ? httpSecurityHeaders : true;
    let defaultprops;
    let edgeLambdaVersion;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingConfig) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, cloudFrontDistributionProps.loggingConfig.bucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        const loggingBucket = createCloudfrontLoggingBucket(scope);
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return cfDistribution;
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders) {
    // Create CloudFront Origin Access Identity User
    const cfnOrigAccessId = new cloudfront.CfnCloudFrontOriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        cloudFrontOriginAccessIdentityConfig: {
            comment: 'Access S3 bucket content only through CloudFront'
        }
    });
    const oaiImported = cloudfront.OriginAccessIdentity.fromOriginAccessIdentityName(scope, 'OAIImported', cfnOrigAccessId.ref);
    let defaultprops;
    let edgeLambdaVersion;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingConfig) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, cloudFrontDistributionProps.loggingConfig.bucket, oaiImported, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        const loggingBucket = createCloudfrontLoggingBucket(scope);
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, oaiImported, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Add S3 Bucket Policy to allow s3:GetObject for CloudFront Origin Access Identity User
    sourceBucket.addToResourcePolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject'],
        resources: [sourceBucket.arnForObjects('*')],
        principals: [new iam.CanonicalUserPrincipal(cfnOrigAccessId.attrS3CanonicalUserId)]
    }));
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    const sourceBucketPolicy = bucketPolicy.node.findChild('Resource');
    sourceBucketPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'F16',
                    reason: `Public website bucket policy requires a wildcard principal`
                }]
        }
    };
    return cfDistribution;
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRmcm9udC1kaXN0cmlidXRpb24taGVscGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2xvdWRmcm9udC1kaXN0cmlidXRpb24taGVscGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7O0FBRUgsc0RBQXNEO0FBQ3RELHNDQUFzQztBQUV0Qyx3Q0FBd0M7QUFFeEMsOENBQThDO0FBQzlDLDZEQUFzRDtBQUN0RCx5RkFBb0o7QUFDcEosbUNBQXdDO0FBQ3hDLG1EQUF1RDtBQUV2RCxvR0FBb0c7QUFDcEcsU0FBUyxvQkFBb0IsQ0FBQyxjQUFvRDtJQUM5RSxNQUFNLGlCQUFpQixHQUFHLGNBQWMsQ0FBQyxJQUFJLENBQUMsWUFBMEMsQ0FBQztJQUN6RixpQkFBaUIsQ0FBQyxVQUFVLENBQUMsUUFBUSxHQUFHO1FBQ3BDLE9BQU8sRUFBRTtZQUNMLGlCQUFpQixFQUFFLENBQUM7b0JBQ2hCLEVBQUUsRUFBRSxLQUFLO29CQUNULE1BQU0sRUFBRSxzS0FBc0s7aUJBQ2pMLENBQUM7U0FDTDtLQUNKLENBQUM7SUFDRixPQUFPLGNBQWMsQ0FBQztBQUMxQixDQUFDO0FBRUQsU0FBUyw2QkFBNkIsQ0FBQyxLQUFvQjtJQUN2RCw0QkFBNEI7SUFDNUIsTUFBTSxhQUFhLEdBQWMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSx5QkFBeUIsRUFBRSxtQ0FBYyxFQUFFLENBQUMsQ0FBQztJQUVuRywrQ0FBK0M7SUFDL0MsTUFBTSxxQkFBcUIsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQWlCLENBQUM7SUFFdkYsK0VBQStFO0lBQy9FLHFCQUFxQixDQUFDLG1CQUFtQixDQUFDLGVBQWUsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO0lBQy9FLHFCQUFxQixDQUFDLFVBQVUsQ0FBQyxRQUFRLEdBQUc7UUFDeEMsT0FBTyxFQUFFO1lBQ0wsaUJBQWlCLEVBQUUsQ0FBQztvQkFDaEIsRUFBRSxFQUFFLEtBQUs7b0JBQ1QsTUFBTSxFQUFFLGlGQUFpRjtpQkFDNUYsRUFBRTtvQkFDQyxFQUFFLEVBQUUsS0FBSztvQkFDVCxNQUFNLEVBQUUsaUZBQWlGO2lCQUM1RixDQUFDO1NBQ0w7S0FDSixDQUFDO0lBRUYsT0FBTyxhQUFhLENBQUM7QUFDekIsQ0FBQztBQUVELDRHQUE0RztBQUM1RyxzQ0FBc0M7QUFDdEMsU0FBUyx5QkFBeUIsQ0FBQyxLQUFvQjtJQUNuRCxNQUFNLGNBQWMsR0FBb0Isb0NBQW9CLENBQUMsS0FBSyxFQUFFO1FBQ2hFLElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7V0F3Q3pCLENBQUM7UUFDSixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQzNCLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztJQUU3QiwwRkFBMEY7SUFDMUYsTUFBTSxxQkFBcUIsR0FBRyxjQUFjLENBQUMsSUFBSSxDQUFDLFlBQWtDLENBQUM7SUFDckYscUJBQXFCLENBQUMsbUJBQW1CLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUVwRSxPQUFPLGNBQWMsQ0FBQztBQUMxQixDQUFDO0FBRUQsU0FBZ0IsbUNBQW1DLENBQUMsS0FBb0IsRUFDcEIsV0FBd0IsRUFDeEIsMkJBQTZFLEVBQzdFLG1CQUE2QjtJQUU3RSxNQUFNLG9CQUFvQixHQUFHLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO0lBQzlFLElBQUksWUFBdUQsQ0FBQztJQUM1RCxJQUFJLGlCQUFpQixDQUFDO0lBRXRCLElBQUksb0JBQW9CLEVBQUU7UUFDdEIsaUJBQWlCLEdBQUcsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSwrQkFBK0IsRUFBRTtZQUMzRSxNQUFNLEVBQUUseUJBQXlCLENBQUMsS0FBSyxDQUFDO1NBQzNDLENBQUMsQ0FBQztLQUNOO0lBRUQsSUFBSSwyQkFBMkIsSUFBSSwyQkFBMkIsQ0FBQyxhQUFhLEVBQUU7UUFDMUUsWUFBWSxHQUFHLHFGQUFrRCxDQUFDLFdBQVcsRUFDekUsMkJBQTJCLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxvQkFBb0IsRUFDdEUsaUJBQWlCLENBQUMsQ0FBQztLQUMxQjtTQUFNO1FBQ0gsTUFBTSxhQUFhLEdBQUcsNkJBQTZCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0QsWUFBWSxHQUFHLHFGQUFrRCxDQUFDLFdBQVcsRUFDekUsYUFBYSxFQUFFLG9CQUFvQixFQUNuQyxpQkFBaUIsQ0FBQyxDQUFDO0tBQzFCO0lBRUQsTUFBTSxPQUFPLEdBQUcsMkJBQTJCLENBQUMsQ0FBQyxDQUFDLHFCQUFhLENBQUMsWUFBWSxFQUFFLDJCQUEyQixDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQztJQUN0SCxxQ0FBcUM7SUFDckMsTUFBTSxjQUFjLEdBQXlDLElBQUksVUFBVSxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSx3QkFBd0IsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNoSixvQkFBb0IsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUVyQyxPQUFPLGNBQWMsQ0FBQztBQUMxQixDQUFDO0FBaENELGtGQWdDQztBQUVELFNBQWdCLDJCQUEyQixDQUFDLEtBQW9CLEVBQ3BCLFlBQXVCLEVBQ3ZCLDJCQUE2RSxFQUM3RSxtQkFBNkI7SUFFckUsZ0RBQWdEO0lBQ2hELE1BQU0sZUFBZSxHQUFHLElBQUksVUFBVSxDQUFDLGlDQUFpQyxDQUFDLEtBQUssRUFBRSxnQ0FBZ0MsRUFBRTtRQUM5RyxvQ0FBb0MsRUFBRTtZQUNsQyxPQUFPLEVBQUUsa0RBQWtEO1NBQzlEO0tBQ0osQ0FBQyxDQUFDO0lBRUgsTUFBTSxXQUFXLEdBQUcsVUFBVSxDQUFDLG9CQUFvQixDQUFDLDRCQUE0QixDQUM1RSxLQUFLLEVBQ0wsYUFBYSxFQUNiLGVBQWUsQ0FBQyxHQUFHLENBQ3RCLENBQUM7SUFFRixJQUFJLFlBQXVELENBQUM7SUFDNUQsSUFBSSxpQkFBaUIsQ0FBQztJQUN0QixNQUFNLG9CQUFvQixHQUFHLENBQUMsbUJBQW1CLEtBQUssU0FBUyxJQUFJLG1CQUFtQixLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztJQUVqSCxJQUFJLG9CQUFvQixFQUFFO1FBQ3RCLGlCQUFpQixHQUFHLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsK0JBQStCLEVBQUU7WUFDM0UsTUFBTSxFQUFFLHlCQUF5QixDQUFDLEtBQUssQ0FBQztTQUMzQyxDQUFDLENBQUM7S0FDTjtJQUVELElBQUksMkJBQTJCLElBQUksMkJBQTJCLENBQUMsYUFBYSxFQUFFO1FBQzFFLFlBQVksR0FBRyw2RUFBMEMsQ0FBQyxZQUFZLEVBQ2xFLDJCQUEyQixDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLG9CQUFvQixFQUNuRixpQkFBaUIsQ0FBQyxDQUFDO0tBQzFCO1NBQU07UUFDSCxNQUFNLGFBQWEsR0FBRyw2QkFBNkIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUMzRCxZQUFZLEdBQUcsNkVBQTBDLENBQUMsWUFBWSxFQUFFLGFBQWEsRUFDakYsV0FBVyxFQUFFLG9CQUFvQixFQUNqQyxpQkFBaUIsQ0FBQyxDQUFDO0tBQzFCO0lBRUQsTUFBTSxPQUFPLEdBQUcsMkJBQTJCLENBQUMsQ0FBQyxDQUFDLHFCQUFhLENBQUMsWUFBWSxFQUFFLDJCQUEyQixDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQztJQUN0SCxxQ0FBcUM7SUFDckMsTUFBTSxjQUFjLEdBQXlDLElBQUksVUFBVSxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSx3QkFBd0IsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNoSixvQkFBb0IsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUVyQyx3RkFBd0Y7SUFDeEYsWUFBWSxDQUFDLG1CQUFtQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztRQUNyRCxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7UUFDekIsU0FBUyxFQUFFLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUM1QyxVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxlQUFlLENBQUMscUJBQXFCLENBQUMsQ0FBQztLQUN0RixDQUFDLENBQUMsQ0FBQztJQUVKLG9EQUFvRDtJQUNwRCxNQUFNLFlBQVksR0FBRyxZQUFZLENBQUMsTUFBeUIsQ0FBQztJQUM1RCxNQUFNLGtCQUFrQixHQUFHLFlBQVksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBdUIsQ0FBQztJQUN6RixrQkFBa0IsQ0FBQyxVQUFVLENBQUMsUUFBUSxHQUFHO1FBQ3JDLE9BQU8sRUFBRTtZQUNMLGlCQUFpQixFQUFFLENBQUM7b0JBQ2hCLEVBQUUsRUFBRSxLQUFLO29CQUNULE1BQU0sRUFBRSw0REFBNEQ7aUJBQ3ZFLENBQUM7U0FDTDtLQUNKLENBQUM7SUFDRixPQUFPLGNBQWMsQ0FBQztBQUMxQixDQUFDO0FBL0RELGtFQStEQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDE5IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgY2xvdWRmcm9udCBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWRmcm9udCc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgYXBpIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IERlZmF1bHRTM1Byb3BzIH0gZnJvbSAnLi9zMy1idWNrZXQtZGVmYXVsdHMnO1xuaW1wb3J0IHsgRGVmYXVsdENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Gb3JTM1Byb3BzLCBEZWZhdWx0Q2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbkZvckFwaUdhdGV3YXlQcm9wcyB9IGZyb20gJy4vY2xvdWRmcm9udC1kaXN0cmlidXRpb24tZGVmYXVsdHMnO1xuaW1wb3J0IHsgb3ZlcnJpZGVQcm9wcyB9IGZyb20gJy4vdXRpbHMnO1xuaW1wb3J0IHsgZGVwbG95TGFtYmRhRnVuY3Rpb24gfSBmcm9tICcuL2xhbWJkYS1oZWxwZXInO1xuXG4vLyBPdmVycmlkZSBDZm5fTmFnIHJ1bGU6IENsb3VkZnJvbnQgVExTLTEuMiBydWxlIChodHRwczovL2dpdGh1Yi5jb20vc3RlbGxpZ2VudC9jZm5fbmFnL2lzc3Vlcy8zODQpXG5mdW5jdGlvbiB1cGRhdGVTZWN1cml0eVBvbGljeShjZkRpc3RyaWJ1dGlvbjogY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKSB7XG4gICAgY29uc3QgY2ZuQ2ZEaXN0cmlidXRpb24gPSBjZkRpc3RyaWJ1dGlvbi5ub2RlLmRlZmF1bHRDaGlsZCBhcyBjbG91ZGZyb250LkNmbkRpc3RyaWJ1dGlvbjtcbiAgICBjZm5DZkRpc3RyaWJ1dGlvbi5jZm5PcHRpb25zLm1ldGFkYXRhID0ge1xuICAgICAgICBjZm5fbmFnOiB7XG4gICAgICAgICAgICBydWxlc190b19zdXBwcmVzczogW3tcbiAgICAgICAgICAgICAgICBpZDogJ1c3MCcsXG4gICAgICAgICAgICAgICAgcmVhc29uOiBgU2luY2UgdGhlIGRpc3RyaWJ1dGlvbiB1c2VzIHRoZSBDbG91ZEZyb250IGRvbWFpbiBuYW1lLCBDbG91ZEZyb250IGF1dG9tYXRpY2FsbHkgc2V0cyB0aGUgc2VjdXJpdHkgcG9saWN5IHRvIFRMU3YxIHJlZ2FyZGxlc3Mgb2YgdGhlIHZhbHVlIG9mIE1pbmltdW1Qcm90b2NvbFZlcnNpb25gXG4gICAgICAgICAgICB9XVxuICAgICAgICB9XG4gICAgfTtcbiAgICByZXR1cm4gY2ZEaXN0cmlidXRpb247XG59XG5cbmZ1bmN0aW9uIGNyZWF0ZUNsb3VkZnJvbnRMb2dnaW5nQnVja2V0KHNjb3BlOiBjZGsuQ29uc3RydWN0KTogczMuQnVja2V0IHtcbiAgICAvLyBDcmVhdGUgdGhlIExvZ2dpbmcgQnVja2V0XG4gICAgY29uc3QgbG9nZ2luZ0J1Y2tldDogczMuQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzY29wZSwgJ0Nsb3VkZnJvbnRMb2dnaW5nQnVja2V0JywgRGVmYXVsdFMzUHJvcHMoKSk7XG5cbiAgICAvLyBFeHRyYWN0IHRoZSBDZm5CdWNrZXQgZnJvbSB0aGUgbG9nZ2luZ0J1Y2tldFxuICAgIGNvbnN0IGxvZ2dpbmdCdWNrZXRSZXNvdXJjZSA9IGxvZ2dpbmdCdWNrZXQubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgczMuQ2ZuQnVja2V0O1xuXG4gICAgLy8gT3ZlcnJpZGUgYWNjZXNzQ29udHJvbCBjb25maWd1cmF0aW9uIGFuZCBhZGQgbWV0YWRhdGEgZm9yIHRoZSBsb2dnaW5nIGJ1Y2tldFxuICAgIGxvZ2dpbmdCdWNrZXRSZXNvdXJjZS5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdBY2Nlc3NDb250cm9sJywgJ0xvZ0RlbGl2ZXJ5V3JpdGUnKTtcbiAgICBsb2dnaW5nQnVja2V0UmVzb3VyY2UuY2ZuT3B0aW9ucy5tZXRhZGF0YSA9IHtcbiAgICAgICAgY2ZuX25hZzoge1xuICAgICAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgICAgICAgICAgaWQ6ICdXMzUnLFxuICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25gXG4gICAgICAgICAgICB9LCB7XG4gICAgICAgICAgICAgICAgaWQ6ICdXNTEnLFxuICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25gXG4gICAgICAgICAgICB9XVxuICAgICAgICB9XG4gICAgfTtcblxuICAgIHJldHVybiBsb2dnaW5nQnVja2V0O1xufVxuXG4vLyBMYW1iZGFARWRnZSBmdW5jdGlvbiB0byBpbnNlcnQgdGhlIEhUVFAgU2VjdXJpdHkgSGVhZGVycyBpbnRvIHRoZSByZXNwb25zZSBjb21pbmcgZnJvbSB0aGUgb3JpZ2luIHNlcnZlcnNcbi8vIGFuZCBiZWZvcmUgaXQgaXMgc2VudCB0byB0aGUgY2xpZW50XG5mdW5jdGlvbiBkZWZhdWx0TGFtYmRhRWRnZUZ1bmN0aW9uKHNjb3BlOiBjZGsuQ29uc3RydWN0KTogbGFtYmRhLkZ1bmN0aW9uIHtcbiAgICBjb25zdCBlZGdlTGFtYmRhRnVuYzogbGFtYmRhLkZ1bmN0aW9uID0gZGVwbG95TGFtYmRhRnVuY3Rpb24oc2NvcGUsIHtcbiAgICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKFwiZXhwb3J0cy5oYW5kbGVyID0gKGV2ZW50LCBjb250ZXh0LCBjYWxsYmFjaykgPT4geyBcXFxuICAgICAgICAgIGNvbnN0IHJlc3BvbnNlID0gZXZlbnQuUmVjb3Jkc1swXS5jZi5yZXNwb25zZTsgXFxcbiAgICAgICAgICBjb25zdCBoZWFkZXJzID0gcmVzcG9uc2UuaGVhZGVyczsgXFxcbiAgICAgICAgICBoZWFkZXJzWyd4LXhzcy1wcm90ZWN0aW9uJ10gPSBbIFxcXG4gICAgICAgICAgICB7IFxcXG4gICAgICAgICAgICAgIGtleTogJ1gtWFNTLVByb3RlY3Rpb24nLCBcXFxuICAgICAgICAgICAgICB2YWx1ZTogJzE7IG1vZGU9YmxvY2snIFxcXG4gICAgICAgICAgICB9IFxcXG4gICAgICAgICAgXTsgXFxcbiAgICAgICAgICBoZWFkZXJzWyd4LWZyYW1lLW9wdGlvbnMnXSA9IFsgXFxcbiAgICAgICAgICAgIHsgXFxcbiAgICAgICAgICAgICAga2V5OiAnWC1GcmFtZS1PcHRpb25zJywgXFxcbiAgICAgICAgICAgICAgdmFsdWU6ICdERU5ZJyBcXFxuICAgICAgICAgICAgfSBcXFxuICAgICAgICAgIF07IFxcXG4gICAgICAgICAgaGVhZGVyc1sneC1jb250ZW50LXR5cGUtb3B0aW9ucyddID0gWyBcXFxuICAgICAgICAgICAgeyBcXFxuICAgICAgICAgICAgICBrZXk6ICdYLUNvbnRlbnQtVHlwZS1PcHRpb25zJywgXFxcbiAgICAgICAgICAgICAgdmFsdWU6ICdub3NuaWZmJyBcXFxuICAgICAgICAgICAgfSBcXFxuICAgICAgICAgIF07IFxcXG4gICAgICAgICAgaGVhZGVyc1snc3RyaWN0LXRyYW5zcG9ydC1zZWN1cml0eSddID0gWyBcXFxuICAgICAgICAgICAgeyBcXFxuICAgICAgICAgICAgICBrZXk6ICdTdHJpY3QtVHJhbnNwb3J0LVNlY3VyaXR5JywgXFxcbiAgICAgICAgICAgICAgdmFsdWU6ICdtYXgtYWdlPTYzMDcyMDAwOyBpbmNsdWRlU3ViZG9tYWluczsgcHJlbG9hZCcgXFxcbiAgICAgICAgICAgIH0gXFxcbiAgICAgICAgICBdOyBcXFxuICAgICAgICAgIGhlYWRlcnNbJ3JlZmVycmVyLXBvbGljeSddID0gWyBcXFxuICAgICAgICAgICAgeyBcXFxuICAgICAgICAgICAgICBrZXk6ICdSZWZlcnJlci1Qb2xpY3knLCBcXFxuICAgICAgICAgICAgICB2YWx1ZTogJ3NhbWUtb3JpZ2luJyBcXFxuICAgICAgICAgICAgfSBcXFxuICAgICAgICAgIF07IFxcXG4gICAgICAgICAgaGVhZGVyc1snY29udGVudC1zZWN1cml0eS1wb2xpY3knXSA9IFsgXFxcbiAgICAgICAgICAgIHsgXFxcbiAgICAgICAgICAgICAga2V5OiAnQ29udGVudC1TZWN1cml0eS1Qb2xpY3knLCBcXFxuICAgICAgICAgICAgICB2YWx1ZTogXFxcImRlZmF1bHQtc3JjICdub25lJzsgYmFzZS11cmkgJ3NlbGYnOyBpbWctc3JjICdzZWxmJzsgc2NyaXB0LXNyYyAnc2VsZic7IHN0eWxlLXNyYyAnc2VsZicgaHR0cHM6OyBvYmplY3Qtc3JjICdub25lJzsgZnJhbWUtYW5jZXN0b3JzICdub25lJzsgZm9udC1zcmMgJ3NlbGYnIGh0dHBzOjsgZm9ybS1hY3Rpb24gJ3NlbGYnOyBtYW5pZmVzdC1zcmMgJ3NlbGYnOyBjb25uZWN0LXNyYyAnc2VsZidcXFwiIFxcXG4gICAgICAgICAgICAgfSBcXFxuICAgICAgICAgIF07IFxcXG4gICAgICAgICAgY2FsbGJhY2sobnVsbCwgcmVzcG9uc2UpOyBcXFxuICAgICAgICB9O1wiKSxcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sICdTZXRIdHRwU2VjdXJpdHlIZWFkZXJzJyk7XG5cbiAgICAvLyBSZW1vdmUgYW55IGVudmlyb25tZW50IHZhcmlhYmxlcyBhcyBpdCBpcyBub3Qgc3VwcG9ydGVkIGJ5IHRoZSBMYW1iZGFARWRnZSBmdW5jdGlvbnMgOihcbiAgICBjb25zdCBjZm5FZGdlTGFtYmRhRnVuY3Rpb24gPSBlZGdlTGFtYmRhRnVuYy5ub2RlLmRlZmF1bHRDaGlsZCBhcyBsYW1iZGEuQ2ZuRnVuY3Rpb247XG4gICAgY2ZuRWRnZUxhbWJkYUZ1bmN0aW9uLmFkZERlbGV0aW9uT3ZlcnJpZGUoJ1Byb3BlcnRpZXMuRW52aXJvbm1lbnQnKTtcblxuICAgIHJldHVybiBlZGdlTGFtYmRhRnVuYztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5KHNjb3BlOiBjZGsuQ29uc3RydWN0LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGFwaUVuZFBvaW50OiBhcGkuUmVzdEFwaSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHM/OiBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Qcm9wcyB8IGFueSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBodHRwU2VjdXJpdHlIZWFkZXJzPzogYm9vbGVhbik6IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiB7XG5cbiAgICBjb25zdCBfaHR0cFNlY3VyaXR5SGVhZGVycyA9IGh0dHBTZWN1cml0eUhlYWRlcnMgPyBodHRwU2VjdXJpdHlIZWFkZXJzIDogdHJ1ZTtcbiAgICBsZXQgZGVmYXVsdHByb3BzOiBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Qcm9wcztcbiAgICBsZXQgZWRnZUxhbWJkYVZlcnNpb247XG5cbiAgICBpZiAoX2h0dHBTZWN1cml0eUhlYWRlcnMpIHtcbiAgICAgICAgZWRnZUxhbWJkYVZlcnNpb24gPSBuZXcgbGFtYmRhLlZlcnNpb24oc2NvcGUsIFwiU2V0SHR0cFNlY3VyaXR5SGVhZGVyc1ZlcnNpb25cIiwge1xuICAgICAgICAgICAgbGFtYmRhOiBkZWZhdWx0TGFtYmRhRWRnZUZ1bmN0aW9uKHNjb3BlKVxuICAgICAgICB9KTtcbiAgICB9XG5cbiAgICBpZiAoY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzICYmIGNsb3VkRnJvbnREaXN0cmlidXRpb25Qcm9wcy5sb2dnaW5nQ29uZmlnKSB7XG4gICAgICAgIGRlZmF1bHRwcm9wcyA9IERlZmF1bHRDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uRm9yQXBpR2F0ZXdheVByb3BzKGFwaUVuZFBvaW50LFxuICAgICAgICAgICAgY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzLmxvZ2dpbmdDb25maWcuYnVja2V0LCBfaHR0cFNlY3VyaXR5SGVhZGVycyxcbiAgICAgICAgICAgIGVkZ2VMYW1iZGFWZXJzaW9uKTtcbiAgICB9IGVsc2Uge1xuICAgICAgICBjb25zdCBsb2dnaW5nQnVja2V0ID0gY3JlYXRlQ2xvdWRmcm9udExvZ2dpbmdCdWNrZXQoc2NvcGUpO1xuICAgICAgICBkZWZhdWx0cHJvcHMgPSBEZWZhdWx0Q2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbkZvckFwaUdhdGV3YXlQcm9wcyhhcGlFbmRQb2ludCxcbiAgICAgICAgICAgIGxvZ2dpbmdCdWNrZXQsIF9odHRwU2VjdXJpdHlIZWFkZXJzLFxuICAgICAgICAgICAgZWRnZUxhbWJkYVZlcnNpb24pO1xuICAgIH1cblxuICAgIGNvbnN0IGNmcHJvcHMgPSBjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHMgPyBvdmVycmlkZVByb3BzKGRlZmF1bHRwcm9wcywgY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzKSA6IGRlZmF1bHRwcm9wcztcbiAgICAvLyBDcmVhdGUgdGhlIENsb3VkZnJvbnQgRGlzdHJpYnV0aW9uXG4gICAgY29uc3QgY2ZEaXN0cmlidXRpb246IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiA9IG5ldyBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24oc2NvcGUsICdDbG91ZEZyb250RGlzdHJpYnV0aW9uJywgY2Zwcm9wcyk7XG4gICAgdXBkYXRlU2VjdXJpdHlQb2xpY3koY2ZEaXN0cmlidXRpb24pO1xuXG4gICAgcmV0dXJuIGNmRGlzdHJpYnV0aW9uO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvclMzKHNjb3BlOiBjZGsuQ29uc3RydWN0LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBzb3VyY2VCdWNrZXQ6IHMzLkJ1Y2tldCxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzPzogY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMgfCBhbnksXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGh0dHBTZWN1cml0eUhlYWRlcnM/OiBib29sZWFuKTogY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uIHtcblxuICAgIC8vIENyZWF0ZSBDbG91ZEZyb250IE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgVXNlclxuICAgIGNvbnN0IGNmbk9yaWdBY2Nlc3NJZCA9IG5ldyBjbG91ZGZyb250LkNmbkNsb3VkRnJvbnRPcmlnaW5BY2Nlc3NJZGVudGl0eShzY29wZSwgJ0Nsb3VkRnJvbnRPcmlnaW5BY2Nlc3NJZGVudGl0eScsIHtcbiAgICAgICAgY2xvdWRGcm9udE9yaWdpbkFjY2Vzc0lkZW50aXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBjb21tZW50OiAnQWNjZXNzIFMzIGJ1Y2tldCBjb250ZW50IG9ubHkgdGhyb3VnaCBDbG91ZEZyb250J1xuICAgICAgICB9XG4gICAgfSk7XG5cbiAgICBjb25zdCBvYWlJbXBvcnRlZCA9IGNsb3VkZnJvbnQuT3JpZ2luQWNjZXNzSWRlbnRpdHkuZnJvbU9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZShcbiAgICAgICAgc2NvcGUsXG4gICAgICAgICdPQUlJbXBvcnRlZCcsXG4gICAgICAgIGNmbk9yaWdBY2Nlc3NJZC5yZWZcbiAgICApO1xuXG4gICAgbGV0IGRlZmF1bHRwcm9wczogY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHM7XG4gICAgbGV0IGVkZ2VMYW1iZGFWZXJzaW9uO1xuICAgIGNvbnN0IF9odHRwU2VjdXJpdHlIZWFkZXJzID0gKGh0dHBTZWN1cml0eUhlYWRlcnMgIT09IHVuZGVmaW5lZCAmJiBodHRwU2VjdXJpdHlIZWFkZXJzID09PSBmYWxzZSkgPyBmYWxzZSA6IHRydWU7XG5cbiAgICBpZiAoX2h0dHBTZWN1cml0eUhlYWRlcnMpIHtcbiAgICAgICAgZWRnZUxhbWJkYVZlcnNpb24gPSBuZXcgbGFtYmRhLlZlcnNpb24oc2NvcGUsIFwiU2V0SHR0cFNlY3VyaXR5SGVhZGVyc1ZlcnNpb25cIiwge1xuICAgICAgICAgICAgbGFtYmRhOiBkZWZhdWx0TGFtYmRhRWRnZUZ1bmN0aW9uKHNjb3BlKVxuICAgICAgICB9KTtcbiAgICB9XG5cbiAgICBpZiAoY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzICYmIGNsb3VkRnJvbnREaXN0cmlidXRpb25Qcm9wcy5sb2dnaW5nQ29uZmlnKSB7XG4gICAgICAgIGRlZmF1bHRwcm9wcyA9IERlZmF1bHRDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uRm9yUzNQcm9wcyhzb3VyY2VCdWNrZXQsXG4gICAgICAgICAgICBjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHMubG9nZ2luZ0NvbmZpZy5idWNrZXQsIG9haUltcG9ydGVkLCBfaHR0cFNlY3VyaXR5SGVhZGVycyxcbiAgICAgICAgICAgIGVkZ2VMYW1iZGFWZXJzaW9uKTtcbiAgICB9IGVsc2Uge1xuICAgICAgICBjb25zdCBsb2dnaW5nQnVja2V0ID0gY3JlYXRlQ2xvdWRmcm9udExvZ2dpbmdCdWNrZXQoc2NvcGUpO1xuICAgICAgICBkZWZhdWx0cHJvcHMgPSBEZWZhdWx0Q2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbkZvclMzUHJvcHMoc291cmNlQnVja2V0LCBsb2dnaW5nQnVja2V0LFxuICAgICAgICAgICAgb2FpSW1wb3J0ZWQsIF9odHRwU2VjdXJpdHlIZWFkZXJzLFxuICAgICAgICAgICAgZWRnZUxhbWJkYVZlcnNpb24pO1xuICAgIH1cblxuICAgIGNvbnN0IGNmcHJvcHMgPSBjbG91ZEZyb250RGlzdHJpYnV0aW9uUHJvcHMgPyBvdmVycmlkZVByb3BzKGRlZmF1bHRwcm9wcywgY2xvdWRGcm9udERpc3RyaWJ1dGlvblByb3BzKSA6IGRlZmF1bHRwcm9wcztcbiAgICAvLyBDcmVhdGUgdGhlIENsb3VkZnJvbnQgRGlzdHJpYnV0aW9uXG4gICAgY29uc3QgY2ZEaXN0cmlidXRpb246IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiA9IG5ldyBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24oc2NvcGUsICdDbG91ZEZyb250RGlzdHJpYnV0aW9uJywgY2Zwcm9wcyk7XG4gICAgdXBkYXRlU2VjdXJpdHlQb2xpY3koY2ZEaXN0cmlidXRpb24pO1xuXG4gICAgLy8gQWRkIFMzIEJ1Y2tldCBQb2xpY3kgdG8gYWxsb3cgczM6R2V0T2JqZWN0IGZvciBDbG91ZEZyb250IE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgVXNlclxuICAgIHNvdXJjZUJ1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbc291cmNlQnVja2V0LmFybkZvck9iamVjdHMoJyonKV0sXG4gICAgICAgIHByaW5jaXBhbHM6IFtuZXcgaWFtLkNhbm9uaWNhbFVzZXJQcmluY2lwYWwoY2ZuT3JpZ0FjY2Vzc0lkLmF0dHJTM0Nhbm9uaWNhbFVzZXJJZCldXG4gICAgfSkpO1xuXG4gICAgLy8gRXh0cmFjdCB0aGUgQ2ZuQnVja2V0UG9saWN5IGZyb20gdGhlIHNvdXJjZUJ1Y2tldFxuICAgIGNvbnN0IGJ1Y2tldFBvbGljeSA9IHNvdXJjZUJ1Y2tldC5wb2xpY3kgYXMgczMuQnVja2V0UG9saWN5O1xuICAgIGNvbnN0IHNvdXJjZUJ1Y2tldFBvbGljeSA9IGJ1Y2tldFBvbGljeS5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBzMy5DZm5CdWNrZXRQb2xpY3k7XG4gICAgc291cmNlQnVja2V0UG9saWN5LmNmbk9wdGlvbnMubWV0YWRhdGEgPSB7XG4gICAgICAgIGNmbl9uYWc6IHtcbiAgICAgICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbe1xuICAgICAgICAgICAgICAgIGlkOiAnRjE2JyxcbiAgICAgICAgICAgICAgICByZWFzb246IGBQdWJsaWMgd2Vic2l0ZSBidWNrZXQgcG9saWN5IHJlcXVpcmVzIGEgd2lsZGNhcmQgcHJpbmNpcGFsYFxuICAgICAgICAgICAgfV1cbiAgICAgICAgfVxuICAgIH07XG4gICAgcmV0dXJuIGNmRGlzdHJpYnV0aW9uO1xufVxuIl19