"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
const elasticsearch = require("@aws-cdk/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildElasticSearch(scope, domainName, options, cfnDomainProps) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, 'CognitoKibanaConfigureRole', {
        assumedBy: new iam.ServicePrincipal('es.amazonaws.com')
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, 'CognitoKibanaConfigureRolePolicy', {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig"
                ],
                resources: [
                    options.userpool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${options.identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`
                ]
            }),
            new iam.PolicyStatement({
                actions: [
                    "iam:PassRole"
                ],
                conditions: {
                    StringLike: { 'iam:PassedToService': 'cognito-identity.amazonaws.com' }
                },
                resources: [
                    cognitoKibanaConfigureRole.roleArn
                ]
            })
        ]
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    let _cfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(domainName, cognitoKibanaConfigureRole, options);
    if (cfnDomainProps) {
        _cfnDomainProps = utils_1.overrideProps(_cfnDomainProps, cfnDomainProps);
    }
    const esDomain = new elasticsearch.CfnDomain(scope, 'ElasticsearchDomain', _cfnDomainProps);
    esDomain.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W28',
                    reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`
                }]
        }
    };
    return esDomain;
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace'
        }),
        threshold: 2000,
        evaluationPeriods: 1,
        statistic: 'Minimum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization'
        }),
        threshold: 80,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure'
        }),
        threshold: 80,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization'
        }),
        threshold: 50,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure'
        }),
        threshold: 50,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWxhc3RpY3NlYXJjaC1oZWxwZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJlbGFzdGljc2VhcmNoLWhlbHBlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUVILDREQUE0RDtBQUM1RCxxRUFBbUY7QUFDbkYsbUNBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsc0RBQXNEO0FBRXRELFNBQWdCLGtCQUFrQixDQUFDLEtBQW9CLEVBQUUsVUFBa0IsRUFDeEMsT0FBeUIsRUFBRSxjQUE2QztJQUV6RyxzRkFBc0Y7SUFDdEYsTUFBTSwwQkFBMEIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLDRCQUE0QixFQUFFO1FBQ25GLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQztLQUN4RCxDQUFDLENBQUM7SUFFSCxNQUFNLGdDQUFnQyxHQUFHLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsa0NBQWtDLEVBQUU7UUFDakcsVUFBVSxFQUFFO1lBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AsOEJBQThCO29CQUM5QixrQ0FBa0M7b0JBQ2xDLGtDQUFrQztvQkFDbEMsb0NBQW9DO29CQUNwQywrQkFBK0I7b0JBQy9CLG9DQUFvQztvQkFDcEMsaUNBQWlDO29CQUNqQyx1Q0FBdUM7b0JBQ3ZDLHFDQUFxQztvQkFDckMsdUNBQXVDO29CQUN2Qyx1Q0FBdUM7b0JBQ3ZDLG9DQUFvQztpQkFDckM7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVztvQkFDNUIsNEJBQTRCLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxpQkFBaUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxHQUFHLEVBQUU7b0JBQzNHLGNBQWMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLFdBQVcsVUFBVSxFQUFFO2lCQUMxRTthQUNGLENBQUM7WUFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLE9BQU8sRUFBRTtvQkFDUCxjQUFjO2lCQUNmO2dCQUNELFVBQVUsRUFBRTtvQkFDVixVQUFVLEVBQUUsRUFBQyxxQkFBcUIsRUFBRSxnQ0FBZ0MsRUFBQztpQkFDdEU7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULDBCQUEwQixDQUFDLE9BQU87aUJBQ25DO2FBQ0YsQ0FBQztTQUNIO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsZ0NBQWdDLENBQUMsWUFBWSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFFMUUsSUFBSSxlQUFlLEdBQUcsOENBQXFCLENBQUMsVUFBVSxFQUFFLDBCQUEwQixFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBRTdGLElBQUksY0FBYyxFQUFFO1FBQ2xCLGVBQWUsR0FBRyxxQkFBYSxDQUFDLGVBQWUsRUFBRSxjQUFjLENBQUMsQ0FBQztLQUNsRTtJQUVELE1BQU0sUUFBUSxHQUFHLElBQUksYUFBYSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFFNUYsUUFBUSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEdBQUc7UUFDN0IsT0FBTyxFQUFFO1lBQ1AsaUJBQWlCLEVBQUUsQ0FBQztvQkFDbEIsRUFBRSxFQUFFLEtBQUs7b0JBQ1QsTUFBTSxFQUFFLDhLQUE4SztpQkFDdkwsQ0FBQztTQUNIO0tBQ0YsQ0FBQztJQUVGLE9BQU8sUUFBUSxDQUFDO0FBQ2xCLENBQUM7QUFoRUQsZ0RBZ0VDO0FBRUQsU0FBZ0IsMEJBQTBCLENBQUMsS0FBb0I7SUFDN0QseUJBQXlCO0lBQ3pCLE1BQU0sTUFBTSxHQUF1QixJQUFJLEtBQUssRUFBRSxDQUFDO0lBRS9DLHFFQUFxRTtJQUNyRSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDeEQsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsbUJBQW1CO1NBQ2hDLENBQUM7UUFDRixTQUFTLEVBQUUsQ0FBQztRQUNaLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNoQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLDJFQUEyRTtLQUM5RixDQUFDLENBQUMsQ0FBQztJQUVKLHdFQUF3RTtJQUN4RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDM0QsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsc0JBQXNCO1NBQ25DLENBQUM7UUFDRixTQUFTLEVBQUUsQ0FBQztRQUNaLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNoQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLHdEQUF3RDtLQUMzRSxDQUFDLENBQUMsQ0FBQztJQUVKLHdFQUF3RTtJQUN4RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsNkJBQTZCLEVBQUU7UUFDckUsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsa0JBQWtCO1NBQy9CLENBQUM7UUFDRixTQUFTLEVBQUUsSUFBSTtRQUNmLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNoQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsK0JBQStCO1FBQ2pGLGdCQUFnQixFQUFFLGlFQUFpRTtLQUNwRixDQUFDLENBQUMsQ0FBQztJQUVKLHNFQUFzRTtJQUN0RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsZ0NBQWdDLEVBQUU7UUFDeEUsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsMkJBQTJCO1NBQ3hDLENBQUM7UUFDRixTQUFTLEVBQUUsQ0FBQztRQUNaLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLDBDQUEwQztLQUM3RCxDQUFDLENBQUMsQ0FBQztJQUVKLDRFQUE0RTtJQUM1RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsc0NBQXNDLEVBQUU7UUFDOUUsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsMEJBQTBCO1NBQ3ZDLENBQUM7UUFDRixTQUFTLEVBQUUsQ0FBQztRQUNaLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNoQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLGdHQUFnRztLQUNuSCxDQUFDLENBQUMsQ0FBQztJQUVKLHVFQUF1RTtJQUN2RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsNEJBQTRCLEVBQUU7UUFDcEUsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsZ0JBQWdCO1NBQzdCLENBQUM7UUFDRixTQUFTLEVBQUUsRUFBRTtRQUNiLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLDBJQUEwSTtLQUM3SixDQUFDLENBQUMsQ0FBQztJQUVKLHlFQUF5RTtJQUN6RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsK0JBQStCLEVBQUU7UUFDdkUsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsbUJBQW1CO1NBQ2hDLENBQUM7UUFDRixTQUFTLEVBQUUsRUFBRTtRQUNiLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLHlGQUF5RjtLQUM1RyxDQUFDLENBQUMsQ0FBQztJQUVKLDZFQUE2RTtJQUM3RSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsa0NBQWtDLEVBQUU7UUFDMUUsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUsc0JBQXNCO1NBQ25DLENBQUM7UUFDRixTQUFTLEVBQUUsRUFBRTtRQUNiLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLDhIQUE4SDtLQUNqSixDQUFDLENBQUMsQ0FBQztJQUVKLCtFQUErRTtJQUMvRSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUscUNBQXFDLEVBQUU7UUFDN0UsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVLEVBQUUseUJBQXlCO1NBQ3RDLENBQUM7UUFDRixTQUFTLEVBQUUsRUFBRTtRQUNiLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLFNBQVM7UUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLHlGQUF5RjtLQUM1RyxDQUFDLENBQUMsQ0FBQztJQUVKLE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFuSUQsZ0VBbUlDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMTkgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgKiBhcyBlbGFzdGljc2VhcmNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1lbGFzdGljc2VhcmNoJztcbmltcG9ydCB7IENmbkRvbWFpbk9wdGlvbnMsIERlZmF1bHRDZm5Eb21haW5Qcm9wcyB9IGZyb20gJy4vZWxhc3RpY3NlYXJjaC1kZWZhdWx0cyc7XG5pbXBvcnQgeyBvdmVycmlkZVByb3BzIH0gZnJvbSAnLi91dGlscyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcblxuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkRWxhc3RpY1NlYXJjaChzY29wZTogY2RrLkNvbnN0cnVjdCwgZG9tYWluTmFtZTogc3RyaW5nLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBvcHRpb25zOiBDZm5Eb21haW5PcHRpb25zLCBjZm5Eb21haW5Qcm9wcz86IGVsYXN0aWNzZWFyY2guQ2ZuRG9tYWluUHJvcHMpOiBlbGFzdGljc2VhcmNoLkNmbkRvbWFpbiB7XG5cbiAgLy8gU2V0dXAgdGhlIElBTSBSb2xlICYgcG9saWN5IGZvciBFUyB0byBjb25maWd1cmUgQ29nbml0byBVc2VyIHBvb2wgYW5kIElkZW50aXR5IHBvb2xcbiAgY29uc3QgY29nbml0b0tpYmFuYUNvbmZpZ3VyZVJvbGUgPSBuZXcgaWFtLlJvbGUoc2NvcGUsICdDb2duaXRvS2liYW5hQ29uZmlndXJlUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZXMuYW1hem9uYXdzLmNvbScpXG4gIH0pO1xuXG4gIGNvbnN0IGNvZ25pdG9LaWJhbmFDb25maWd1cmVSb2xlUG9saWN5ID0gbmV3IGlhbS5Qb2xpY3koc2NvcGUsICdDb2duaXRvS2liYW5hQ29uZmlndXJlUm9sZVBvbGljeScsIHtcbiAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICBcImNvZ25pdG8taWRwOkRlc2NyaWJlVXNlclBvb2xcIixcbiAgICAgICAgICBcImNvZ25pdG8taWRwOkNyZWF0ZVVzZXJQb29sQ2xpZW50XCIsXG4gICAgICAgICAgXCJjb2duaXRvLWlkcDpEZWxldGVVc2VyUG9vbENsaWVudFwiLFxuICAgICAgICAgIFwiY29nbml0by1pZHA6RGVzY3JpYmVVc2VyUG9vbENsaWVudFwiLFxuICAgICAgICAgIFwiY29nbml0by1pZHA6QWRtaW5Jbml0aWF0ZUF1dGhcIixcbiAgICAgICAgICBcImNvZ25pdG8taWRwOkFkbWluVXNlckdsb2JhbFNpZ25PdXRcIixcbiAgICAgICAgICBcImNvZ25pdG8taWRwOkxpc3RVc2VyUG9vbENsaWVudHNcIixcbiAgICAgICAgICBcImNvZ25pdG8taWRlbnRpdHk6RGVzY3JpYmVJZGVudGl0eVBvb2xcIixcbiAgICAgICAgICBcImNvZ25pdG8taWRlbnRpdHk6VXBkYXRlSWRlbnRpdHlQb29sXCIsXG4gICAgICAgICAgXCJjb2duaXRvLWlkZW50aXR5OlNldElkZW50aXR5UG9vbFJvbGVzXCIsXG4gICAgICAgICAgXCJjb2duaXRvLWlkZW50aXR5OkdldElkZW50aXR5UG9vbFJvbGVzXCIsXG4gICAgICAgICAgXCJlczpVcGRhdGVFbGFzdGljc2VhcmNoRG9tYWluQ29uZmlnXCJcbiAgICAgICAgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgb3B0aW9ucy51c2VycG9vbC51c2VyUG9vbEFybixcbiAgICAgICAgICBgYXJuOmF3czpjb2duaXRvLWlkZW50aXR5OiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTppZGVudGl0eXBvb2wvJHtvcHRpb25zLmlkZW50aXR5cG9vbC5yZWZ9YCxcbiAgICAgICAgICBgYXJuOmF3czplczoke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06ZG9tYWluLyR7ZG9tYWluTmFtZX1gXG4gICAgICAgIF1cbiAgICAgIH0pLFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgXCJpYW06UGFzc1JvbGVcIlxuICAgICAgICBdLFxuICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgU3RyaW5nTGlrZTogeydpYW06UGFzc2VkVG9TZXJ2aWNlJzogJ2NvZ25pdG8taWRlbnRpdHkuYW1hem9uYXdzLmNvbSd9XG4gICAgICAgIH0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGNvZ25pdG9LaWJhbmFDb25maWd1cmVSb2xlLnJvbGVBcm5cbiAgICAgICAgXVxuICAgICAgfSlcbiAgICBdXG4gIH0pO1xuICBjb2duaXRvS2liYW5hQ29uZmlndXJlUm9sZVBvbGljeS5hdHRhY2hUb1JvbGUoY29nbml0b0tpYmFuYUNvbmZpZ3VyZVJvbGUpO1xuXG4gIGxldCBfY2ZuRG9tYWluUHJvcHMgPSBEZWZhdWx0Q2ZuRG9tYWluUHJvcHMoZG9tYWluTmFtZSwgY29nbml0b0tpYmFuYUNvbmZpZ3VyZVJvbGUsIG9wdGlvbnMpO1xuXG4gIGlmIChjZm5Eb21haW5Qcm9wcykge1xuICAgIF9jZm5Eb21haW5Qcm9wcyA9IG92ZXJyaWRlUHJvcHMoX2NmbkRvbWFpblByb3BzLCBjZm5Eb21haW5Qcm9wcyk7XG4gIH1cblxuICBjb25zdCBlc0RvbWFpbiA9IG5ldyBlbGFzdGljc2VhcmNoLkNmbkRvbWFpbihzY29wZSwgJ0VsYXN0aWNzZWFyY2hEb21haW4nLCBfY2ZuRG9tYWluUHJvcHMpO1xuXG4gIGVzRG9tYWluLmNmbk9wdGlvbnMubWV0YWRhdGEgPSB7XG4gICAgY2ZuX25hZzoge1xuICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgIGlkOiAnVzI4JyxcbiAgICAgICAgcmVhc29uOiBgVGhlIEVTIERvbWFpbiBpcyBwYXNzZWQgZHluYW1pY2FsbHkgYXMgYXMgcGFyYW1ldGVyIGFuZCBleHBsaWNpdGx5IHNwZWNpZmllZCB0byBlbnN1cmUgdGhhdCBJQU0gcG9saWNpZXMgYXJlIGNvbmZpZ3VyZWQgdG8gbG9ja2Rvd24gYWNjZXNzIHRvIHRoaXMgc3BlY2lmaWMgRVMgaW5zdGFuY2Ugb25seWBcbiAgICAgIH1dXG4gICAgfVxuICB9O1xuXG4gIHJldHVybiBlc0RvbWFpbjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkRWxhc3RpY1NlYXJjaENXQWxhcm1zKHNjb3BlOiBjZGsuQ29uc3RydWN0KTogY2xvdWR3YXRjaC5BbGFybVtdIHtcbiAgLy8gU2V0dXAgQ1cgQWxhcm1zIGZvciBFU1xuICBjb25zdCBhbGFybXM6IGNsb3Vkd2F0Y2guQWxhcm1bXSA9IG5ldyBBcnJheSgpO1xuXG4gIC8vIENsdXN0ZXJTdGF0dXMucmVkIG1heGltdW0gaXMgPj0gMSBmb3IgMSBtaW51dGUsIDEgY29uc2VjdXRpdmUgdGltZVxuICBhbGFybXMucHVzaChuZXcgY2xvdWR3YXRjaC5BbGFybShzY29wZSwgJ1N0YXR1c1JlZEFsYXJtJywge1xuICAgIG1ldHJpYzogbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9FUycsXG4gICAgICBtZXRyaWNOYW1lOiAnQ2x1c3RlclN0YXR1cy5yZWQnXG4gICAgfSksXG4gICAgdGhyZXNob2xkOiAxLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIHN0YXRpc3RpYzogJ01heGltdW0nLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQXQgbGVhc3Qgb25lIHByaW1hcnkgc2hhcmQgYW5kIGl0cyByZXBsaWNhcyBhcmUgbm90IGFsbG9jYXRlZCB0byBhIG5vZGUuICdcbiAgfSkpO1xuXG4gIC8vIENsdXN0ZXJTdGF0dXMueWVsbG93IG1heGltdW0gaXMgPj0gMSBmb3IgMSBtaW51dGUsIDEgY29uc2VjdXRpdmUgdGltZVxuICBhbGFybXMucHVzaChuZXcgY2xvdWR3YXRjaC5BbGFybShzY29wZSwgJ1N0YXR1c1llbGxvd0FsYXJtJywge1xuICAgIG1ldHJpYzogbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9FUycsXG4gICAgICBtZXRyaWNOYW1lOiAnQ2x1c3RlclN0YXR1cy55ZWxsb3cnXG4gICAgfSksXG4gICAgdGhyZXNob2xkOiAxLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIHN0YXRpc3RpYzogJ01heGltdW0nLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQXQgbGVhc3Qgb25lIHJlcGxpY2Egc2hhcmQgaXMgbm90IGFsbG9jYXRlZCB0byBhIG5vZGUuJ1xuICB9KSk7XG5cbiAgLy8gRnJlZVN0b3JhZ2VTcGFjZSBtaW5pbXVtIGlzIDw9IDIwNDgwIGZvciAxIG1pbnV0ZSwgMSBjb25zZWN1dGl2ZSB0aW1lXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnRnJlZVN0b3JhZ2VTcGFjZVRvb0xvd0FsYXJtJywge1xuICAgIG1ldHJpYzogbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9FUycsXG4gICAgICBtZXRyaWNOYW1lOiAnRnJlZVN0b3JhZ2VTcGFjZSdcbiAgICB9KSxcbiAgICB0aHJlc2hvbGQ6IDIwMDAsXG4gICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgc3RhdGlzdGljOiAnTWluaW11bScsXG4gICAgcGVyaW9kOiBjZGsuRHVyYXRpb24uc2Vjb25kcyg2MCksXG4gICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjbG91ZHdhdGNoLkNvbXBhcmlzb25PcGVyYXRvci5MRVNTX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xELFxuICAgIGFsYXJtRGVzY3JpcHRpb246ICdBIG5vZGUgaW4geW91ciBjbHVzdGVyIGlzIGRvd24gdG8gMjAgR2lCIG9mIGZyZWUgc3RvcmFnZSBzcGFjZS4nXG4gIH0pKTtcblxuICAvLyBDbHVzdGVySW5kZXhXcml0ZXNCbG9ja2VkIGlzID49IDEgZm9yIDUgbWludXRlcywgMSBjb25zZWN1dGl2ZSB0aW1lXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnSW5kZXhXcml0ZXNCbG9ja2VkVG9vSGlnaEFsYXJtJywge1xuICAgIG1ldHJpYzogbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9FUycsXG4gICAgICBtZXRyaWNOYW1lOiAnQ2x1c3RlckluZGV4V3JpdGVzQmxvY2tlZCdcbiAgICB9KSxcbiAgICB0aHJlc2hvbGQ6IDEsXG4gICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgc3RhdGlzdGljOiAnTWF4aW11bScsXG4gICAgcGVyaW9kOiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnWW91ciBjbHVzdGVyIGlzIGJsb2NraW5nIHdyaXRlIHJlcXVlc3RzLidcbiAgfSkpO1xuXG4gIC8vIEF1dG9tYXRlZFNuYXBzaG90RmFpbHVyZSBtYXhpbXVtIGlzID49IDEgZm9yIDEgbWludXRlLCAxIGNvbnNlY3V0aXZlIHRpbWVcbiAgYWxhcm1zLnB1c2gobmV3IGNsb3Vkd2F0Y2guQWxhcm0oc2NvcGUsICdBdXRvbWF0ZWRTbmFwc2hvdEZhaWx1cmVUb29IaWdoQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0VTJyxcbiAgICAgIG1ldHJpY05hbWU6ICdBdXRvbWF0ZWRTbmFwc2hvdEZhaWx1cmUnXG4gICAgfSksXG4gICAgdGhyZXNob2xkOiAxLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIHN0YXRpc3RpYzogJ01heGltdW0nLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQW4gYXV0b21hdGVkIHNuYXBzaG90IGZhaWxlZC4gVGhpcyBmYWlsdXJlIGlzIG9mdGVuIHRoZSByZXN1bHQgb2YgYSByZWQgY2x1c3RlciBoZWFsdGggc3RhdHVzLidcbiAgfSkpO1xuXG4gIC8vIENQVVV0aWxpemF0aW9uIG1heGltdW0gaXMgPj0gODAlIGZvciAxNSBtaW51dGVzLCAzIGNvbnNlY3V0aXZlIHRpbWVzXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnQ1BVVXRpbGl6YXRpb25Ub29IaWdoQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0VTJyxcbiAgICAgIG1ldHJpY05hbWU6ICdDUFVVdGlsaXphdGlvbidcbiAgICB9KSxcbiAgICB0aHJlc2hvbGQ6IDgwLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoOTAwKSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJzEwMCUgQ1BVIHV0aWxpemF0aW9uIGlzIG5vdCB1bmNvbW1vbiwgYnV0IHN1c3RhaW5lZCBoaWdoIHVzYWdlIGlzIHByb2JsZW1hdGljLiBDb25zaWRlciB1c2luZyBsYXJnZXIgaW5zdGFuY2UgdHlwZXMgb3IgYWRkaW5nIGluc3RhbmNlcy4nXG4gIH0pKTtcblxuICAvLyBKVk1NZW1vcnlQcmVzc3VyZSBtYXhpbXVtIGlzID49IDgwJSBmb3IgNSBtaW51dGVzLCAzIGNvbnNlY3V0aXZlIHRpbWVzXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnSlZNTWVtb3J5UHJlc3N1cmVUb29IaWdoQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0VTJyxcbiAgICAgIG1ldHJpY05hbWU6ICdKVk1NZW1vcnlQcmVzc3VyZSdcbiAgICB9KSxcbiAgICB0aHJlc2hvbGQ6IDgwLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoOTAwKSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0F2ZXJhZ2UgSlZNIG1lbW9yeSBwcmVzc3VyZSBvdmVyIGxhc3QgMTUgbWludXRlcyB0b28gaGlnaC4gQ29uc2lkZXIgc2NhbGluZyB2ZXJ0aWNhbGx5LidcbiAgfSkpO1xuXG4gIC8vIE1hc3RlckNQVVV0aWxpemF0aW9uIG1heGltdW0gaXMgPj0gNTAlIGZvciAxNSBtaW51dGVzLCAzIGNvbnNlY3V0aXZlIHRpbWVzXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnTWFzdGVyQ1BVVXRpbGl6YXRpb25Ub29IaWdoQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0VTJyxcbiAgICAgIG1ldHJpY05hbWU6ICdNYXN0ZXJDUFVVdGlsaXphdGlvbidcbiAgICB9KSxcbiAgICB0aHJlc2hvbGQ6IDUwLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoOTAwKSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0F2ZXJhZ2UgQ1BVIHV0aWxpemF0aW9uIG92ZXIgbGFzdCA0NSBtaW51dGVzIHRvbyBoaWdoLiBDb25zaWRlciB1c2luZyBsYXJnZXIgaW5zdGFuY2UgdHlwZXMgZm9yIHlvdXIgZGVkaWNhdGVkIG1hc3RlciBub2Rlcy4nXG4gIH0pKTtcblxuICAvLyBNYXN0ZXJKVk1NZW1vcnlQcmVzc3VyZSBtYXhpbXVtIGlzID49IDgwJSBmb3IgMTUgbWludXRlcywgMSBjb25zZWN1dGl2ZSB0aW1lXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnTWFzdGVySlZNTWVtb3J5UHJlc3N1cmVUb29IaWdoQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0VTJyxcbiAgICAgIG1ldHJpY05hbWU6ICdNYXN0ZXJKVk1NZW1vcnlQcmVzc3VyZSdcbiAgICB9KSxcbiAgICB0aHJlc2hvbGQ6IDUwLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoOTAwKSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0F2ZXJhZ2UgSlZNIG1lbW9yeSBwcmVzc3VyZSBvdmVyIGxhc3QgMTUgbWludXRlcyB0b28gaGlnaC4gQ29uc2lkZXIgc2NhbGluZyB2ZXJ0aWNhbGx5LidcbiAgfSkpO1xuXG4gIHJldHVybiBhbGFybXM7XG59XG4iXX0=