"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.flagOverriddenDefaults = void 0;
const deepdiff = require("deep-diff");
const log = require("npmlog");
/**
 * Emits a warning to the console when a prescriptive default value is overridden by the user.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 */
function flagOverriddenDefaults(defaultProps, userProps) {
    // Compare the properties and return any overrides
    const overrides = findOverrides(defaultProps, userProps);
    // Emit a warning for each override
    for (let i = 0; i < ((overrides !== undefined) ? overrides.length : 0); i++) {
        const e = Object.assign(overrides[i]);
        // Determine appropriate logging granularity
        const valuesAreReadable = (checkReadability(e.lhs) &&
            checkReadability(e.rhs));
        // Format the path for readability
        const path = formatOverridePath(e.path);
        // Style the log output
        log.prefixStyle.bold = true;
        log.prefixStyle.fg = 'red';
        log.enableColor();
        // Output
        const details = (valuesAreReadable) ? ` Default value: '${e.lhs}'. You provided: '${e.rhs}'.` : '';
        log.warn('AWS_SOLUTIONS_CONSTRUCTS_WARNING: ', `An override has been provided for the property: ${path}.` + details);
    }
}
exports.flagOverriddenDefaults = flagOverriddenDefaults;
/**
 * Performs a diff check of the userProps against the defaultProps to detect overridden properties.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 * @return {Array} an array containing the overridden values.
 */
function findOverrides(defaultProps, userProps) {
    const diff = deepdiff.diff(defaultProps, userProps);
    // Filter the results
    return (diff !== undefined) ? diff === null || diff === void 0 ? void 0 : diff.filter((e) => {
        var _a, _b;
        return (e.kind === 'E' && // only return overrides
            !((_a = e.path) === null || _a === void 0 ? void 0 : _a.includes('node')) && // stop traversing once the object graph hits the node
            !((_b = e.path) === null || _b === void 0 ? void 0 : _b.includes('bind')) // stop traversing once the object graph hits internal functions
        );
    }) : [];
}
/**
 * Converts the path object from the deep-diff module to a user-readable, bracket notation format.
 * @param {string | string[]} path either a string value or an array of strings.
 * @return {string} the formatted override path.
 */
function formatOverridePath(path) {
    return (path !== undefined && path.length > 1) ? path.toString()
        .replace(/,/g, '][')
        .replace(/\]{1}/, '')
        .replace(/$/, ']') : path;
}
/**
 * Check the readability of an input value and, in the context of the override warning service, return true if it
 * meets the established criteria. This function is used to determine whether more-detailed log output can be given.
 * @param {any} value input to be evaluated against the given criteria.
 * @return {boolean} true if the value meets the given criteria.
 * @return {boolean} false if the value does not meet the given criteria.
 */
function checkReadability(value) {
    return (typeof (value) === 'string' && // strings only
        !value.includes('${Token[') && // no circular refs
        value !== '' // no empty strings
    );
}
//# sourceMappingURL=data:application/json;base64,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