"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
function buildS3Bucket(scope, props, bucketId) {
    // Conditional s3 Bucket creation
    if (!props.existingBucketObj) {
        if (props.bucketProps) {
            return s3BucketWithLogging(scope, props.bucketProps, bucketId);
        }
        else {
            return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
        }
    }
    else {
        return props.existingBucketObj;
    }
}
exports.buildS3Bucket = buildS3Bucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    // Create the Application Bucket
    let bucketprops;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        bucketprops = s3_bucket_defaults_1.DefaultS3Props();
    }
    else {
        // Create the Logging Bucket
        const loggingBucket = new s3.Bucket(scope, _loggingBucketId, s3_bucket_defaults_1.DefaultS3Props());
        // Extract the CfnBucket from the loggingBucket
        const loggingBucketResource = loggingBucket.node.findChild('Resource');
        // Override accessControl configuration and add metadata for the logging bucket
        loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
        loggingBucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W35',
                        reason: `This S3 bucket is used as the access logging bucket for another bucket`
                    }, {
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
        bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    return s3Bucket;
}
//# sourceMappingURL=data:application/json;base64,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