"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
const s3_bucket_helper_1 = require("../lib/s3-bucket-helper");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("@aws-cdk/aws-s3");
test('cloudfront distribution with default params', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check bucket policy metadata', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::BucketPolicy', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "F16",
                        reason: "Public website bucket policy requires a wildcard principal"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('check bucket metadata', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W35",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    },
                    {
                        id: "W51",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('test cloudfront check bucket policy', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResourceLike("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "s3:GetObject*",
                        "s3:GetBucket*",
                        "s3:List*"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::cloudfront:user/CloudFront Origin Access Identity ",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity"
                                    }
                                ]
                            ]
                        }
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "S3Bucket07682993",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "S3Bucket07682993",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                },
                {
                    Action: "s3:GetObject",
                    Effect: "Allow",
                    Principal: {
                        CanonicalUser: {
                            "Fn::GetAtt": [
                                "CloudFrontOriginAccessIdentity",
                                "S3CanonicalUserId"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "S3Bucket07682993",
                                        "Arn"
                                    ]
                                },
                                "/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('test cloudfront with no security headers ', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {}, false);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "origin1",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
test('test cloudfront override cloudfront custom domain names ', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const myprops = {
        aliasConfiguration: {
            acmCertRef: '/acm/mycertificate',
            names: ['mydomains']
        }
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            Aliases: [
                "mydomains"
            ],
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "origin1",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                AcmCertificateArn: "/acm/mycertificate",
                SslSupportMethod: "sni-only"
            }
        }
    });
});
test('test cloudfront override cloudfront logging bucket ', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const loggingBucket = new aws_s3_1.Bucket(stack, 'loggingbucket');
    const myprops = {
        loggingConfig: {
            bucket: loggingBucket,
            includeCookies: true
        }
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "loggingbucket6D73BD53",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: true
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "origin1",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
test('test cloudfront override properties', () => {
    const stack = new core_1.Stack();
    const sourceBucket = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // Create CloudFront Origin Access Identity User
    const cfnOrigAccessId = new cloudfront.CfnCloudFrontOriginAccessIdentity(stack, 'CloudFrontOriginAccessIdentity1', {
        cloudFrontOriginAccessIdentityConfig: {
            comment: 'Access S3 bucket content only through CloudFront'
        }
    });
    const oaiImported = cloudfront.OriginAccessIdentity.fromOriginAccessIdentityName(stack, 'OAIImported1', cfnOrigAccessId.ref);
    const props = {
        originConfigs: [{
                s3OriginSource: {
                    s3BucketSource: sourceBucket,
                    originAccessIdentity: oaiImported
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        allowedMethods: cloudfront.CloudFrontAllowedMethods.ALL,
                        cachedMethods: cloudfront.CloudFrontAllowedCachedMethods.GET_HEAD_OPTIONS
                    }]
            }]
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "DELETE",
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PATCH",
                    "POST",
                    "PUT"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "origin1",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontOriginAccessIdentity1"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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