"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('snapshot test buildUserPool default params', () => {
    const stack = new core_1.Stack();
    defaults.buildUserPool(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('snapshot test buildUserPoolClient default params', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    defaults.buildUserPoolClient(stack, userpool);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for buildUserPool', () => {
    const stack = new core_1.Stack();
    const userpoolProps = {
        userPoolName: 'test',
        signInAliases: { username: false, email: true, phone: true }
    };
    defaults.buildUserPool(stack, userpoolProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        UsernameAttributes: [
            "email",
            "phone_number"
        ],
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test"
    });
});
test('Test override for buildUserPoolClient', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclientProps = {
        userPoolClientName: 'test',
        userPool: userpool
    };
    defaults.buildUserPoolClient(stack, userpool, userpoolclientProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        UserPoolId: {
            Ref: "CognitoUserPool53E37E69"
        },
        ClientName: "test"
    });
});
test('Test override for buildIdentityPool', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    defaults.buildIdentityPool(stack, userpool, userpoolclient, {
        allowUnauthenticatedIdentities: true
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPool', {
        AllowUnauthenticatedIdentities: true,
        CognitoIdentityProviders: [
            {
                ClientId: {
                    Ref: "CognitoUserPoolClient5AB59AE4"
                },
                ProviderName: {
                    "Fn::GetAtt": [
                        "CognitoUserPool53E37E69",
                        "ProviderName"
                    ]
                },
                ServerSideTokenCheck: true
            }
        ]
    });
});
test('Test setupCognitoForElasticSearch', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain"
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPoolRoleAttachment', {
        IdentityPoolId: {
            Ref: "CognitoIdentityPool"
        },
        Roles: {
            authenticated: {
                "Fn::GetAtt": [
                    "CognitoAuthorizedRole14E74FE0",
                    "Arn"
                ]
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRoleWithWebIdentity",
                    Condition: {
                        "StringEquals": {
                            "cognito-identity.amazonaws.com:aud": {
                                Ref: "CognitoIdentityPool"
                            }
                        },
                        "ForAnyValue:StringLike": {
                            "cognito-identity.amazonaws.com:amr": "authenticated"
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        Federated: "cognito-identity.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "es:ESHttp*",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:es:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":domain/test-domain/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "CognitoAccessPolicy"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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