"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const sqs = require("@aws-cdk/aws-sqs");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'primary-queue');
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment w/ custom properties
// --------------------------------------------------------------
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper setup
    const encKey = defaults.buildEncryptionKey(stack);
    // Helper declaration
    defaults.buildQueue(stack, 'primary-queue', {
        queueProps: {
            queueName: "custom-queue-props",
            encryption: sqs.QueueEncryption.KMS,
            encryptionMasterKey: encKey
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test dead letter queue deployment/configuration
// --------------------------------------------------------------
test('Test dead letter queue deployment/configuration', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper setup
    const encKey = defaults.buildEncryptionKey(stack);
    const dlq = defaults.buildQueue(stack, 'dead-letter-queue');
    const dlqi = defaults.buildDeadLetterQueue({
        deadLetterQueue: dlq,
        maxReceiveCount: 3
    });
    // Helper declaration
    defaults.buildQueue(stack, 'primary-queue', {
        queueProps: {
            queueName: "not-the-dead-letter-queue-props",
            encryption: sqs.QueueEncryption.KMS,
            encryptionMasterKey: encKey
        },
        deadLetterQueue: dlqi
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test dead letter queue deployment/configuration w/o mrc
// --------------------------------------------------------------
test('Test dead letter queue deployment/configuration w/o mrc', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper setup
    const encKey = defaults.buildEncryptionKey(stack);
    const dlq = defaults.buildQueue(stack, 'dead-letter-queue');
    const dlqi = defaults.buildDeadLetterQueue({
        deadLetterQueue: dlq
    });
    // Helper declaration
    defaults.buildQueue(stack, 'primary-queue', {
        queueProps: {
            queueName: "not-the-dead-letter-queue-props",
            encryption: sqs.QueueEncryption.KMS,
            encryptionMasterKey: encKey
        },
        deadLetterQueue: dlqi
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test existingQueueObj
// --------------------------------------------------------------
test('Test existingQueueObj', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper setup
    const existingQueue = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    // Helper declaration
    defaults.buildQueue(stack, 'primary-queue', {
        existingQueueObj: existingQueue
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
//# sourceMappingURL=data:application/json;base64,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