"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventField = exports.RuleTargetInput = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * The input to send to the event target.
 */
class RuleTargetInput {
    /**
     * Pass text to the event target.
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by EventField.from() to substitute in parts
     * of the matched event.
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target.
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON.
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
    /**
     *
     */
    constructor() {
    }
}
exports.RuleTargetInput = RuleTargetInput;
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by EventBridge, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        const self = this;
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return self.keyPlaceholder(key);
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        if (Object.keys(inputPathsMap).length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved),
            inputPathsMap,
        };
    }
    /**
     * Return a template placeholder for the given key
     *
     * In object scope we'll need to get rid of surrounding quotes later on, so
     * return a bracing that's unlikely to occur naturally (like tokens).
     */
    keyPlaceholder(key) {
        if (this.inputType !== InputType.Object) {
            return `<${key}>`;
        }
        return UNLIKELY_OPENING_STRING + key + UNLIKELY_CLOSING_STRING;
    }
    /**
     * Removing surrounding quotes from any object placeholders
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     */
    unquoteKeyPlaceholders(sub) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.uncachedString({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return resolved.replace(OPENING_STRING_REGEX, '<').replace(CLOSING_STRING_REGEX, '>');
            }
            return resolved;
        }
    }
}
const UNLIKELY_OPENING_STRING = '<<${';
const UNLIKELY_CLOSING_STRING = '}>>';
const OPENING_STRING_REGEX = new RegExp(regexQuote('"' + UNLIKELY_OPENING_STRING), 'g');
const CLOSING_STRING_REGEX = new RegExp(regexQuote(UNLIKELY_CLOSING_STRING + '"'), 'g');
/**
 * Represents a field in the event pattern.
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event.
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event.
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event.
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event.
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event.
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event.
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event.
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    /**
     * Produce the Token's value at resolution time.
     */
    resolve(_ctx) {
        return this.path;
    }
    /**
     * Return a string representation of this resolvable object.
     *
     * Returns a reversible string representation.
     */
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON.
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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