"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
/**
 * @stability stable
 */
class Method extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const integration = (_c = (_b = props.integration) !== null && _b !== void 0 ? _b : this.resource.defaultIntegration) !== null && _c !== void 0 ? _c : new mock_1.MockIntegration();
        const bindResult = integration.bind(this);
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(bindResult),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_d = options.authorizationScopes) !== null && _d !== void 0 ? _d : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                method: {
                    ...methodProps,
                    integrationToken: bindResult === null || bindResult === void 0 ? void 0 : bindResult.deploymentToken,
                },
            });
        }
    }
    /**
     * (deprecated) The RestApi associated with this Method.
     *
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:.
     *
     * arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled, or can be explicitly assigned.
     * When not configured, {stage} will be set to '*', as a shorthand for 'all stages'.
     *
     * @stability stable
     * @attribute true
     */
    get methodArn() {
        var _b;
        const stage = (_b = this.api.deploymentStage) === null || _b === void 0 ? void 0 : _b.stageName;
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     *
     * This stage is used by the AWS Console UI when testing the method.
     *
     * @stability stable
     */
    get testMethodArn() {
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(bindResult) {
        var _b, _c;
        const options = (_b = bindResult.options) !== null && _b !== void 0 ? _b : {};
        let credentials;
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // eslint-disable-next-line max-len
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: bindResult.type,
            uri: bindResult.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: bindResult.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
            timeoutInMillis: (_c = options.timeout) === null || _c === void 0 ? void 0 : _c.toMilliseconds(),
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _b;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_b = options.requestValidator) === null || _b === void 0 ? void 0 : _b.requestValidatorId;
    }
}
exports.Method = Method;
_a = JSII_RTTI_SYMBOL_1;
Method[_a] = { fqn: "@aws-cdk/aws-apigateway.Method", version: "1.111.0" };
/**
 * @stability stable
 */
var AuthorizationType;
(function (AuthorizationType) {
    AuthorizationType["NONE"] = "NONE";
    AuthorizationType["IAM"] = "AWS_IAM";
    AuthorizationType["CUSTOM"] = "CUSTOM";
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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