import pypega.environment
import json

def register_new(system_name:str, environment_url:str, production_level:int=0):
    return PegaDMCandidate(system_name=system_name, environment_url=environment_url, production_level=production_level)

class PegaDMCandidate(pypega.environment.PegaEnvironment):
    ## Candidate - Application endpoints
    API_ENDPOINT_CANDIDATE_GET_RULESET = "/api/cicdext/v1/ruleset/{ruleset_name}/{ruleset_version}"
    API_ENDPOINT_CANDIDATE_GET_APPLICATION = "/api/cicdext/v1/application/{appName}/{appVersion}"
    API_ENDPOINT_CANDIDATE_CREATE_RULESET = "/api/cicdext/v1/ruleset/{ruleset_name}/{ruleset_version}"
    API_ENDPOINT_CANDIDATE_GET_ACCESSGORUPS = "/api/cicdext/v1/accessgroups"
    
    def __init__(self, system_name, environment_url, production_level):
        """_summary_

        Args:
            system_name (_type_): The short name for the Pega environment, generally .pxSystemName
            environment_url (_type_): The full URL to the environment, beginning with "https://" and ending with "/prweb"
            production_level (_type_): The production level of the environment (1 = Dev, 5 = Prod)
        """
        pypega.pypega.environment.PegaEnvironment.__init__(self, system_name, environment_url, production_level)

    def get_application(self, application_name: str, application_version: str):
        """ Gets details about the given application

        Args:
            application_name (str): The Pega application name to source details for
            application_version (str): The version number (e.g. 01.01.01) to retreieve details for.

        Returns:
            json: JSON data generated by Pega API
        """
        response = self.api_request_get(
            self.API_ENDPOINT_CANDIDATE_GET_APPLICATION.format(
                appName=application_name, appVersion=application_version
            )
        )
        if response.ok:
            return json.loads(response.content)
        else:
            return None

    def get_ruleset(self, ruleset_name: str, ruleset_version: str):
        """Gets details about the given ruleset

        Args:
            ruleset_name (str): The Pega ruleset to retrieve details for
            ruleset_version (str): The ruleset version (e.g. 01-01-01) to retreieve details for

        Returns:
            json: JSON data generated by Pega API
        """
        response = self.api_request_get(
            self.API_ENDPOINT_CANDIDATE_GET_RULESET.format(
                ruleset_name=ruleset_name, ruleset_version=ruleset_version
            )
        )
        if response.ok:
            return json.loads(response.content)
        else:
            return None

    def create_ruleset(self, ruleset_name: str, ruleset_version: str):
        """Creates a new ruleset version matching the details

        Args:
            ruleset_name (str): The ruleset name to give the new ruleset
            ruleset_version (str): The version number to give the new ruleset (e.g. 01-01-01)

        Returns:
            bool: True/false if the ruleset was created
        """
        response = self.api_request_post(
            self.API_ENDPOINT_CANDIDATE_CREATE_RULESET.format(
                ruleset_name=ruleset_name, ruleset_version=ruleset_version
            ),
            "",
        )
        return response.ok