# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pakettic']

package_data = \
{'': ['*']}

install_requires = \
['pyparsing>=3.0.8,<4.0.0', 'tqdm>=4.64.0,<5.0.0', 'zopflipy>=1.7,<2.0']

entry_points = \
{'console_scripts': ['pakettic = pakettic.main:main']}

setup_kwargs = {
    'name': 'pakettic',
    'version': '0.1.0',
    'description': 'A tool for minifying and compressing TIC-80 fantasy console carts',
    'long_description': '# Pakettic\n\nPakettic is a command-line tool for minifying and compressing\n[TIC-80](http://tic80.com/) fantasy console carts. The tool is written in Python\n(3.9+) and used especially for\n[sizecoding](http://www.sizecoding.org/wiki/TIC-80). It compresses existing\ncarts approximately ~1.2% better than best alternatives, and by using its [magic\ncomments](#magic-comments), pakettic might find code that compresses even\nbetter.\n\n## Installation\n\nInstalling with pip:\n\n```bash\n$ pip install pakettic\n```\n\nInstalling the latest main branch from GitHub:\n\n```bash\n$ pip install git+https://github.com/vsariola/pakettic.git@main\n```\n\nInstalling a checked out version of the repository:\n\n```bash\n$ pip install -e path/to/pakettic\n```\n\nInstalling a checked out version of the repository using\n[poetry](https://python-poetry.org/) for a nice virtual environment with locked\ndependencies (run inside the pakettic folder):\n\n```bash\n$ poetry install\n```\n\n## Usage\n\nTo compress a cart, run:\n\n```bash\n$ pakettic path/to/cart.tic\n```\n\nIf your PATH is not configured to include pip installed executables, you\ncan use\n\n```bash\n$ python -m pakettic path/to/cart.tic\n```\n\nIf you installed using poetry into a virtual environment, you need to\nprepend `poetry run` before every command e.g.\n\n```bash\n$ poetry run pakettic path/to/cart.tic\n```\n\nPakettic supports both .tic and .lua carts. Multiple input files may be\ndefined. Input files are globbed, so `?`, `*`, and `**` work as\nwildcards for a single character, multiple characters and a directory,\nrespectively.\n\nFor a full list of command line options, see:\n\n```bash\n$ pakettic --help\n```\n\nSee also [tips for command line arguments](#tips-for-command-line-arguments)\n\nRunning all tests:\n\n```bash\n$ poetry run python -m unittest discover -s tests\n```\n\n## Features\n\nPakettic first parses the LUA-script to an abstract syntax tree, and\nthen uses a local optimization algorithm\n([simulated annealing](https://en.wikipedia.org/wiki/Simulated_annealing),\n[late acceptance hill climbing](https://arxiv.org/pdf/1806.09328.pdf) or\nits variant diversified late acceptance search) to randomly mutate the\nsyntax tree & see if it compresses better. Implemented mutations\ninclude:\n  - shortening variable names\n  - flipping comparisons `>`, `<`, `>=`, `<=`, `~=`, and `==`\n  - reordering arithmetic operators `+`, `-`, `*` and `/` and bit logic\n    operators `&`, `~` and `|`\n  - converting `a^2` into `a*a` and vice versa\n  - using either single or double quotes for all strings\n  - converting whole hexadecimals into decimals\n  - convert `for a,b,1 do` into `for a,b do` and vice versa\n  - reordering statements: statements that can be reordered are marked\n    with [magic comments](#magic-comments)\n  - alternative expressions: alternatives are marked with\n    [magic comments](#magic-comments)\n\nInternally, pakettic uses [zopfli](https://github.com/google/zopfli) for\nactual compression.\n\n`load\'...\'` is parsed as `function()...end` so you can easily recompress\nalready compressed carts. Conversely, `function()...end` is replaced\nwith `load\'...\'` during compression.\n\nUnnecessary parentheses are removed from expressions so you do not have\nto worry about those.\n\n## Magic comments\n\n### Reorderable statements\n\nThe algorithm will try to reorder statements between `--{` and `--}`.\nFor example:\n\n```lua\n --{\n a="hello"\n b="world"\n --}\n```\n\nwill try both `a="hello"b="world"` and `b="world"a="hello"` to see if\ncompresses better.\n\nNotice that only complete statements can be reordered. Thus, this will\nNOT work:\n\n```lua\n --{\n for x=0,239 do\n  for y=0,135 do\n --}\n  end\n end\n```\n\nA good rule of thumb is that you should be able to replace `--{` and\n`--}` with `do` and `end`, respectively, and still have valid code.\n\nStatements between `--{!` and `--}` are not ordered, so you can make\nblocks of statements that are kept in order within a pair of `--{` and\n`--}` tags.\n\n### Alternative expressions\n\nThere is a special `--|` operator that allows alternative expressions to\nbe tested, to see if they compress better. For example: `5--|4--|6`\nmeans that the algorithm will try 4 and 6 in place of the 5. This will\nnaturally show up as a comment in LUA so you will have to continue the\nexpression on next line if this is in the middle of an expression. `--|`\nhas the lowest precedence, even lower than `^` so put parentheses if you\nwant to try more complicated expressions e.g. `(x//256)--|(x>>8)`\n\n## Tips for command line arguments\n\n- Cranking up the Zopfli settings can usually save a few more bytes,\n  with the expense of slowing down the optimization considerably. Try\n  `-z3` to set the Zopfli-level. By default, the Zopfli-level is 2, and\n  it goes up to 5.\n- The algorithm uses a pseudorandom generator. Sometimes using a\n  different seed finds a few byte better or worse solution. Use command\n  line argument `--seed` to try different seeds.\n- Similarly, different optimization heuristics produce slightly\n  different results. Try different heuristics e.g. with `-alahc`,\n  `-adlas` or `-aanneal`.\n- To avoid re-optimizing all the expressions every time, do a long\n  optimization run, study the results and change your expressions to the\n  forms that pack well. Set the number of steps with `-s`. Use\n  command-line argument `-p` to always print a reasonably readable\n  version of the best solution when one is found.\n- By default, pakettic only includes CODE and DEFAULT chunks. DEFAULT\n  indicates that before loading the cart, TIC-80 loads the default cart,\n  setting default palette, waveforms etc. If you don\'t need the default\n  values (e.g. you set the palette yourself), save one byte by only\n  including CODE chunk in the cart: `-ccode`\n- Working on a tweet-cart? Use `-l` to output LUA carts, which are\n  uncompressed. The optimization algorithm then just optimizes the\n  uncompressed size of the code.\n\n## Known issues\n\n- At the moment, all the branches of swappable operators are assumed to\n  be without side effects. If they have side-effects, the swapping might\n  inadvertedly swap the execution order of the two branches.\n\n## Credits\n\n[Veikko Sariola](https://github.com/vsariola) aka pestis/brainlez\nCoders!\n\n## License\n\n[MIT](https://choosealicense.com/licenses/mit/)\n\nThe test corpus carts have their own licenses, see the license files in the\nsubdirectories of the [corpus](corpus/) directory.\n',
    'author': 'Veikko Sariola',
    'author_email': '5684185+vsariola@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/vsariola/pakettic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
