import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import { Construct } from 'constructs';
import { BaseTargetGroupProps, ITargetGroup, LoadBalancerTargetProps, TargetGroupAttributes, TargetGroupBase, TargetGroupImportProps } from '../shared/base-target-group';
import { Protocol } from '../shared/enums';
import { INetworkListener } from './network-listener';
/**
 * Properties for a new Network Target Group.
 *
 * @stability stable
 */
export interface NetworkTargetGroupProps extends BaseTargetGroupProps {
    /**
     * The port on which the listener listens for requests.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * Protocol for target group, expects TCP, TLS, UDP, or TCP_UDP.
     *
     * @default - TCP
     * @stability stable
     */
    readonly protocol?: Protocol;
    /**
     * Indicates whether Proxy Protocol version 2 is enabled.
     *
     * @default false
     * @stability stable
     */
    readonly proxyProtocolV2?: boolean;
    /**
     * Indicates whether client IP preservation is enabled.
     *
     * @default false if the target group type is IP address and the
     * target group protocol is TCP or TLS. Otherwise, true.
     * @stability stable
     */
    readonly preserveClientIp?: boolean;
    /**
     * The targets to add to this target group.
     *
     * Can be `Instance`, `IPAddress`, or any self-registering load balancing
     * target. If you use either `Instance` or `IPAddress` as targets, all
     * target must be of the same type.
     *
     * @default - No targets.
     * @stability stable
     */
    readonly targets?: INetworkLoadBalancerTarget[];
}
/**
 * Define a Network Target Group.
 *
 * @stability stable
 */
export declare class NetworkTargetGroup extends TargetGroupBase implements INetworkTargetGroup {
    /**
     * Import an existing target group.
     *
     * @stability stable
     */
    static fromTargetGroupAttributes(scope: Construct, id: string, attrs: TargetGroupAttributes): INetworkTargetGroup;
    /**
     * (deprecated) Import an existing listener.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope: Construct, id: string, props: TargetGroupImportProps): INetworkTargetGroup;
    private readonly listeners;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: NetworkTargetGroupProps);
    /**
     * Add a load balancing target to this target group.
     *
     * @stability stable
     */
    addTarget(...targets: INetworkLoadBalancerTarget[]): void;
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @stability stable
     */
    registerListener(listener: INetworkListener): void;
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricHealthyHostCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricUnHealthyHostCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Full name of first load balancer.
     *
     * @stability stable
     */
    get firstLoadBalancerFullName(): string;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    private metric;
}
/**
 * A network target group.
 *
 * @stability stable
 */
export interface INetworkTargetGroup extends ITargetGroup {
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @stability stable
     */
    registerListener(listener: INetworkListener): void;
    /**
     * Add a load balancing target to this target group.
     *
     * @stability stable
     */
    addTarget(...targets: INetworkLoadBalancerTarget[]): void;
}
/**
 * Interface for constructs that can be targets of an network load balancer.
 *
 * @stability stable
 */
export interface INetworkLoadBalancerTarget {
    /**
     * Attach load-balanced target to a TargetGroup.
     *
     * May return JSON to directly add to the [Targets] list, or return undefined
     * if the target will register itself with the load balancer.
     *
     * @stability stable
     */
    attachToNetworkTargetGroup(targetGroup: INetworkTargetGroup): LoadBalancerTargetProps;
}
