"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApiId: props.api.restApiId,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Hook into synthesis before it occurs and make any final adjustments.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) {
            // Ignore IRestApi that are imported
            /*
             * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
             * Quoting from CloudFormation's docs - "If you create an AWS::ApiGateway::RestApi resource and its methods (using AWS::ApiGateway::Method) in
             * the same template as your deployment, the deployment must depend on the RestApi's methods. To create a dependency, add a DependsOn attribute
             * to the deployment. If you don't, AWS CloudFormation creates the deployment right after it creates the RestApi resource that doesn't contain
             * any methods, and AWS CloudFormation encounters the following error: The REST API doesn't contain any methods."
             */
            /*
             * Adding a dependency between LatestDeployment and Method construct, using ConstructNode.addDependencies(), creates additional dependencies
             * between AWS::ApiGateway::Deployment and the AWS::Lambda::Permission nodes (children under Method), causing cyclic dependency errors. Hence,
             * falling back to declaring dependencies between the underlying CfnResources.
             */
            this.api.methods.map(m => m.node.defaultChild).forEach(m => this.resource.addDependsOn(m));
        }
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.hashComponents = new Array();
        this.originalLogicalId = core_1.Stack.of(this).getLogicalId(this);
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    /**
     * Hooks into synthesis to calculate a logical ID that hashes all the components
     * add via `addToLogicalId`.
     */
    prepare() {
        const stack = core_1.Stack.of(this);
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (this.hashComponents.length > 0) {
            const md5 = crypto.createHash('md5');
            this.hashComponents.map(x => stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            this.overrideLogicalId(this.originalLogicalId + md5.digest("hex"));
        }
        super.prepare();
    }
}
//# sourceMappingURL=data:application/json;base64,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