"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
class DomainName extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${targetApi.node.uniqueId}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options
        });
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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