"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.resource = props.resource;
        this.restApi = props.resource.restApi;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = (_a = authorizer) === null || _a === void 0 ? void 0 : _a.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = ((_b = authorizer) === null || _b === void 0 ? void 0 : _b.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if (((_c = authorizer) === null || _c === void 0 ? void 0 : _c.authorizationType) && authorizationTypeOption && ((_d = authorizer) === null || _d === void 0 ? void 0 : _d.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer instanceof authorizer_1.Authorizer) {
            authorizer._attachToApi(this.restApi);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.restApi.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: options.requestValidator ? options.requestValidator.requestValidatorId : undefined
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        props.resource.restApi._attachMethod(this);
        const deployment = props.resource.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, this.resource.path, stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, this.resource.path, 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error(`'credentialsPassthrough' and 'credentialsRole' are mutually exclusive`);
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error(`'connectionType' of VPC_LINK requires 'vpcLink' prop to be set`);
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error(`cannot set 'vpcLink' where 'connectionType' is INTERNET`);
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
//# sourceMappingURL=data:application/json;base64,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