"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
const util = require("./util");
/**
 * Represents a reference to a REST API's Empty model, which is available
 * as part of the model collection by default. This can be used for mapping
 * JSON responses from an integration to what is returned to a client,
 * where strong typing is not required. In the absence of any defined
 * model, the Empty model will be used to return the response payload
 * unmapped.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Empty Schema",
 *   "type" : "object"
 * }
 *
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/models-mappings.html#models-mappings-models
 * @deprecated You should use @see Model.EMPTY_MODEL
 */
class EmptyModel {
    constructor() {
        this.modelId = 'Empty';
    }
}
exports.EmptyModel = EmptyModel;
/**
 * Represents a reference to a REST API's Error model, which is available
 * as part of the model collection by default. This can be used for mapping
 * error JSON responses from an integration to a client, where a simple
 * generic message field is sufficient to map and return an error payload.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Error Schema",
 *   "type" : "object",
 *   "properties" : {
 *     "message" : { "type" : "string" }
 *   }
 * }
 * @deprecated You should use @see Model.ERROR_MODEL
 */
class ErrorModel {
    constructor() {
        this.modelId = 'Error';
    }
}
exports.ErrorModel = ErrorModel;
class Model extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.modelName,
        });
        const modelProps = {
            name: this.physicalName,
            restApiId: props.restApi.restApiId,
            contentType: (_a = props.contentType, (_a !== null && _a !== void 0 ? _a : 'application/json')),
            description: props.description,
            schema: util.JsonSchemaMapper.toCfnJsonSchema(props.schema)
        };
        const resource = new apigateway_generated_1.CfnModel(this, 'Resource', modelProps);
        this.modelId = this.getResourceNameAttribute(resource.ref);
        const deployment = (props.restApi instanceof restapi_1.RestApi) ? props.restApi.latestDeployment : undefined;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ model: modelProps });
        }
    }
    static fromModelName(scope, id, modelName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.modelId = modelName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Model = Model;
/**
 * Represents a reference to a REST API's Error model, which is available
 * as part of the model collection by default. This can be used for mapping
 * error JSON responses from an integration to a client, where a simple
 * generic message field is sufficient to map and return an error payload.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Error Schema",
 *   "type" : "object",
 *   "properties" : {
 *     "message" : { "type" : "string" }
 *   }
 * }
 */
Model.ERROR_MODEL = new ErrorModel();
/**
 * Represents a reference to a REST API's Empty model, which is available
 * as part of the model collection by default. This can be used for mapping
 * JSON responses from an integration to what is returned to a client,
 * where strong typing is not required. In the absence of any defined
 * model, the Empty model will be used to return the response payload
 * unmapped.
 *
 * Definition
 * {
 *   "$schema" : "http://json-schema.org/draft-04/schema#",
 *   "title" : "Empty Schema",
 *   "type" : "object"
 * }
 *
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/models-mappings.html#models-mappings-models
 */
Model.EMPTY_MODEL = new EmptyModel();
//# sourceMappingURL=data:application/json;base64,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