"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.configureDeployment(props);
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
    }
    static fromRestApiId(scope, id, restApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this]
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this
        });
    }
    /**
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return [`The REST API doesn't contain any methods`];
        }
        return [];
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error(`Cannot set 'deployOptions' if 'deploy' is disabled`);
            }
        }
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn
        });
        resource.node.addDependency(apiResource);
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId)
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.restApi = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzdGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc3RhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSx3Q0FBd0M7QUFDeEMsd0NBQWtHO0FBQ2xHLHVDQUE0QztBQUM1QyxpRUFBZ0U7QUFFaEUsNkNBQTBDO0FBQzFDLCtDQUE4RDtBQUc5RCxtQ0FBOEM7QUFDOUMseURBQStFO0FBQy9FLHlDQUFzRTtBQUN0RSxtQ0FBOEM7QUFDOUMsNkNBQXlEO0FBbUt6RDs7Ozs7OztHQU9HO0FBQ0gsTUFBYSxPQUFRLFNBQVEsZUFBUTtJQStDbkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFzQixFQUFHO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxXQUFXLElBQUksRUFBRTtTQUN0QyxDQUFDLENBQUM7UUFYTDs7V0FFRztRQUNhLFlBQU8sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1FBVTVDLE1BQU0sUUFBUSxHQUFHLElBQUksaUNBQVUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2hELElBQUksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN2QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxzQkFBc0IsRUFBRSxLQUFLLENBQUMsc0JBQXNCO1lBQ3BELGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMscUJBQXFCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQztZQUNyRCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNsRSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDN0IsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDO1FBRWxDLElBQUksQ0FBQyxTQUFTLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUU5QixJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFaEMsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUN4RixJQUFJLGNBQWMsRUFBRTtZQUNsQixJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDeEM7UUFFRCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksWUFBWSxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDdkUsSUFBSSxDQUFDLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQztRQUV6RCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3REO0lBQ0gsQ0FBQztJQS9FTSxNQUFNLENBQUMsYUFBYSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFNBQWlCO1FBQ3pFLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixjQUFTLEdBQUcsU0FBUyxDQUFDO1lBQ3hDLENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUEyRUQ7OztPQUdHO0lBQ0gsSUFBVyxVQUFVO1FBQ25CLE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQztJQUMxQixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsZ0JBQWdCO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDO0lBQ2hDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsR0FBRztRQUNaLE9BQU8sSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQzNCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksVUFBVSxDQUFDLE9BQWUsR0FBRztRQUNsQyxJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLG9IQUFvSCxDQUFDLENBQUM7U0FDdkk7UUFFRCxPQUFPLElBQUksQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksYUFBYSxDQUFDLEVBQVUsRUFBRSxPQUEwQjtRQUN6RCxNQUFNLFVBQVUsR0FBRyxJQUFJLHdCQUFVLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUMxQyxHQUFHLE9BQU87WUFDVixPQUFPLEVBQUUsSUFBSTtTQUNkLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ3JCLElBQUksQ0FBQyxXQUFXLEdBQUcsVUFBVSxDQUFDO1NBQy9CO1FBQ0QsT0FBTyxVQUFVLENBQUM7SUFDcEIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksWUFBWSxDQUFDLEVBQVUsRUFBRSxRQUF3QixFQUFFO1FBQ3hELE9BQU8sSUFBSSxzQkFBUyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLEVBQVU7UUFDekIsT0FBTyxJQUFJLGdCQUFNLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUMxQixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUM7U0FDbEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksUUFBUSxDQUFDLEVBQVUsRUFBRSxLQUFtQjtRQUM3QyxPQUFPLElBQUksYUFBSyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDekIsR0FBRyxLQUFLO1lBQ1IsT0FBTyxFQUFFLElBQUk7U0FDZCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxtQkFBbUIsQ0FBQyxFQUFVLEVBQUUsS0FBOEI7UUFDbkUsT0FBTyxJQUFJLG1DQUFnQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDcEMsR0FBRyxLQUFLO1lBQ1IsT0FBTyxFQUFFLElBQUk7U0FDZCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLGdCQUFnQixDQUFDLFNBQWlCLEdBQUcsRUFBRSxPQUFlLElBQUksRUFBRSxRQUFnQixHQUFHO1FBQ3BGLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLElBQUksR0FBRyxDQUFDLENBQUM7U0FDNUQ7UUFFRCxJQUFJLE1BQU0sQ0FBQyxXQUFXLEVBQUUsS0FBSyxLQUFLLEVBQUU7WUFDbEMsTUFBTSxHQUFHLEdBQUcsQ0FBQztTQUNkO1FBRUQsT0FBTyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUM5QixPQUFPLEVBQUUsYUFBYTtZQUN0QixRQUFRLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDeEIsR0FBRyxFQUFFLEdBQUc7WUFDUixZQUFZLEVBQUUsR0FBRyxLQUFLLElBQUksTUFBTSxHQUFHLElBQUksRUFBRTtTQUMxQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxhQUFhLENBQUMsTUFBYztRQUNqQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUM1QixDQUFDO0lBRUQ7O09BRUc7SUFDTyxRQUFRO1FBQ2hCLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzdCLE9BQU8sQ0FBRSwwQ0FBMEMsQ0FBRSxDQUFDO1NBQ3ZEO1FBRUQsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0lBRU8sbUJBQW1CLENBQUMsS0FBbUI7UUFDN0MsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUNoRSxJQUFJLE1BQU0sRUFBRTtZQUVWLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLHVCQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtnQkFDMUQsV0FBVyxFQUFFLGdEQUFnRDtnQkFDN0QsR0FBRyxFQUFFLElBQUk7Z0JBQ1QsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjthQUMzQyxDQUFDLENBQUM7WUFFSCw2R0FBNkc7WUFDN0csMkRBQTJEO1lBQzNELE1BQU0sU0FBUyxHQUFHLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxJQUFJLE1BQU0sQ0FBQztZQUVuRixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksYUFBSyxDQUFDLElBQUksRUFBRSxtQkFBbUIsU0FBUyxFQUFFLEVBQUU7Z0JBQ3JFLFVBQVUsRUFBRSxJQUFJLENBQUMsaUJBQWlCO2dCQUNsQyxHQUFHLEtBQUssQ0FBQyxhQUFhO2FBQ3ZCLENBQUMsQ0FBQztZQUVILElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsQ0FBQztTQUNyRzthQUFNO1lBQ0wsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO2dCQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7YUFDdkU7U0FDRjtJQUNILENBQUM7SUFFTyx1QkFBdUIsQ0FBQyxXQUF1QjtRQUNyRCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQ2hELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQywwQkFBMEIsQ0FBQztZQUMvRCxlQUFlLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLG1EQUFtRCxDQUFDLENBQUM7U0FDbkgsQ0FBQyxDQUFDO1FBRUgsTUFBTSxRQUFRLEdBQUcsSUFBSSxpQ0FBVSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDL0MsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLE9BQU87U0FDaEMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVPLGtCQUFrQixDQUFDLEtBQW1COztRQUM1QyxJQUFJLEtBQUssQ0FBQyxhQUFhLElBQUksS0FBSyxDQUFDLHFCQUFxQixFQUFFO1lBQ3RELE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztTQUN0RztRQUNELElBQUksS0FBSyxDQUFDLHFCQUFxQixFQUFFO1lBQy9CLE9BQU87Z0JBQ0wsS0FBSyxFQUFFLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLO2dCQUN4QyxjQUFjLGNBQUUsS0FBSyxDQUFDLHFCQUFxQiwwQ0FBRSxZQUFZLDBDQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUM7YUFDekcsQ0FBQztTQUNIO1FBQ0QsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLE9BQU8sRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSxDQUFDO1NBQ3ZDO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztDQUNGO0FBalJELDBCQWlSQztBQXVCRCxJQUFZLGdCQVVYO0FBVkQsV0FBWSxnQkFBZ0I7SUFDMUI7O09BRUc7SUFDSCxxQ0FBaUIsQ0FBQTtJQUVqQjs7T0FFRztJQUNILDZDQUF5QixDQUFBO0FBQzNCLENBQUMsRUFWVyxnQkFBZ0IsR0FBaEIsd0JBQWdCLEtBQWhCLHdCQUFnQixRQVUzQjtBQUVELElBQVksWUFlWDtBQWZELFdBQVksWUFBWTtJQUN0Qjs7T0FFRztJQUNILDZCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILHFDQUFxQixDQUFBO0lBRXJCOztPQUVHO0lBQ0gsbUNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQWZXLFlBQVksR0FBWixvQkFBWSxLQUFaLG9CQUFZLFFBZXZCO0FBRUQsTUFBTSxZQUFhLFNBQVEsdUJBQVk7SUFTckMsWUFBWSxHQUFZLEVBQUUsS0FBbUIsRUFBRSxVQUFrQjtRQUMvRCxLQUFLLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRXRCLElBQUksQ0FBQyxjQUFjLEdBQUcsU0FBUyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7UUFDbkQsSUFBSSxDQUFDLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQztRQUN2RCxJQUFJLENBQUMsMkJBQTJCLEdBQUcsS0FBSyxDQUFDLDJCQUEyQixDQUFDO1FBQ3JFLElBQUksQ0FBQyxPQUFPLEdBQUcsR0FBRyxDQUFDO1FBQ25CLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzdCLElBQUksQ0FBQyxJQUFJLEdBQUcsR0FBRyxDQUFDO1FBRWhCLElBQUksSUFBSSxDQUFDLDJCQUEyQixFQUFFO1lBQ3BDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsMkJBQTJCLENBQUMsQ0FBQztTQUN6RDtJQUNILENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElWcGNFbmRwb2ludCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQ2ZuT3V0cHV0LCBDb25zdHJ1Y3QsIElSZXNvdXJjZSBhcyBJUmVzb3VyY2VCYXNlLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEFwaUtleSwgSUFwaUtleSB9IGZyb20gJy4vYXBpLWtleSc7XG5pbXBvcnQgeyBDZm5BY2NvdW50LCBDZm5SZXN0QXBpIH0gZnJvbSAnLi9hcGlnYXRld2F5LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBDb3JzT3B0aW9ucyB9IGZyb20gJy4vY29ycyc7XG5pbXBvcnQgeyBEZXBsb3ltZW50IH0gZnJvbSAnLi9kZXBsb3ltZW50JztcbmltcG9ydCB7IERvbWFpbk5hbWUsIERvbWFpbk5hbWVPcHRpb25zIH0gZnJvbSAnLi9kb21haW4tbmFtZSc7XG5pbXBvcnQgeyBJbnRlZ3JhdGlvbiB9IGZyb20gJy4vaW50ZWdyYXRpb24nO1xuaW1wb3J0IHsgTWV0aG9kLCBNZXRob2RPcHRpb25zIH0gZnJvbSAnLi9tZXRob2QnO1xuaW1wb3J0IHsgTW9kZWwsIE1vZGVsT3B0aW9ucyB9IGZyb20gJy4vbW9kZWwnO1xuaW1wb3J0IHsgUmVxdWVzdFZhbGlkYXRvciwgUmVxdWVzdFZhbGlkYXRvck9wdGlvbnMgfSBmcm9tICcuL3JlcXVlc3R2YWxpZGF0b3InO1xuaW1wb3J0IHsgSVJlc291cmNlLCBSZXNvdXJjZUJhc2UsIFJlc291cmNlT3B0aW9ucyB9IGZyb20gJy4vcmVzb3VyY2UnO1xuaW1wb3J0IHsgU3RhZ2UsIFN0YWdlT3B0aW9ucyB9IGZyb20gJy4vc3RhZ2UnO1xuaW1wb3J0IHsgVXNhZ2VQbGFuLCBVc2FnZVBsYW5Qcm9wcyB9IGZyb20gJy4vdXNhZ2UtcGxhbic7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSVJlc3RBcGkgZXh0ZW5kcyBJUmVzb3VyY2VCYXNlIHtcbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGlzIEFQSSBHYXRld2F5IFJlc3RBcGkuXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHJlc3RBcGlJZDogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc3RBcGlQcm9wcyBleHRlbmRzIFJlc291cmNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgYSBEZXBsb3ltZW50IHNob3VsZCBiZSBhdXRvbWF0aWNhbGx5IGNyZWF0ZWQgZm9yIHRoaXMgQVBJLFxuICAgKiBhbmQgcmVjcmVhdGVkIHdoZW4gdGhlIEFQSSBtb2RlbCAocmVzb3VyY2VzLCBtZXRob2RzKSBjaGFuZ2VzLlxuICAgKlxuICAgKiBTaW5jZSBBUEkgR2F0ZXdheSBkZXBsb3ltZW50cyBhcmUgaW1tdXRhYmxlLCBXaGVuIHRoaXMgb3B0aW9uIGlzIGVuYWJsZWRcbiAgICogKGJ5IGRlZmF1bHQpLCBhbiBBV1M6OkFwaUdhdGV3YXk6OkRlcGxveW1lbnQgcmVzb3VyY2Ugd2lsbCBhdXRvbWF0aWNhbGx5XG4gICAqIGNyZWF0ZWQgd2l0aCBhIGxvZ2ljYWwgSUQgdGhhdCBoYXNoZXMgdGhlIEFQSSBtb2RlbCAobWV0aG9kcywgcmVzb3VyY2VzXG4gICAqIGFuZCBvcHRpb25zKS4gVGhpcyBtZWFucyB0aGF0IHdoZW4gdGhlIG1vZGVsIGNoYW5nZXMsIHRoZSBsb2dpY2FsIElEIG9mXG4gICAqIHRoaXMgQ2xvdWRGb3JtYXRpb24gcmVzb3VyY2Ugd2lsbCBjaGFuZ2UsIGFuZCBhIG5ldyBkZXBsb3ltZW50IHdpbGwgYmVcbiAgICogY3JlYXRlZC5cbiAgICpcbiAgICogSWYgdGhpcyBpcyBzZXQsIGBsYXRlc3REZXBsb3ltZW50YCB3aWxsIHJlZmVyIHRvIHRoZSBgRGVwbG95bWVudGAgb2JqZWN0XG4gICAqIGFuZCBgZGVwbG95bWVudFN0YWdlYCB3aWxsIHJlZmVyIHRvIGEgYFN0YWdlYCB0aGF0IHBvaW50cyB0byB0aGlzXG4gICAqIGRlcGxveW1lbnQuIFRvIGN1c3RvbWl6ZSB0aGUgc3RhZ2Ugb3B0aW9ucywgdXNlIHRoZSBgZGVwbG95U3RhZ2VPcHRpb25zYFxuICAgKiBwcm9wZXJ0eS5cbiAgICpcbiAgICogQSBDbG91ZEZvcm1hdGlvbiBPdXRwdXQgd2lsbCBhbHNvIGJlIGRlZmluZWQgd2l0aCB0aGUgcm9vdCBVUkwgZW5kcG9pbnRcbiAgICogb2YgdGhpcyBSRVNUIEFQSS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgdGhlIEFQSSBHYXRld2F5IHN0YWdlIHRoYXQgd2lsbCBhbHdheXMgcG9pbnQgdG8gdGhlIGxhdGVzdFxuICAgKiBkZXBsb3ltZW50IHdoZW4gYGRlcGxveWAgaXMgZW5hYmxlZC4gSWYgYGRlcGxveWAgaXMgZGlzYWJsZWQsXG4gICAqIHRoaXMgdmFsdWUgY2Fubm90IGJlIHNldC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBCYXNlZCBvbiBkZWZhdWx0cyBvZiBgU3RhZ2VPcHRpb25zYC5cbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveU9wdGlvbnM/OiBTdGFnZU9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFJldGFpbnMgb2xkIGRlcGxveW1lbnQgcmVzb3VyY2VzIHdoZW4gdGhlIEFQSSBjaGFuZ2VzLiBUaGlzIGFsbG93c1xuICAgKiBtYW51YWxseSByZXZlcnRpbmcgc3RhZ2VzIHRvIHBvaW50IHRvIG9sZCBkZXBsb3ltZW50cyB2aWEgdGhlIEFXU1xuICAgKiBDb25zb2xlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmV0YWluRGVwbG95bWVudHM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBIG5hbWUgZm9yIHRoZSBBUEkgR2F0ZXdheSBSZXN0QXBpIHJlc291cmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIElEIG9mIHRoZSBSZXN0QXBpIGNvbnN0cnVjdC5cbiAgICovXG4gIHJlYWRvbmx5IHJlc3RBcGlOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDdXN0b20gaGVhZGVyIHBhcmFtZXRlcnMgZm9yIHRoZSByZXF1ZXN0LlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbGkvbGF0ZXN0L3JlZmVyZW5jZS9hcGlnYXRld2F5L2ltcG9ydC1yZXN0LWFwaS5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcGFyYW1ldGVycy5cbiAgICovXG4gIHJlYWRvbmx5IHBhcmFtZXRlcnM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBBIHBvbGljeSBkb2N1bWVudCB0aGF0IGNvbnRhaW5zIHRoZSBwZXJtaXNzaW9ucyBmb3IgdGhpcyBSZXN0QXBpXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcG9saWN5LlxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5PzogaWFtLlBvbGljeURvY3VtZW50O1xuXG4gIC8qKlxuICAgKiBBIGRlc2NyaXB0aW9uIG9mIHRoZSBwdXJwb3NlIG9mIHRoaXMgQVBJIEdhdGV3YXkgUmVzdEFwaSByZXNvdXJjZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBkZXNjcmlwdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgRW5kcG9pbnRDb25maWd1cmF0aW9uIHByb3BlcnR5IHR5cGUgc3BlY2lmaWVzIHRoZSBlbmRwb2ludCB0eXBlcyBvZiBhIFJFU1QgQVBJXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtYXBpZ2F0ZXdheS1yZXN0YXBpLWVuZHBvaW50Y29uZmlndXJhdGlvbi5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZW5kcG9pbnQgY29uZmlndXJhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRDb25maWd1cmF0aW9uPzogRW5kcG9pbnRDb25maWd1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgdGhlIGVuZHBvaW50IHR5cGVzIG9mIHRoZSBBUEkuIFVzZSB0aGlzIHByb3BlcnR5IHdoZW4gY3JlYXRpbmdcbiAgICogYW4gQVBJLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGVuZHBvaW50IHR5cGVzLlxuICAgKiBAZGVwcmVjYXRlZCB0aGlzIHByb3BlcnR5IGlzIGRlcHJlY2F0ZWQsIHVzZSBlbmRwb2ludENvbmZpZ3VyYXRpb24gaW5zdGVhZFxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRUeXBlcz86IEVuZHBvaW50VHlwZVtdO1xuXG4gIC8qKlxuICAgKiBUaGUgc291cmNlIG9mIHRoZSBBUEkga2V5IGZvciBtZXRlcmluZyByZXF1ZXN0cyBhY2NvcmRpbmcgdG8gYSB1c2FnZVxuICAgKiBwbGFuLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE1ldGVyaW5nIGlzIGRpc2FibGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgYXBpS2V5U291cmNlVHlwZT86IEFwaUtleVNvdXJjZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIGJpbmFyeSBtZWRpYSBtaW1lLXR5cGVzIHRoYXQgYXJlIHN1cHBvcnRlZCBieSB0aGUgUmVzdEFwaVxuICAgKiByZXNvdXJjZSwgc3VjaCBhcyBcImltYWdlL3BuZ1wiIG9yIFwiYXBwbGljYXRpb24vb2N0ZXQtc3RyZWFtXCJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBSZXN0QXBpIHN1cHBvcnRzIG9ubHkgVVRGLTgtZW5jb2RlZCB0ZXh0IHBheWxvYWRzLlxuICAgKi9cbiAgcmVhZG9ubHkgYmluYXJ5TWVkaWFUeXBlcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0byByb2xsIGJhY2sgdGhlIHJlc291cmNlIGlmIGEgd2FybmluZyBvY2N1cnMgd2hpbGUgQVBJXG4gICAqIEdhdGV3YXkgaXMgY3JlYXRpbmcgdGhlIFJlc3RBcGkgcmVzb3VyY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBmYWlsT25XYXJuaW5ncz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgbnVsbGFibGUgaW50ZWdlciB0aGF0IGlzIHVzZWQgdG8gZW5hYmxlIGNvbXByZXNzaW9uICh3aXRoIG5vbi1uZWdhdGl2ZVxuICAgKiBiZXR3ZWVuIDAgYW5kIDEwNDg1NzYwICgxME0pIGJ5dGVzLCBpbmNsdXNpdmUpIG9yIGRpc2FibGUgY29tcHJlc3Npb25cbiAgICogKHdoZW4gdW5kZWZpbmVkKSBvbiBhbiBBUEkuIFdoZW4gY29tcHJlc3Npb24gaXMgZW5hYmxlZCwgY29tcHJlc3Npb24gb3JcbiAgICogZGVjb21wcmVzc2lvbiBpcyBub3QgYXBwbGllZCBvbiB0aGUgcGF5bG9hZCBpZiB0aGUgcGF5bG9hZCBzaXplIGlzXG4gICAqIHNtYWxsZXIgdGhhbiB0aGlzIHZhbHVlLiBTZXR0aW5nIGl0IHRvIHplcm8gYWxsb3dzIGNvbXByZXNzaW9uIGZvciBhbnlcbiAgICogcGF5bG9hZCBzaXplLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENvbXByZXNzaW9uIGlzIGRpc2FibGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbWluaW11bUNvbXByZXNzaW9uU2l6ZT86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSBBUEkgR2F0ZXdheSBSZXN0QXBpIHJlc291cmNlIHRoYXQgeW91IHdhbnQgdG8gY2xvbmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICovXG4gIHJlYWRvbmx5IGNsb25lRnJvbT86IElSZXN0QXBpO1xuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGNvbmZpZ3VyZSBhbiBBV1MgQ2xvdWRXYXRjaCByb2xlIGZvciBBUEkgR2F0ZXdheS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY2xvdWRXYXRjaFJvbGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmUgYSBjdXN0b20gZG9tYWluIG5hbWUgYW5kIG1hcCBpdCB0byB0aGlzIEFQSS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBkb21haW4gbmFtZSBpcyBkZWZpbmVkLCB1c2UgYGFkZERvbWFpbk5hbWVgIG9yIGRpcmVjdGx5IGRlZmluZSBhIGBEb21haW5OYW1lYC5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU/OiBEb21haW5OYW1lT3B0aW9ucztcblxuICAvKipcbiAgICogRXhwb3J0IG5hbWUgZm9yIHRoZSBDZm5PdXRwdXQgY29udGFpbmluZyB0aGUgQVBJIGVuZHBvaW50XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gd2hlbiBubyBleHBvcnQgbmFtZSBpcyBnaXZlbiwgb3V0cHV0IHdpbGwgYmUgY3JlYXRlZCB3aXRob3V0IGV4cG9ydFxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRFeHBvcnROYW1lPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBSRVNUIEFQSSBpbiBBbWF6b24gQVBJIEdhdGV3YXkuXG4gKlxuICogVXNlIGBhZGRSZXNvdXJjZWAgYW5kIGBhZGRNZXRob2RgIHRvIGNvbmZpZ3VyZSB0aGUgQVBJIG1vZGVsLlxuICpcbiAqIEJ5IGRlZmF1bHQsIHRoZSBBUEkgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGRlcGxveWVkIGFuZCBhY2Nlc3NpYmxlIGZyb20gYVxuICogcHVibGljIGVuZHBvaW50LlxuICovXG5leHBvcnQgY2xhc3MgUmVzdEFwaSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVJlc3RBcGkge1xuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVJlc3RBcGlJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCByZXN0QXBpSWQ6IHN0cmluZyk6IElSZXN0QXBpIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElSZXN0QXBpIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSByZXN0QXBpSWQgPSByZXN0QXBpSWQ7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhpcyBBUEkgR2F0ZXdheSBSZXN0QXBpLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGlJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVzb3VyY2UgSUQgb2YgdGhlIHJvb3QgcmVzb3VyY2UuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZXN0QXBpUm9vdFJlc291cmNlSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogUmVwcmVzZW50cyB0aGUgcm9vdCByZXNvdXJjZSAoXCIvXCIpIG9mIHRoaXMgQVBJLiBVc2UgaXQgdG8gZGVmaW5lIHRoZSBBUEkgbW9kZWw6XG4gICAqXG4gICAqICAgIGFwaS5yb290LmFkZE1ldGhvZCgnQU5ZJywgcmVkaXJlY3RUb0hvbWVQYWdlKTsgLy8gXCJBTlkgL1wiXG4gICAqICAgIGFwaS5yb290LmFkZFJlc291cmNlKCdmcmllbmRzJykuYWRkTWV0aG9kKCdHRVQnLCBnZXRGcmllbmRzSGFuZGxlcik7IC8vIFwiR0VUIC9mcmllbmRzXCJcbiAgICpcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByb290OiBJUmVzb3VyY2U7XG5cbiAgLyoqXG4gICAqIEFQSSBHYXRld2F5IHN0YWdlIHRoYXQgcG9pbnRzIHRvIHRoZSBsYXRlc3QgZGVwbG95bWVudCAoaWYgZGVmaW5lZCkuXG4gICAqXG4gICAqIElmIGBkZXBsb3lgIGlzIGRpc2FibGVkLCB5b3Ugd2lsbCBuZWVkIHRvIGV4cGxpY2l0bHkgYXNzaWduIHRoaXMgdmFsdWUgaW4gb3JkZXIgdG9cbiAgICogc2V0IHVwIGludGVncmF0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBkZXBsb3ltZW50U3RhZ2UhOiBTdGFnZTtcblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2YgbWV0aG9kcyBib3VuZCB0byB0aGlzIFJlc3RBcGlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtZXRob2RzID0gbmV3IEFycmF5PE1ldGhvZD4oKTtcblxuICBwcml2YXRlIF9kb21haW5OYW1lPzogRG9tYWluTmFtZTtcbiAgcHJpdmF0ZSBfbGF0ZXN0RGVwbG95bWVudDogRGVwbG95bWVudCB8IHVuZGVmaW5lZDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVzdEFwaVByb3BzID0geyB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnJlc3RBcGlOYW1lIHx8IGlkLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuUmVzdEFwaSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBuYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIHBvbGljeTogcHJvcHMucG9saWN5LFxuICAgICAgZmFpbE9uV2FybmluZ3M6IHByb3BzLmZhaWxPbldhcm5pbmdzLFxuICAgICAgbWluaW11bUNvbXByZXNzaW9uU2l6ZTogcHJvcHMubWluaW11bUNvbXByZXNzaW9uU2l6ZSxcbiAgICAgIGJpbmFyeU1lZGlhVHlwZXM6IHByb3BzLmJpbmFyeU1lZGlhVHlwZXMsXG4gICAgICBlbmRwb2ludENvbmZpZ3VyYXRpb246IHRoaXMuY29uZmlndXJlRW5kcG9pbnRzKHByb3BzKSxcbiAgICAgIGFwaUtleVNvdXJjZVR5cGU6IHByb3BzLmFwaUtleVNvdXJjZVR5cGUsXG4gICAgICBjbG9uZUZyb206IHByb3BzLmNsb25lRnJvbSA/IHByb3BzLmNsb25lRnJvbS5yZXN0QXBpSWQgOiB1bmRlZmluZWQsXG4gICAgICBwYXJhbWV0ZXJzOiBwcm9wcy5wYXJhbWV0ZXJzXG4gICAgfSk7XG4gICAgdGhpcy5ub2RlLmRlZmF1bHRDaGlsZCA9IHJlc291cmNlO1xuXG4gICAgdGhpcy5yZXN0QXBpSWQgPSByZXNvdXJjZS5yZWY7XG5cbiAgICB0aGlzLmNvbmZpZ3VyZURlcGxveW1lbnQocHJvcHMpO1xuXG4gICAgY29uc3QgY2xvdWRXYXRjaFJvbGUgPSBwcm9wcy5jbG91ZFdhdGNoUm9sZSAhPT0gdW5kZWZpbmVkID8gcHJvcHMuY2xvdWRXYXRjaFJvbGUgOiB0cnVlO1xuICAgIGlmIChjbG91ZFdhdGNoUm9sZSkge1xuICAgICAgdGhpcy5jb25maWd1cmVDbG91ZFdhdGNoUm9sZShyZXNvdXJjZSk7XG4gICAgfVxuXG4gICAgdGhpcy5yb290ID0gbmV3IFJvb3RSZXNvdXJjZSh0aGlzLCBwcm9wcywgcmVzb3VyY2UuYXR0clJvb3RSZXNvdXJjZUlkKTtcbiAgICB0aGlzLnJlc3RBcGlSb290UmVzb3VyY2VJZCA9IHJlc291cmNlLmF0dHJSb290UmVzb3VyY2VJZDtcblxuICAgIGlmIChwcm9wcy5kb21haW5OYW1lKSB7XG4gICAgICB0aGlzLmFkZERvbWFpbk5hbWUoJ0N1c3RvbURvbWFpbicsIHByb3BzLmRvbWFpbk5hbWUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgZmlyc3QgZG9tYWluIG5hbWUgbWFwcGVkIHRvIHRoaXMgQVBJLCBpZiBkZWZpbmVkIHRocm91Z2ggdGhlIGBkb21haW5OYW1lYFxuICAgKiBjb25maWd1cmF0aW9uIHByb3AsIG9yIGFkZGVkIHZpYSBgYWRkRG9tYWluTmFtZWBcbiAgICovXG4gIHB1YmxpYyBnZXQgZG9tYWluTmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy5fZG9tYWluTmFtZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBUEkgR2F0ZXdheSBkZXBsb3ltZW50IHRoYXQgcmVwcmVzZW50cyB0aGUgbGF0ZXN0IGNoYW5nZXMgb2YgdGhlIEFQSS5cbiAgICogVGhpcyByZXNvdXJjZSB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgdXBkYXRlZCBldmVyeSB0aW1lIHRoZSBSRVNUIEFQSSBtb2RlbCBjaGFuZ2VzLlxuICAgKiBUaGlzIHdpbGwgYmUgdW5kZWZpbmVkIGlmIGBkZXBsb3lgIGlzIGZhbHNlLlxuICAgKi9cbiAgcHVibGljIGdldCBsYXRlc3REZXBsb3ltZW50KCkge1xuICAgIHJldHVybiB0aGlzLl9sYXRlc3REZXBsb3ltZW50O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBkZXBsb3llZCByb290IFVSTCBvZiB0aGlzIFJFU1QgQVBJLlxuICAgKi9cbiAgcHVibGljIGdldCB1cmwoKSB7XG4gICAgcmV0dXJuIHRoaXMudXJsRm9yUGF0aCgpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIFVSTCBmb3IgYW4gSFRUUCBwYXRoLlxuICAgKlxuICAgKiBGYWlscyBpZiBgZGVwbG95bWVudFN0YWdlYCBpcyBub3Qgc2V0IGVpdGhlciBieSBgZGVwbG95YCBvciBleHBsaWNpdGx5LlxuICAgKi9cbiAgcHVibGljIHVybEZvclBhdGgocGF0aDogc3RyaW5nID0gJy8nKTogc3RyaW5nIHtcbiAgICBpZiAoIXRoaXMuZGVwbG95bWVudFN0YWdlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBkZXRlcm1pbmUgZGVwbG95bWVudCBzdGFnZSBmb3IgQVBJIGZyb20gXCJkZXBsb3ltZW50U3RhZ2VcIi4gVXNlIFwiZGVwbG95XCIgb3IgZXhwbGljaXRseSBzZXQgXCJkZXBsb3ltZW50U3RhZ2VcIicpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLmRlcGxveW1lbnRTdGFnZS51cmxGb3JQYXRoKHBhdGgpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYW4gQVBJIEdhdGV3YXkgZG9tYWluIG5hbWUgYW5kIG1hcHMgaXQgdG8gdGhpcyBBUEkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0IGlkXG4gICAqIEBwYXJhbSBvcHRpb25zIGN1c3RvbSBkb21haW4gb3B0aW9uc1xuICAgKi9cbiAgcHVibGljIGFkZERvbWFpbk5hbWUoaWQ6IHN0cmluZywgb3B0aW9uczogRG9tYWluTmFtZU9wdGlvbnMpOiBEb21haW5OYW1lIHtcbiAgICBjb25zdCBkb21haW5OYW1lID0gbmV3IERvbWFpbk5hbWUodGhpcywgaWQsIHtcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgICBtYXBwaW5nOiB0aGlzXG4gICAgfSk7XG4gICAgaWYgKCF0aGlzLl9kb21haW5OYW1lKSB7XG4gICAgICB0aGlzLl9kb21haW5OYW1lID0gZG9tYWluTmFtZTtcbiAgICB9XG4gICAgcmV0dXJuIGRvbWFpbk5hbWU7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHVzYWdlIHBsYW4uXG4gICAqL1xuICBwdWJsaWMgYWRkVXNhZ2VQbGFuKGlkOiBzdHJpbmcsIHByb3BzOiBVc2FnZVBsYW5Qcm9wcyA9IHt9KTogVXNhZ2VQbGFuIHtcbiAgICByZXR1cm4gbmV3IFVzYWdlUGxhbih0aGlzLCBpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhbiBBcGlLZXlcbiAgICovXG4gIHB1YmxpYyBhZGRBcGlLZXkoaWQ6IHN0cmluZyk6IElBcGlLZXkge1xuICAgIHJldHVybiBuZXcgQXBpS2V5KHRoaXMsIGlkLCB7XG4gICAgICByZXNvdXJjZXM6IFt0aGlzXVxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgbW9kZWwuXG4gICAqL1xuICBwdWJsaWMgYWRkTW9kZWwoaWQ6IHN0cmluZywgcHJvcHM6IE1vZGVsT3B0aW9ucyk6IE1vZGVsIHtcbiAgICByZXR1cm4gbmV3IE1vZGVsKHRoaXMsIGlkLCB7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIHJlc3RBcGk6IHRoaXNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IHJlcXVlc3QgdmFsaWRhdG9yLlxuICAgKi9cbiAgcHVibGljIGFkZFJlcXVlc3RWYWxpZGF0b3IoaWQ6IHN0cmluZywgcHJvcHM6IFJlcXVlc3RWYWxpZGF0b3JPcHRpb25zKTogUmVxdWVzdFZhbGlkYXRvciB7XG4gICAgcmV0dXJuIG5ldyBSZXF1ZXN0VmFsaWRhdG9yKHRoaXMsIGlkLCB7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIHJlc3RBcGk6IHRoaXNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBUaGUgXCJleGVjdXRlLWFwaVwiIEFSTi5cbiAgICogQGRlZmF1bHQgXCIqXCIgcmV0dXJucyB0aGUgZXhlY3V0ZSBBUEkgQVJOIGZvciBhbGwgbWV0aG9kcy9yZXNvdXJjZXMgaW5cbiAgICogdGhpcyBBUEkuXG4gICAqIEBwYXJhbSBtZXRob2QgVGhlIG1ldGhvZCAoZGVmYXVsdCBgKmApXG4gICAqIEBwYXJhbSBwYXRoIFRoZSByZXNvdXJjZSBwYXRoLiBNdXN0IHN0YXJ0IHdpdGggJy8nIChkZWZhdWx0IGAqYClcbiAgICogQHBhcmFtIHN0YWdlIFRoZSBzdGFnZSAoZGVmYXVsdCBgKmApXG4gICAqL1xuICBwdWJsaWMgYXJuRm9yRXhlY3V0ZUFwaShtZXRob2Q6IHN0cmluZyA9ICcqJywgcGF0aDogc3RyaW5nID0gJy8qJywgc3RhZ2U6IHN0cmluZyA9ICcqJykge1xuICAgIGlmICghcGF0aC5zdGFydHNXaXRoKCcvJykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgXCJwYXRoXCIgbXVzdCBiZWdpbiB3aXRoIGEgXCIvXCI6ICcke3BhdGh9J2ApO1xuICAgIH1cblxuICAgIGlmIChtZXRob2QudG9VcHBlckNhc2UoKSA9PT0gJ0FOWScpIHtcbiAgICAgIG1ldGhvZCA9ICcqJztcbiAgICB9XG5cbiAgICByZXR1cm4gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdleGVjdXRlLWFwaScsXG4gICAgICByZXNvdXJjZTogdGhpcy5yZXN0QXBpSWQsXG4gICAgICBzZXA6ICcvJyxcbiAgICAgIHJlc291cmNlTmFtZTogYCR7c3RhZ2V9LyR7bWV0aG9kfSR7cGF0aH1gXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogSW50ZXJuYWwgQVBJIHVzZWQgYnkgYE1ldGhvZGAgdG8ga2VlcCBhbiBpbnZlbnRvcnkgb2YgbWV0aG9kcyBhdCB0aGUgQVBJXG4gICAqIGxldmVsIGZvciB2YWxpZGF0aW9uIHB1cnBvc2VzLlxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfYXR0YWNoTWV0aG9kKG1ldGhvZDogTWV0aG9kKSB7XG4gICAgdGhpcy5tZXRob2RzLnB1c2gobWV0aG9kKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJmb3JtcyB2YWxpZGF0aW9uIG9mIHRoZSBSRVNUIEFQSS5cbiAgICovXG4gIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICBpZiAodGhpcy5tZXRob2RzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIFsgYFRoZSBSRVNUIEFQSSBkb2Vzbid0IGNvbnRhaW4gYW55IG1ldGhvZHNgIF07XG4gICAgfVxuXG4gICAgcmV0dXJuIFtdO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVEZXBsb3ltZW50KHByb3BzOiBSZXN0QXBpUHJvcHMpIHtcbiAgICBjb25zdCBkZXBsb3kgPSBwcm9wcy5kZXBsb3kgPT09IHVuZGVmaW5lZCA/IHRydWUgOiBwcm9wcy5kZXBsb3k7XG4gICAgaWYgKGRlcGxveSkge1xuXG4gICAgICB0aGlzLl9sYXRlc3REZXBsb3ltZW50ID0gbmV3IERlcGxveW1lbnQodGhpcywgJ0RlcGxveW1lbnQnLCB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiAnQXV0b21hdGljYWxseSBjcmVhdGVkIGJ5IHRoZSBSZXN0QXBpIGNvbnN0cnVjdCcsXG4gICAgICAgIGFwaTogdGhpcyxcbiAgICAgICAgcmV0YWluRGVwbG95bWVudHM6IHByb3BzLnJldGFpbkRlcGxveW1lbnRzXG4gICAgICB9KTtcblxuICAgICAgLy8gZW5jb2RlIHRoZSBzdGFnZSBuYW1lIGludG8gdGhlIGNvbnN0cnVjdCBpZCwgc28gaWYgd2UgY2hhbmdlIHRoZSBzdGFnZSBuYW1lLCBpdCB3aWxsIHJlY3JlYXRlIGEgbmV3IHN0YWdlLlxuICAgICAgLy8gc3RhZ2UgbmFtZSBpcyBwYXJ0IG9mIHRoZSBlbmRwb2ludCwgc28gdGhhdCBtYWtlcyBzZW5zZS5cbiAgICAgIGNvbnN0IHN0YWdlTmFtZSA9IChwcm9wcy5kZXBsb3lPcHRpb25zICYmIHByb3BzLmRlcGxveU9wdGlvbnMuc3RhZ2VOYW1lKSB8fCAncHJvZCc7XG5cbiAgICAgIHRoaXMuZGVwbG95bWVudFN0YWdlID0gbmV3IFN0YWdlKHRoaXMsIGBEZXBsb3ltZW50U3RhZ2UuJHtzdGFnZU5hbWV9YCwge1xuICAgICAgICBkZXBsb3ltZW50OiB0aGlzLl9sYXRlc3REZXBsb3ltZW50LFxuICAgICAgICAuLi5wcm9wcy5kZXBsb3lPcHRpb25zXG4gICAgICB9KTtcblxuICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCAnRW5kcG9pbnQnLCB7IGV4cG9ydE5hbWU6IHByb3BzLmVuZHBvaW50RXhwb3J0TmFtZSwgdmFsdWU6IHRoaXMudXJsRm9yUGF0aCgpIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAocHJvcHMuZGVwbG95T3B0aW9ucykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBzZXQgJ2RlcGxveU9wdGlvbnMnIGlmICdkZXBsb3knIGlzIGRpc2FibGVkYCk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVDbG91ZFdhdGNoUm9sZShhcGlSZXNvdXJjZTogQ2ZuUmVzdEFwaSkge1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0Nsb3VkV2F0Y2hSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2FwaWdhdGV3YXkuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQW1hem9uQVBJR2F0ZXdheVB1c2hUb0Nsb3VkV2F0Y2hMb2dzJyldLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuQWNjb3VudCh0aGlzLCAnQWNjb3VudCcsIHtcbiAgICAgIGNsb3VkV2F0Y2hSb2xlQXJuOiByb2xlLnJvbGVBcm5cbiAgICB9KTtcblxuICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeShhcGlSZXNvdXJjZSk7XG4gIH1cblxuICBwcml2YXRlIGNvbmZpZ3VyZUVuZHBvaW50cyhwcm9wczogUmVzdEFwaVByb3BzKTogQ2ZuUmVzdEFwaS5FbmRwb2ludENvbmZpZ3VyYXRpb25Qcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHByb3BzLmVuZHBvaW50VHlwZXMgJiYgcHJvcHMuZW5kcG9pbnRDb25maWd1cmF0aW9uKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIHRoZSBSZXN0QXBpIHByb3BzLCBlbmRwb2ludFR5cGVzIG9yIGVuZHBvaW50Q29uZmlndXJhdGlvbiwgaXMgYWxsb3dlZCcpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuZW5kcG9pbnRDb25maWd1cmF0aW9uKSB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICB0eXBlczogcHJvcHMuZW5kcG9pbnRDb25maWd1cmF0aW9uLnR5cGVzLFxuICAgICAgICB2cGNFbmRwb2ludElkczogcHJvcHMuZW5kcG9pbnRDb25maWd1cmF0aW9uPy52cGNFbmRwb2ludHM/Lm1hcCh2cGNFbmRwb2ludCA9PiB2cGNFbmRwb2ludC52cGNFbmRwb2ludElkKVxuICAgICAgfTtcbiAgICB9XG4gICAgaWYgKHByb3BzLmVuZHBvaW50VHlwZXMpIHtcbiAgICAgIHJldHVybiB7IHR5cGVzOiBwcm9wcy5lbmRwb2ludFR5cGVzIH07XG4gICAgfVxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cbn1cblxuLyoqXG4gKiBUaGUgZW5kcG9pbnQgY29uZmlndXJhdGlvbiBvZiBhIFJFU1QgQVBJLCBpbmNsdWRpbmcgVlBDcyBhbmQgZW5kcG9pbnQgdHlwZXMuXG4gKlxuICogRW5kcG9pbnRDb25maWd1cmF0aW9uIGlzIGEgcHJvcGVydHkgb2YgdGhlIEFXUzo6QXBpR2F0ZXdheTo6UmVzdEFwaSByZXNvdXJjZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmRwb2ludENvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogQSBsaXN0IG9mIGVuZHBvaW50IHR5cGVzIG9mIGFuIEFQSSBvciBpdHMgY3VzdG9tIGRvbWFpbiBuYW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGVuZHBvaW50IHR5cGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZXM6IEVuZHBvaW50VHlwZVtdO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgVlBDIEVuZHBvaW50cyBhZ2FpbnN0IHdoaWNoIHRvIGNyZWF0ZSBSb3V0ZTUzIEFMSUFTZXNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBBTElBU2VzIGFyZSBjcmVhdGVkIGZvciB0aGUgZW5kcG9pbnQuXG4gICAqL1xuICByZWFkb25seSB2cGNFbmRwb2ludHM/OiBJVnBjRW5kcG9pbnRbXTtcbn1cblxuZXhwb3J0IGVudW0gQXBpS2V5U291cmNlVHlwZSB7XG4gIC8qKlxuICAgKiBUbyByZWFkIHRoZSBBUEkga2V5IGZyb20gdGhlIGBYLUFQSS1LZXlgIGhlYWRlciBvZiBhIHJlcXVlc3QuXG4gICAqL1xuICBIRUFERVIgPSAnSEVBREVSJyxcblxuICAvKipcbiAgICogVG8gcmVhZCB0aGUgQVBJIGtleSBmcm9tIHRoZSBgVXNhZ2VJZGVudGlmaWVyS2V5YCBmcm9tIGEgY3VzdG9tIGF1dGhvcml6ZXIuXG4gICAqL1xuICBBVVRIT1JJWkVSID0gJ0FVVEhPUklaRVInLFxufVxuXG5leHBvcnQgZW51bSBFbmRwb2ludFR5cGUge1xuICAvKipcbiAgICogRm9yIGFuIGVkZ2Utb3B0aW1pemVkIEFQSSBhbmQgaXRzIGN1c3RvbSBkb21haW4gbmFtZS5cbiAgICovXG4gIEVER0UgPSAnRURHRScsXG5cbiAgLyoqXG4gICAqIEZvciBhIHJlZ2lvbmFsIEFQSSBhbmQgaXRzIGN1c3RvbSBkb21haW4gbmFtZS5cbiAgICovXG4gIFJFR0lPTkFMID0gJ1JFR0lPTkFMJyxcblxuICAvKipcbiAgICogRm9yIGEgcHJpdmF0ZSBBUEkgYW5kIGl0cyBjdXN0b20gZG9tYWluIG5hbWUuXG4gICAqL1xuICBQUklWQVRFID0gJ1BSSVZBVEUnXG59XG5cbmNsYXNzIFJvb3RSZXNvdXJjZSBleHRlbmRzIFJlc291cmNlQmFzZSB7XG4gIHB1YmxpYyByZWFkb25seSBwYXJlbnRSZXNvdXJjZT86IElSZXNvdXJjZTtcbiAgcHVibGljIHJlYWRvbmx5IHJlc3RBcGk6IFJlc3RBcGk7XG4gIHB1YmxpYyByZWFkb25seSByZXNvdXJjZUlkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBwYXRoOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbiB8IHVuZGVmaW5lZDtcbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRNZXRob2RPcHRpb25zPzogTWV0aG9kT3B0aW9ucyB8IHVuZGVmaW5lZDtcbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucz86IENvcnNPcHRpb25zIHwgdW5kZWZpbmVkO1xuXG4gIGNvbnN0cnVjdG9yKGFwaTogUmVzdEFwaSwgcHJvcHM6IFJlc3RBcGlQcm9wcywgcmVzb3VyY2VJZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoYXBpLCAnRGVmYXVsdCcpO1xuXG4gICAgdGhpcy5wYXJlbnRSZXNvdXJjZSA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLmRlZmF1bHRJbnRlZ3JhdGlvbiA9IHByb3BzLmRlZmF1bHRJbnRlZ3JhdGlvbjtcbiAgICB0aGlzLmRlZmF1bHRNZXRob2RPcHRpb25zID0gcHJvcHMuZGVmYXVsdE1ldGhvZE9wdGlvbnM7XG4gICAgdGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMgPSBwcm9wcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM7XG4gICAgdGhpcy5yZXN0QXBpID0gYXBpO1xuICAgIHRoaXMucmVzb3VyY2VJZCA9IHJlc291cmNlSWQ7XG4gICAgdGhpcy5wYXRoID0gJy8nO1xuXG4gICAgaWYgKHRoaXMuZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zKSB7XG4gICAgICB0aGlzLmFkZENvcnNQcmVmbGlnaHQodGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMpO1xuICAgIH1cbiAgfVxufVxuIl19