"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod, resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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