"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit ? burstLimit : undefined,
                rateLimit: rateLimit ? rateLimit : undefined
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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