"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.substr(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error(`"actionParams" requires that "action" will be set`);
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + url_1.format({ query: actionParams }).substr(1);
        }
        return {
            apiType: 'action',
            apiValue: action
        };
    }
    throw new Error(`Either "path" or "action" are required`);
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!("$schema" in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
exports.JsonSchemaMapper = JsonSchemaMapper;
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
    id: '$id'
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
//# sourceMappingURL=data:application/json;base64,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