"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
module.exports = {
    'default token authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.Authorization'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'default request authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: [],
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'invalid request authorizer config'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        test.throws(() => new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(1),
            identitySources: [],
        }), Error, 'At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        test.done();
    },
    'token authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySource: 'method.request.header.whoami',
            validationRegex: 'a-hacker',
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            IdentityValidationExpression: 'a-hacker',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60
        }));
        test.done();
    },
    'request authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('whoami')],
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60
        }));
        test.done();
    },
    'token authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole'
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName)
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    }
                ],
            }
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'request authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole'
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: []
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName)
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    }
                ],
            }
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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