"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'adds an OPTIONS method to a resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://amazon.com']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'allowCredentials'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://amazon.com'],
            allowCredentials: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                            "method.response.header.Access-Control-Allow-Credentials": "'true'"
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                        "method.response.header.Access-Control-Allow-Credentials": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'allowMethods'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://aws.amazon.com'],
            allowMethods: ['GET', 'PUT']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://aws.amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'GET,PUT'",
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'allowMethods ANY will expand to all supported methods'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://aws.amazon.com'],
            allowMethods: ['ANY']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://aws.amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'allowMethods ANY cannot be used with any other method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // THEN
        test.throws(() => resource.addCorsPreflight({
            allowOrigins: ['https://aws.amazon.com'],
            allowMethods: ['ANY', 'PUT']
        }), /ANY cannot be used with any other method. Received: ANY,PUT/);
        test.done();
    },
    'statusCode can be used to set the response status code'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://aws.amazon.com'],
            statusCode: 200
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://aws.amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                        },
                        StatusCode: "200"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                    },
                    StatusCode: "200"
                }
            ]
        }));
        test.done();
    },
    'allowOrigins must contain at least one origin'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        test.throws(() => resource.addCorsPreflight({
            allowOrigins: []
        }), /allowOrigins must contain at least one origin/);
        test.done();
    },
    'allowOrigins can be used to specify multiple origins'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://twitch.tv', 'https://amazon.com', 'https://aws.amazon.com']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://twitch.tv'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'"
                        },
                        ResponseTemplates: {
                            "application/json": "#set($origin = $input.params(\"Origin\"))\n#if($origin == \"\") #set($origin = $input.params(\"origin\")) #end\n#if($origin.matches(\"https://amazon.com\") || $origin.matches(\"https://aws.amazon.com\"))\n  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)\n#end"
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'maxAge can be used to specify Access-Control-Max-Age'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://amazon.com'],
            maxAge: core_1.Duration.minutes(60)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                            "method.response.header.Access-Control-Max-Age": `'${60 * 60}'`
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                        "method.response.header.Access-Control-Max-Age": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'disableCache will set Max-Age to -1'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://amazon.com'],
            disableCache: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                            "method.response.header.Access-Control-Max-Age": `'-1'`
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                        "method.response.header.Access-Control-Max-Age": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'maxAge and disableCache are mutually exclusive'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // THEN
        test.throws(() => resource.addCorsPreflight({
            allowOrigins: ['https://amazon.com'],
            disableCache: true,
            maxAge: core_1.Duration.seconds(10)
        }), /The options "maxAge" and "disableCache" are mutually exclusive/);
        test.done();
    },
    'exposeHeaders can be used to specify Access-Control-Expose-Headers'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const resource = api.root.addResource('MyResource');
        // WHEN
        resource.addCorsPreflight({
            allowOrigins: ['https://amazon.com'],
            exposeHeaders: ['Authorization', 'Foo']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'https://amazon.com'",
                            "method.response.header.Vary": "'Origin'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                            "method.response.header.Access-Control-Expose-Headers": "'Authorization,Foo'",
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Vary": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                        "method.response.header.Access-Control-Expose-Headers": true,
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'defaultCorsPreflightOptions can be used to specify CORS for all resource tree'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        const resource = api.root.addResource('MyResource', {
            defaultCorsPreflightOptions: {
                allowOrigins: ['https://amazon.com'],
            }
        });
        resource.addResource('MyChildResource');
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::ApiGateway::Method', 2)); // on both resources
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceD5CDB490' }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apiMyResourceMyChildResource2DC010C5' }
        }));
        test.done();
    },
    'defaultCorsPreflightOptions can be specified at the API level to apply to all resources'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            defaultCorsPreflightOptions: {
                allowOrigins: ['https://amazon.com'],
            }
        });
        const child1 = api.root.addResource('child1');
        child1.addResource('child2');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { "Fn::GetAtt": ["apiC8550315", "RootResourceId"] }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apichild1841A5840' }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'apichild1child26A9A7C47' }
        }));
        test.done();
    },
    'Vary: Origin is sent back if Allow-Origin is not "*"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        api.root.addResource('AllowAll', {
            defaultCorsPreflightOptions: {
                allowOrigins: apigw.Cors.ALL_ORIGINS
            }
        });
        api.root.addResource('AllowSpecific', {
            defaultCorsPreflightOptions: {
                allowOrigins: ['http://specific.com']
            }
        });
        // THENB
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ResourceId: {
                Ref: "apiAllowAll2F5BC564"
            },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'*'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'"
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Access-Control-Allow-Methods": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ResourceId: {
                Ref: "apiAllowSpecific77DD8AF1"
            },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            "method.response.header.Access-Control-Allow-Headers": "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            "method.response.header.Access-Control-Allow-Origin": "'http://specific.com'",
                            "method.response.header.Access-Control-Allow-Methods": "'OPTIONS,GET,PUT,POST,DELETE,PATCH,HEAD'",
                            "method.response.header.Vary": "'Origin'"
                        },
                        StatusCode: "204"
                    }
                ],
                RequestTemplates: {
                    "application/json": "{ statusCode: 200 }"
                },
                Type: "MOCK"
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        "method.response.header.Access-Control-Allow-Headers": true,
                        "method.response.header.Access-Control-Allow-Origin": true,
                        "method.response.header.Access-Control-Allow-Methods": true,
                        "method.response.header.Vary": true
                    },
                    StatusCode: "204"
                }
            ]
        }));
        test.done();
    },
    'If "*" is specified in allow-origin, it cannot be mixed with specific origins'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        test.throws(() => api.root.addResource('AllowAll', {
            defaultCorsPreflightOptions: {
                allowOrigins: ['https://bla.com', '*', 'https://specific']
            }
        }), /Invalid "allowOrigins" - cannot mix "\*" with specific origins: https:\/\/bla\.com,\*,https:\/\/specific/);
        test.done();
    },
    'defaultCorsPreflightOptions can be used to specify CORS for all resource tree [LambdaRestApi]'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const handler = new lambda.Function(stack, 'handler', {
            handler: 'index.handler',
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        new apigw.LambdaRestApi(stack, 'lambda-rest-api', {
            handler,
            defaultCorsPreflightOptions: {
                allowOrigins: ['https://amazon.com'],
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::ApiGateway::Method', 4)); // two ANY and two OPTIONS resources
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: {
                "Fn::GetAtt": [
                    "lambdarestapiAAD10924",
                    "RootResourceId"
                ]
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: {
                Ref: "lambdarestapiproxyE3AE07E3"
            }
        }));
        test.done();
    },
    'CORS and proxy resources'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'API', {
            defaultCorsPreflightOptions: { allowOrigins: ['*'] }
        });
        api.root.addProxy();
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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